package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * CoCo壱番屋の住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 3.16
 * 2005/12/04
 */
public class Cocoichi {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://www.ichibanya.co.jp/shop/search/search.html?fcity=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&fpref=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator
		+ "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "cocoichi_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * CoCo壱番屋の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param id 市区町村コード
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String id,
		final String city, final String prefecture,
		final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Cocoichi.CACHE_DIR).exists()) {
			new File(Cocoichi.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Cocoichi.CACHE_DIR + File.separator
			+ Cocoichi.PREFIX + id + Cocoichi.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(
				cacheFile), "SJIS");
			ISJUtil.parseAddresses(Cocoichi.getAddresses(city,
				prefecture, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(
			new FileInputStream(new File(cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "ココイチ");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * CoCo壱番屋の市区町村別ページを解析して住所の一覧を取得します。
	 * @param city 市区町村名
	 * @param prefecture 都道府県名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String city,
		final String prefecture, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final Pattern pattern = Pattern.compile("(.+市).+区$");
			final Matcher matcher = pattern.matcher(city);
			final URL url = matcher.matches() ? new URL(URL1
				+ URLEncoder.encode(matcher.group(1), ENCODING)
				+ URL2 + URLEncoder.encode(prefecture, ENCODING))
				: new URL(URL1 + URLEncoder.encode(city, ENCODING)
					+ URL2 + URLEncoder.encode(prefecture, ENCODING));
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(
				new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern1 = Pattern
				.compile("<p class=\"s-listtext1\">([^<>]+)</p>");
			final Pattern pattern2 = Pattern
				.compile("<p class=\"s-listtext2\">([^<>〜]+)</p>");
			String caption = null;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher1 = pattern1.matcher(line);
				if (matcher1.find()) {
					caption = matcher1.group(1);
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					final String address = matcher2.group(1);
					if (caption != null) {
						if (address.contains(city)) {
							ret.put(address, caption);
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
