package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * マクドナルドの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * 2005/12/03
 */
public class McDonalds {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://www.mcdonalds.co.jp/cgi-bin/shop/search3/store_list.cgi?keyword=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&Image1.x=0&Image1.y=0";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "mcdonalds_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * マクドナルドの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param cityID 市区町村コード
	 * @param prefectureLabel 都道府県名
	 * @param cityLabel 市区町村名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String cityID, final String prefectureLabel,
		final String cityLabel, final Map<String, Point2D> isj, MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(McDonalds.CACHE_DIR).exists()) {
			new File(McDonalds.CACHE_DIR).mkdirs();
		}
		final String cacheFile = McDonalds.CACHE_DIR + File.separator + McDonalds.PREFIX + cityID
			+ McDonalds.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(McDonalds.getAddresses(prefectureLabel, cityLabel, panel), out,
				isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
			cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "マック");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * マクドナルドの市区町村別ページを解析して住所の一覧を取得します。
	 * @param prefectureLabel 都道府県名
	 * @param cityLabel 市区町村名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String prefectureLabel,
		final String cityLabel, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + URLEncoder.encode(prefectureLabel + cityLabel, ENCODING)
				+ URL2);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern = Pattern
				.compile("<a href=\"/cgi-bin/shop/search3/store_data.cgi\\?strcode=[0-9]+\" target=\"_blank\">([^<>]+)</a>");
			final Pattern pattern2 = Pattern.compile("<td width=\"[0-9]+%\">([^<>]+)</td>");
			String caption = null;
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					caption = matcher.group(1);
					continue;
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						final String address = matcher2.group(1);
						ret.put(address.replace(" ", ""), caption.replace(" ", ""));
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
