package test;

import java.awt.Color;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.Scanner;
import java.util.regex.Pattern;

import javax.swing.JFrame;
import javax.swing.WindowConstants;


/**
 * ステージエディタです。
 * @author Kumano Tatsuo
 * Created on 2005/08/28
 */
public class StageEditor {

	/**
	 * @param args
	 * @throws FileNotFoundException ファイル未検出例外
	 */
	public static void main(final String[] args) throws FileNotFoundException {
		final JFrame frame = new JFrame("ステージエディタ");
		frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		frame.setLocationByPlatform(true);
		frame.setSize(640, 480);
		final DemoPanel panel = new DemoPanel();
//		final File dataFile = new File("car" + File.separator + "tsukuba.dat");
//		final String stageName = "TsukubaData";
		final File dataFile = new File("car" + File.separator + "tsukuba2.dat");
		final String stageName = "TsukubaData2";
		final StageEditor editor = new StageEditor(dataFile, stageName);
		if (dataFile.exists()) {
			editor.load();
			editor.update(panel);
		}
		panel.addMouseListener(new MouseAdapter() {
			public void mouseReleased(MouseEvent e) {
			}

			public void mousePressed(MouseEvent e) {
				if (e.getButton() == MouseEvent.BUTTON1) {
					switch (editor.tool) {
					case ADD_POINT:
						if (editor.selection == null) {
							for (int i = 0; i < editor.polygons.length; i++) {
								if (editor.polygons[i] == null) {
									editor.polygons[i] = editor.new Node(
											panel
													.toVirtualLocation(new Point2D.Double(
															e.getX(), e.getY())),
											null, null);
									editor.selection = editor.polygons[i];
									break;
								}
							}
						} else {
							final Node node = editor.new Node(panel
									.toVirtualLocation(new Point2D.Double(e
											.getX(), e.getY())),
									editor.selection.next, editor.selection);
							editor.selection.next = node;
							editor.selection = node;
						}
						break;
					case SELECT_POINT:
						final Point2D query = panel
								.toVirtualLocation(new Point2D.Double(e.getX(),
										e.getY()));
						double min = Double.POSITIVE_INFINITY;
						int minI = -1;
						for (Node polygon : editor.polygons) {
							int i = 0;
							for (Node node = polygon; node != null; node = node.next) {
								final double distance = query
										.distance(node.location);
								if (distance < min) {
									editor.selection = node;
									min = distance;
									minI = i;
								}
								i++;
							}
						}
						System.out.println("DEBUG: selected " + minI + "th point.");
						break;
					}
					editor.update(panel);
				}
			}
		});
		frame.addKeyListener(new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				switch (e.getKeyCode()) {
				case KeyEvent.VK_A:
					editor.tool = Tool.ADD_POINT;
					System.out.println("INFO: tool changed to ADD_POINT");
					break;
				case KeyEvent.VK_S:
					if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
						try {
							editor.save();
							System.out.println("INFO: saved.");
						} catch (FileNotFoundException e1) {
							e1.printStackTrace();
						}
					} else {
						editor.tool = Tool.SELECT_POINT;
						System.out
								.println("INFO: tool changed to SELECT_POINT");
					}
					break;
				case KeyEvent.VK_C:
					editor.isOpen = !editor.isOpen;
					editor.update(panel);
					break;
				case KeyEvent.VK_SEMICOLON:
					if ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0) {
						editor.radius++;
						editor.update(panel);
					}
					break;
				case KeyEvent.VK_MINUS:
					editor.radius = Math.max(0, editor.radius - 1);
					editor.update(panel);
					break;
				}
				if (editor.selection != null) {
					double d;
					switch (e.getModifiers()) {
					case InputEvent.CTRL_MASK:
						d = 10;
						break;
					case InputEvent.SHIFT_MASK:
						d = 0.1;
						break;
					default:
						d = 1;
					}
					switch (e.getKeyCode()) {
					case KeyEvent.VK_RIGHT:
						editor.selection.location.setLocation(
								editor.selection.location.getX() + d,
								editor.selection.location.getY());
						break;
					case KeyEvent.VK_LEFT:
						editor.selection.location.setLocation(
								editor.selection.location.getX() - d,
								editor.selection.location.getY());
						break;
					case KeyEvent.VK_DOWN:
						editor.selection.location.setLocation(
								editor.selection.location.getX(),
								editor.selection.location.getY() + d);
						break;
					case KeyEvent.VK_UP:
						editor.selection.location.setLocation(
								editor.selection.location.getX(),
								editor.selection.location.getY() - d);
						break;
					case KeyEvent.VK_DELETE:
						if (editor.selection.previous == null) {
							if (editor.selection.next != null) {
								editor.selection.location = editor.selection.next.location;
								editor.selection.next = editor.selection.next.next;
								if (editor.selection.next != null) {
									if (editor.selection.next.next != null) {
										editor.selection.next.next.previous = editor.selection;
									}
								}
							} else {
								for (int i = 0; i < editor.polygons.length; i++) {
									if (editor.polygons[i] == editor.selection) {
										editor.polygons[i] = null;
										editor.selection = null;
										break;
									}
								}
							}
						} else {
							editor.selection.previous.next = editor.selection.next;
							if (editor.selection.next != null) {
								editor.selection.next.previous = editor.selection.previous;
							}
							editor.selection = editor.selection.previous;
						}
						break;
					case KeyEvent.VK_ESCAPE:
						editor.selection = null;
						break;
					case KeyEvent.VK_PAGE_DOWN:
						if (editor.selection.next != null) {
							editor.selection = editor.selection.next;
						}
						break;
					case KeyEvent.VK_PAGE_UP:
						if (editor.selection.previous != null) {
							editor.selection = editor.selection.previous;
						}
						break;
					}
					editor.update(panel);
				}
			}
		});
		frame.add(panel);
		frame.setVisible(true);
	}

	/**
	 * コンストラクタです。
	 * @param dataFile データファイル
	 * @param stageName ステージ名
	 */
	public StageEditor(final File dataFile, final String stageName) {
		this.tool = Tool.SELECT_POINT;
		this.polygons = new Node[100];
		this.selection = null;
		this.dataFile = dataFile;
		this.stageName = stageName;
		this.isOpen = false;
		this.radius = 5;
	}

	/**
	 * 選択されているツール
	 */
	Tool tool;

	/**
	 * 領域の一覧
	 */
	Node[] polygons;

	/**
	 * 選択されている頂点
	 */
	Node selection;

	/**
	 * 読み書きするデータファイル
	 */
	final File dataFile;
	
	/**
	 * ステージ名
	 */
	final String stageName;

	/**
	 * パスを開放するかどうか
	 */
	boolean isOpen;

	/**
	 * 点の半径
	 */
	int radius;

	/**
	 * 選択されているツールを表す列挙型です。
	 */
	enum Tool {
		/**
		 * 点を追加するツール
		 */
		ADD_POINT,
		/**
		 * 点を選択するツール
		 */
		SELECT_POINT,
	}

	/**
	 * 1つの頂点を表すクラスです。
	 */
	class Node {
		/**
		 * コンストラクタです。
		 * @param location 座標
		 * @param next 次の頂点
		 * @param previous 前の頂点
		 */
		Node(final Point2D location, final Node next, final Node previous) {
			this.location = location;
			this.next = next;
			this.previous = previous;
		}

		/**
		 * 座標
		 */
		Point2D location;

		/**
		 * 次の頂点
		 */
		Node next;

		/**
		 * 前の頂点
		 */
		Node previous;

		public String toString() {
			return this.location.toString();
		}
	}

	/**
	 * 描画を更新します。
	 * @param panel パネル
	 */
	void update(final DemoPanel panel) {
		panel.clear();
		for (final Node polygon : this.polygons) {
			for (Node node = polygon; node != null; node = node.next) {
				if (node.next != null || !this.isOpen) {
					final Point2D p1 = node.location;
					final Point2D p2 = node.next != null ? node.next.location
							: polygon.location;
					final Line2D line = new Line2D.Double(p1, p2);
					panel.add(line);
				}
			}
			for (Node node = polygon; node != null; node = node.next) {
				final double r = this.radius;
				final Point2D point = node.location;
				final Ellipse2D ellipse = new Ellipse2D.Double(
						point.getX() - r, point.getY() - r, r * 2, r * 2);
				panel.add(ellipse);
			}
		}
		if (this.selection != null) {
			final double r = this.radius * 1.2;
			final Ellipse2D selection = new Ellipse2D.Double(
					this.selection.location.getX() - r, this.selection.location
							.getY()
							- r, r * 2, r * 2);
			panel.add(selection);
			panel.setBorderColor(selection, Color.RED);
			panel.setFillColor(selection, null);
			if (this.selection.next != null) {
				final Ellipse2D selection2 = new Ellipse2D.Double(
						this.selection.next.location.getX() - r,
						this.selection.next.location.getY() - r, r * 2, r * 2);
				panel.add(selection2);
				panel.setBorderColor(selection2, Color.BLUE);
				panel.setFillColor(selection2, null);
			}
		}
		panel.repaint();
	}

	/**
	 * @throws FileNotFoundException ファイル未検出例外
	 */
	void save() throws FileNotFoundException {
		final PrintWriter out = new PrintWriter(this.dataFile);
		final PrintWriter out2 = new PrintWriter(new File("car" + File.separator + this.stageName + ".java"));
		out2.println("package car;");
		out2.println();
		out2.println("import java.awt.geom.Point2D;");
		out2.println();
		out2.println("/**");
		out2.println(" * ステージデータです。このファイルを変更しても、破棄されます。");
		out2.println(" */");
		out2.println("public class " + this.stageName + " {");
		out2.println("\t/**");
		out2.println("\t * データ");
		out2.println("\t */");
		out2.println("\tpublic final Point2D[][] DATA = new Point2D["
				+ this.polygons.length + "][];");
		out2.println();
		out2.println("\t/**");
		out2.println("\t * コンストラクタです。");
		out2.println("\t * @param zoomX x方向の倍率");
		out2.println("\t * @param zoomY y方向の倍率");
		out2.println("\t */");
		out2.println("\tpublic " + this.stageName + "(final double zoomX, final double zoomY) {");
		int i = 0;
		for (final Node polygon : this.polygons) {
			out2.print("\t\tthis.DATA[" + i + "] = new Point2D[] {");
			for (Node node = polygon; node != null; node = node.next) {
				if (node.next != null) {
					out.print(node.location.getX() + ", "
							+ node.location.getY() + ", ");
					out2.print("new Point2D.Double(" + node.location.getX()
							+ " * zoomX, " + node.location.getY() + " * zoomY), ");

				} else {
					out.print(node.location.getX() + ", "
							+ node.location.getY());
					out2.print("new Point2D.Double(" + node.location.getX()
							+ " * zoomX, " + node.location.getY() + " * zoomY)");
				}
			}
			out.println();
			out2.println("};");
			i++;
		}
		out2.println("\t}");
		out2.println("}");
		out.close();
		out2.close();
	}

	/**
	 * @throws FileNotFoundException ファイル未検出例外
	 */
	public void load() throws FileNotFoundException {
		final Scanner scanner = new Scanner(this.dataFile);
		int i = 0;
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final Scanner scanner2 = new Scanner(line);
			scanner2.useDelimiter(Pattern.compile(",[\\ \t]*"));
			Node lastNode = null;
			while (scanner2.hasNextDouble()) {
				final double x = scanner2.nextDouble();
				if (scanner2.hasNextDouble()) {
					final double y = scanner2.nextDouble();
					if (lastNode == null) {
						final Node node = new Node(new Point2D.Double(x, y),
								null, null);
						this.polygons[i] = node;
						lastNode = node;
					} else {
						final Node node = new Node(new Point2D.Double(x, y),
								null, lastNode);
						lastNode.next = node;
						lastNode = node;
					}
				}
			}
			scanner2.close();
			i++;
		}
		scanner.close();
	}

}
