package server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import common.Sha;

/**
 * レートを管理するユーティリティクラスです。
 * レートの計算方法は
 * http://www5f.biglobe.ne.jp/~g-coco/chess/notebook/rating.html
 * の「簡易式」を用いています。
 * @author Kumano Tatsuo
 * Created on 2005/01/22
 */
public class Rate {
    /**
     * プロパティ
     */
    private final static Properties prop = new Properties();

    /**
     * 記録するファイル
     */
    private final static File file = new File(ServerConstants.SERVER_RATE_FILE);

    /**
     * レートを計算して書き込みます。
     * @param winName 勝者の名前
     * @param winPass 勝者のパスワード
     * @param loseName 敗者の名前
     * @param losePass 敗者のパスワード
     */
    public static void record(String winName, String winPass, String loseName, String losePass) {
        try {
            if (!file.exists()) {
                file.createNewFile();
            }
            double winRate = getRate(winName, winPass);
            double loseRate = getRate(loseName, losePass);
            double r = 16 - 0.04 * (winRate - loseRate);
            writeRate(winName, winPass, winRate + r);
            writeRate(loseName, losePass, loseRate - r);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * レートを記録します。
     * @param name 名前
     * @param pass パスワード
     * @param rate レート
     */
    private static void writeRate(String name, String pass, double rate) {
        try {
            prop.load(new FileInputStream(ServerConstants.SERVER_RATE_FILE));
            prop.setProperty(Sha.digest(name.getBytes()) + "_" + pass, String.valueOf(rate));
            prop.store(new FileOutputStream(ServerConstants.SERVER_RATE_FILE), "Football");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * レートを読み込みます。
     * @param name 名前
     * @param pass パスワード
     * @return レート
     */
    public static double getRate(String name, String pass) {
        try {
            if (!file.exists()) {
                file.createNewFile();
            }
            prop.load(new FileInputStream(ServerConstants.SERVER_RATE_FILE));
            return Double.parseDouble(prop.getProperty(Sha.digest(name.getBytes()) + "_" + pass,
                    ServerConstants.SERVER_RATE_DEFAULT));
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Double.parseDouble(ServerConstants.SERVER_RATE_DEFAULT);
    }
}
