package server;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Calendar;
import java.util.Random;
import common.CommonConstants;
import common.Replace;

/**
 * プレイヤを表すクラスです。
 * @author Kumano Tatsuo
 * Created on 2005/01/18 22:18:24
 */
public class Player {
    /**
     * クライアントからの通信を待ち受けるスレッドです。
     * @author Kumano Tatsuo
     * Created on 2005/01/18 23:07:58
     */
    private class InThread extends Thread {
        public void run() {
            try {
                while (true) {
                    byte command = Player.this.in.readByte();
                    switch (command) {
                    case CommonConstants.COMMAND_MY_NAME:
                        String name = Replace.replace(Player.this.in.readUTF(),
                                CommonConstants.TABOO_EXPRESSION, "");
                        name = name.substring(0, Math.min(name.length(), CommonConstants.NAME_LENGTH));
                        if (name.length() == 0) {
                            name = String.valueOf(new Random().nextInt(256));
                        }
                        System.out.println("LOG: user " + name + " logged in on "
                                + Calendar.getInstance().getTime());
                        Player.this.name = name;
                        break;
                    case CommonConstants.COMMAND_MY_PASS:
                        Player.this.pass = Player.this.in.readUTF();
                        break;
                    case CommonConstants.COMMAND_LEFT_PRESSED:
                        Player.this.isLeft = true;
                        Player.this.isRight = false;
                        Player.this.isUp = false;
                        Player.this.isDown = false;
                        break;
                    case CommonConstants.COMMAND_RIGHT_PRESSED:
                        Player.this.isRight = true;
                        Player.this.isLeft = false;
                        Player.this.isUp = false;
                        Player.this.isDown = false;
                        break;
                    case CommonConstants.COMMAND_UP_PRESSED:
                        Player.this.isUp = true;
                        Player.this.isLeft = false;
                        Player.this.isRight = false;
                        Player.this.isDown = false;
                        break;
                    case CommonConstants.COMMAND_DOWN_PRESSED:
                        Player.this.isDown = true;
                        Player.this.isLeft = false;
                        Player.this.isRight = false;
                        Player.this.isUp = false;
                        break;
                    case CommonConstants.COMMAND_BOMB_PRESSED:
                        Player.this.isBomb = true;
                        break;
                    case CommonConstants.COMMAND_KEY_RELEASED:
                        Player.this.isLeft = false;
                        Player.this.isRight = false;
                        Player.this.isUp = false;
                        Player.this.isDown = false;
                        break;
                    default:
                        System.err.println("WARNING: UNKNOWN COMMAND " + command);
                        break;
                    }
                }
            } catch (IOException e) {
                Player.this.isDisconnected = true;
                System.out.println("LOG: user " + Player.this.name + " disconnected on "
                        + Calendar.getInstance().getTime());
            }
        }
    }

    /**
     * 切断されたかどうか
     */
    boolean isDisconnected;

    /**
     * プレイヤの列番号
     */
    private byte col;

    /**
     * 入力ストリーム
     */
    final DataInputStream in;

    /**
     * 爆弾キーが押されたかどうか
     */
    boolean isBomb;

    /**
     * 下キーが押されているかどうか
     */
    boolean isDown;

    /**
     * 左キーが押されているかどうか
     */
    boolean isLeft;

    /**
     * ゲーム中かどうか
     */
    private boolean isPlaying;

    /**
     * 右キーが押されているかどうか
     */
    boolean isRight;

    /**
     * 上キーが押されているかどうか
     */
    boolean isUp;

    /**
     * 名前
     */
    String name;

    /**
     * 番号
     */
    final private int number;

    /**
     * 出力ストリーム
     */
    final private DataOutputStream out;

    /**
     * パスワード
     */
    String pass;

    /**
     * プレイヤの行番号
     */
    private byte row;

    /**
     * アニメーションが停止する時刻
     */
    private long stopTime;

    /**
     * 爆弾の数
     */
    private int bombNumber;

    /**
     * 火力
     */
    private byte power;
    
    /**
     * プレイヤを初期化します。
     * @param in 入力ストリーム
     * @param out 出力ストリーム
     * @param number 番号
     * @throws IOException 
     */
    public Player(DataInputStream in, DataOutputStream out, int number) throws IOException {
        this.in = in;
        this.out = out;
        this.number = number;
        this.bombNumber = ServerConstants.SERVER_PLAYER_DEFAULT_BOMB_NUMBER;
        this.power = ServerConstants.SERVER_PLAYER_DEFAULT_POWER;
        if (this.out != null) {
            this.out.writeByte(CommonConstants.COMMAND_YOUR_NUMBER);
            this.out.writeInt(this.number);
        }
        new InThread().start();
    }

    /**
     * @return プレイヤの行
     */
    public byte getCol() {
        return this.col;
    }

    /**
     * @return プレイヤの名前
     */
    public String getName() {
        return this.name;
    }

    /**
     * @return プレイヤのパスワード
     */
    public String getPass() {
        return this.pass;
    }

    /**
     * @param isBomb 爆弾キーが押されたかどうか
     */
    public void setBomb(boolean isBomb) {
        this.isBomb = isBomb;
    }
    /**
     * @return プレイヤの行
     */
    public byte getRow() {
        return this.row;
    }

    /**
     * @return アニメーションが停止する時刻
     */
    public long getStopTime() {
        return this.stopTime;
    }

    /**
     * @return 爆弾キーが押されたかどうか
     */
    public boolean isBomb() {
        return this.isBomb;
    }

    /**
     * @return 下キーが押されているかどうか
     */
    public boolean isDown() {
        return this.isDown;
    }

    /**
     * @return 左キーが押されているかどうか
     */
    public boolean isLeft() {
        return this.isLeft;
    }

    /**
     * @return ゲーム中かどうか
     */
    public boolean isPlaying() {
        return this.isPlaying;
    }

    /**
     * @return 右キーが押されているかどうか
     */
    public boolean isRight() {
        return this.isRight;
    }

    /**
     * @return 上キーが押されているかどうか
     */
    public boolean isUp() {
        return this.isUp;
    }

    /**
     * @param col プレイヤの列
     */
    public void setCol(byte col) {
        this.col = col;
    }

    /**
     * @param isPlaying ゲーム中かどうか
     */
    public void setPlaying(boolean isPlaying) {
        this.isPlaying = isPlaying;
    }

    /**
     * @param row プレイヤの行
     */
    public void setRow(byte row) {
        this.row = row;
    }

    /**
     * @param stopTime アニメーションが停止する時刻
     */
    public void setStopTime(long stopTime) {
        this.stopTime = stopTime;
    }

    public String toString() {
        return "[number = " + this.number + ", name = " + this.name + ", pass = " + this.pass
                + ", row = " + this.row + ", col = " + this.col + "]";
    }

    /**
     * @return 番号
     */
    public int getNumber() {
        return this.number;
    }

    /**
     * プレイヤの場所をクライアントに伝えます。
     * @param number 番号
     * @param time 停止するまでの時間
     * @param row 行
     * @param col 列
     * @throws IOException 
     */
    public void tellLocation(int number, int time, byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_PLAYER_LOCATION);
        this.out.writeInt(number);
        this.out.writeInt(time);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * ボールの場所をクライアントに伝えます。
     * @param number 番号
     * @param time 停止するまでの時間
     * @param row 行
     * @param col 列
     * @throws IOException 
     */
    public void tellBombLocation(int number, int time, byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_BOMB_LOCATION);
        this.out.writeInt(number);
        this.out.writeInt(time);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * プレイヤの名前をクライアントに伝えます。
     * @param number 番号
     * @param name 名前
     * @throws IOException
     */
    public void tellName(int number, String name) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_PLAYER_NAME);
        this.out.writeInt(number);
        this.out.writeUTF(name);
    }

    /**
     * ブロックがないことをクライアントに伝えます。
     * @param row 行
     * @param col 列
     * @throws IOException
     */
    public void tellNoBlock(byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_DATA_NO_BLOCK);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * 壊せないブロックがあることをクライアントに伝えます。
     * @param row 行
     * @param col 列
     * @throws IOException
     */
    public void tellHardBlock(byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_DATA_HARD_BLOCK);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * 壊せるブロックがあることをクライアントに伝えます。
     * @param row 行
     * @param col 列
     * @throws IOException
     */
    public void tellSoftBlock(byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_DATA_SOFT_BLOCK);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * アイテムがあることをクライアントに伝えます。
     * @param row 行
     * @param col 列
     * @throws IOException
     */
    public void tellItem(byte row, byte col) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_DATA_ITEM);
        this.out.writeByte(row);
        this.out.writeByte(col);
    }

    /**
     * プレイヤが死んだことをクライアントに伝えます。
     * @param number 番号
     * @throws IOException
     */
    public void tellDead(int number) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_PLAYER_DEAD);
        this.out.writeInt(number);
    }

    /**
     * プレイヤのレートをクライアントに伝えます。
     * @param number 番号
     * @param rate レート
     * @throws IOException
     */
    public void tellRate(int number, int rate) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_PLAYER_RATE);
        this.out.writeInt(number);
        this.out.writeInt(rate);
    }

    /**
     * 爆弾が爆発したことを伝えます。
     * @param number 番号
     * @throws IOException
     */
    public void tellBombBlow(int number) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_BOMB_BLOW);
        this.out.writeInt(number);
    }
    
    /**
     * 水平方向の炎を伝えます。
     * @param row 行
     * @param col 列
     * @param length 長さ
     * @throws IOException
     */
    public void tellHorizontalBlow(byte row, byte col, byte length) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_HORIZONTAL_BLOW);
        this.out.writeByte(row);
        this.out.writeByte(col);
        this.out.writeByte(length);
    }
    /**
     * 垂直方向の炎を伝えます。
     * @param row 行
     * @param col 列
     * @param length 長さ
     * @throws IOException
     */
    public void tellVerticalBlow(byte row, byte col, byte length) throws IOException {
        this.out.writeByte(CommonConstants.COMMAND_VERTICAL_BLOW);
        this.out.writeByte(row);
        this.out.writeByte(col);
        this.out.writeByte(length);
    }
    
    /**
     * @return 切断されたかどうか
     */
    public boolean isDisconnected() {
        return this.isDisconnected;
    }

    /**
     * @return 爆弾の数
     */
    public int getBombNumber() {
        return this.bombNumber;
    }

    /**
     * @param bombNumber 爆弾の数
     */
    public void setBombNumber(int bombNumber) {
        this.bombNumber = bombNumber;
    }
    /**
     * @return 火力
     */
    public byte getPower() {
        return this.power;
    }
    /**
     * @param power 火力
     */
    public void setPower(byte power) {
        this.power = power;
    }
}
