package map;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.print.PrinterException;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import javax.print.PrintException;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.KeyStroke;
import javax.swing.filechooser.FileFilter;

/**
 * 地図を表示するフレームです。
 * @author Kumano Tatsuo
 * Created on 2003/11/01, 16:00
 */
public class MainFrame extends JFrame {
	/**
	 * 地図を表示するパネル
	 */
	final MapPanel	panel;

	/**
	 * 新しくフレームを初期化します。
	 * @param maps 地図
	 * @param panel 地図を表示するパネル
	 * @throws IOException 入出力例外
	 */
	public MainFrame(final Map<String, MapData> maps, final MapPanel panel) throws IOException {
		this.panel = panel;
		this.setTitle("Map");
		this.setLayout(new BorderLayout());
		final JSplitPane splitPane = new JSplitPane();
		this.add(splitPane, BorderLayout.CENTER);
		final JPanel sidePanel = new SidePanel(maps, panel);
		sidePanel.setPreferredSize(new Dimension(200, 0));
		sidePanel.setMinimumSize(new Dimension(0, 0));
		splitPane.setLeftComponent(sidePanel);
		splitPane.setRightComponent(panel);
		splitPane.setOneTouchExpandable(true);
		final JMenuBar menuBar = new JMenuBar();
		this.add(menuBar, BorderLayout.NORTH);
		final JMenu fileMenu = new JMenu("ファイル(F)");
		fileMenu.setMnemonic('F');
		menuBar.add(fileMenu);
		final JMenuItem imageExportItem = new JMenuItem("ラスタ画像としてエクスポート（PNG、JPEG、BMPファイル）(I)...");
		imageExportItem.setMnemonic('I');
		imageExportItem.addActionListener(new ImageExportListener());
		fileMenu.add(imageExportItem);
		final JMenuItem psExportItem = new JMenuItem("PSファイルとしてエクスポート(E)...");
		psExportItem.setMnemonic('E');
		psExportItem.addActionListener(new PSExportListener());
		fileMenu.add(psExportItem);
		fileMenu.addSeparator();
		final JMenuItem printItem = new JMenuItem("印刷(P)...");
		printItem.setMnemonic('P');
		printItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, InputEvent.CTRL_DOWN_MASK));
		printItem.addActionListener(new PrintListener());
		fileMenu.add(printItem);
		fileMenu.addSeparator();
		final JMenuItem exitItem = new JMenuItem("終了(X)");
		exitItem.setMnemonic('X');
		exitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F4, InputEvent.ALT_DOWN_MASK));
		exitItem.addActionListener(new ExitListener());
		fileMenu.add(exitItem);
		final JMenu viewMenu = new JMenu("表示(V)");
		viewMenu.setMnemonic('V');
		menuBar.add(viewMenu);
		final JMenuItem zoomInItem = new JMenuItem("拡大(I)");
		zoomInItem.setMnemonic('I');
		zoomInItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_SEMICOLON,
				InputEvent.SHIFT_DOWN_MASK | InputEvent.CTRL_DOWN_MASK));
		zoomInItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.zoomIn();
			}
		});
		viewMenu.add(zoomInItem);
		final JMenuItem zoomOutItem = new JMenuItem("縮小(O)");
		zoomOutItem.setMnemonic('O');
		zoomOutItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_MINUS,
				InputEvent.CTRL_DOWN_MASK));
		zoomOutItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.zoomOut();
			}
		});
		viewMenu.add(zoomOutItem);
		viewMenu.addSeparator();
		final JMenuItem zoomWholeItem = new JMenuItem("全域表示(H)");
		zoomWholeItem.setMnemonic('H');
		zoomWholeItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_0,
				InputEvent.CTRL_DOWN_MASK));
		zoomWholeItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.zoomWhole();
			}
		});
		viewMenu.add(zoomWholeItem);
		final JMenuItem zoomWideItem = new JMenuItem("広域表示(W)");
		zoomWideItem.setMnemonic('W');
		zoomWideItem.setAccelerator(KeyStroke
				.getKeyStroke(KeyEvent.VK_1, InputEvent.CTRL_DOWN_MASK));
		zoomWideItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.zoomWide();
			}
		});
		viewMenu.add(zoomWideItem);
		final JMenuItem zoomDetailItem = new JMenuItem("詳細表示(D)");
		zoomDetailItem.setMnemonic('D');
		zoomDetailItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_2,
				InputEvent.CTRL_DOWN_MASK));
		zoomDetailItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.zoomDetail();
			}
		});
		viewMenu.add(zoomDetailItem);
		viewMenu.addSeparator();
		final JMenuItem moveNorthItem = new JMenuItem("北へ移動(R)");
		moveNorthItem.setMnemonic('R');
		moveNorthItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP,
				InputEvent.ALT_DOWN_MASK));
		moveNorthItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.scroll(0, -getHeight() / 4);
			}
		});
		viewMenu.add(moveNorthItem);
		final JMenuItem moveSouthItem = new JMenuItem("南へ移動(U)");
		moveSouthItem.setMnemonic('U');
		moveSouthItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN,
				InputEvent.ALT_DOWN_MASK));
		moveSouthItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.scroll(0, getHeight() / 4);
			}
		});
		viewMenu.add(moveSouthItem);
		final JMenuItem moveWestItem = new JMenuItem("西へ移動(T)");
		moveWestItem.setMnemonic('T');
		moveWestItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_LEFT,
				InputEvent.ALT_DOWN_MASK));
		moveWestItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.scroll(-getWidth() / 4, 0);
			}
		});
		viewMenu.add(moveWestItem);
		final JMenuItem moveEastItem = new JMenuItem("東へ移動(E)");
		moveEastItem.setMnemonic('E');
		moveEastItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT,
				InputEvent.ALT_DOWN_MASK));
		moveEastItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.scroll(getWidth() / 4, 0);
			}
		});
		viewMenu.add(moveEastItem);
		viewMenu.addSeparator();
		final JMenuItem fontSizeIncrementItem = new JMenuItem("文字を大きく(L)");
		fontSizeIncrementItem.setMnemonic('L');
		fontSizeIncrementItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.increaseFontSize();
			}
		});
		viewMenu.add(fontSizeIncrementItem);
		final JMenuItem fontSizeDecrementItem = new JMenuItem("文字を小さく(S)");
		fontSizeDecrementItem.setMnemonic('S');
		fontSizeDecrementItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.decreaseFontSize();
			}
		});
		viewMenu.add(fontSizeDecrementItem);
		viewMenu.addSeparator();
		final JMenuItem fontSizeResetItem = new JMenuItem("文字を標準サイズに(N)");
		fontSizeResetItem.setMnemonic('N');
		fontSizeResetItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.resetFontSize();
			}
		});
		viewMenu.add(fontSizeResetItem);
		viewMenu.addSeparator();
		final JMenuItem darkerItem = new JMenuItem("色を鮮やかに(V)");
		darkerItem.setMnemonic('V');
		darkerItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.increaseSaturation();
			}
		});
		viewMenu.add(darkerItem);
		final JMenuItem brighterItem = new JMenuItem("色を淡く(G)");
		brighterItem.setMnemonic('G');
		brighterItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.decreaseSaturation();
			}
		});
		viewMenu.add(brighterItem);
		viewMenu.addSeparator();
		final JMenuItem brightnessResetItem = new JMenuItem("標準の色合い(M)");
		brightnessResetItem.setMnemonic('M');
		brightnessResetItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				panel.resetSaturation();
			}
		});
		viewMenu.add(brightnessResetItem);
	}

	/**
	 * 印刷の実装です。
	 */
	private class PrintListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				MainFrame.this.panel.print();
			} catch (PrintException e1) {
				e1.printStackTrace();
			}
		}
	}

	/**
	 * PSファイルを出力する実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/03/20 16:42:34
	 */
	private class PSExportListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				final JFileChooser chooser = new JFileChooser(new File("."));
				chooser.addChoosableFileFilter(new FileFilter() {
					@Override
					public boolean accept(File file) {
						return file.getName().toLowerCase().endsWith(".ps");
					}

					@Override
					public String getDescription() {
						return "PSファイル（*.ps）";
					}
				});
				chooser.showDialog(MainFrame.this, "エクスポート");
				final File selectedFile = chooser.getSelectedFile();
				if (selectedFile != null) {
					final String fileName = selectedFile.getName();
					if (fileName != null) {
						if (fileName.toLowerCase().endsWith(".ps")) {
							MainFrame.this.panel.printPS(selectedFile);
						} else {
							MainFrame.this.panel.printPS(new File(fileName + ".ps"));
						}
					}
				}
			} catch (PrinterException e1) {
				e1.printStackTrace();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	/**
	 * ラスタ画像に出力する実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/02/28 15:39:42
	 */
	private class ImageExportListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				final JFileChooser chooser = new JFileChooser(new File("."));
				chooser.addChoosableFileFilter(new FileFilter() {
					@Override
					public boolean accept(File file) {
						return file.getName().toLowerCase().endsWith(".png")
								|| file.getName().toLowerCase().endsWith(".jpg")
								|| file.getName().toLowerCase().endsWith(".bmp");
					}

					@Override
					public String getDescription() {
						return "ラスタ画像ファイル（*.png、*.jpg、*.bmp）";
					}
				});
				chooser.showDialog(MainFrame.this, "エクスポート");
				final File selectedFile = chooser.getSelectedFile();
				if (selectedFile != null) {
					final String fileName = selectedFile.getName();
					if (fileName != null) {
						if (fileName.toLowerCase().endsWith(".bmp")) {
							MainFrame.this.panel.printBitmap(selectedFile, "bmp");
						} else if (fileName.toLowerCase().endsWith(".jpg")) {
							MainFrame.this.panel.printBitmap(selectedFile, "jpg");
						} else if (fileName.toLowerCase().endsWith(".png")) {
							MainFrame.this.panel.printBitmap(selectedFile, "png");
						}
					}
				}
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}
	}

	/**
	 * 終了の実装です。
	 * @author Kumano Tatsuo
	 * Created on 2005/02/28 17:49:02
	 */
	private class ExitListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			System.exit(0);
		}
	}
}
