/*
 * $Id: MBR.java,v 1.2 2002/03/17 08:56:46 nie Exp $
 */

package jp.jasminesoft.xmlcat;

import java.io.Serializable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;

/**
 * MBR は矩形を表現するクラスです。
 *
 * @author	Yoshinori Nie
 * @version     $Revision: 1.2 $ $Date: 2002/03/17 08:56:46 $
 */

public class MBR implements Serializable {
    static final long serialVersionUID = -5295278249650569180L;

    /*
     * 以下、左下隅を原点 (0,0) と仮定しています。
     */

    /**
     * x1 は MBR の左下隅の X 座標値です。
     * デフォルト値は、Float の最大値としておきます。
     */
    private float x1 = Float.POSITIVE_INFINITY;

    /**
     * y1 は MBR の左下隅の Y 座標値です。
     * デフォルト値は、Float の最大値としておきます。
     */
    private float y1 = Float.POSITIVE_INFINITY;

    /**
     * x2 は MBR の右上隅の X 座標値です。
     * デフォルト値は、Float の最小値としておきます。
     */
    private float x2 = 0.0f;

    /**
     * y2 は MBR の右上隅の Y 座標値です。
     * デフォルト値は、Float の最小値としておきます。
     */
    private float y2 = 0.0f;

    /**
     * デフォルトコンストラクタ
     */
    public MBR() {}
   
    /**
     * コンストラクタ。
     * 4 点の座標値を引数とします。
     *
     * @param		x1 (左下隅 X 座標値)
     * @param		y1 (左下隅 Y 座標値)
     * @param		x2 (右上隅 X 座標値)
     * @param		y2 (右上隅 Y 座標値)
     */
    public MBR(float x1, float y1, float x2, float y2) {
	this.x1 = x1;
	this.y1 = y1;
	this.x2 = x2;
	this.y2 = y2;
	rearrange();
    }

    /**
     * コンストラクタ。
     * 4 点の座標値が含まれた文字列を引数とします。
     *
     * @param	str 4 点の座標値が含まれた文字列
     */
    public MBR(String str) {
	StringTokenizer st = new StringTokenizer(str);
	try {
	    x1 = Float.parseFloat(st.nextToken());
	    y1 = Float.parseFloat(st.nextToken());
	    x2 = Float.parseFloat(st.nextToken());
	    y2 = Float.parseFloat(st.nextToken());
	} catch (NoSuchElementException e) {
	    // 再初期化
	    x1 = Float.POSITIVE_INFINITY;
	    y1 = Float.POSITIVE_INFINITY;
	    x2 = 0.0f;
	    y2 = 0.0f;
	}
	rearrange();
    }

    /**
     * 座標値を正しい順番に並べ直すためのメソッド。
     * コンストラクタから呼び出される。
     */
    private void rearrange() {
	float tmp;
	if (this.x1 > this.x2) {
	    tmp = this.x1;
	    this.x1 = this.x2;
	    this.x2 = tmp;
	}
	if (this.y1 > this.y2) {
	    tmp = this.y1;
	    this.y1 = this.y2;
	    this.y2 = tmp;
	}
    }

    /**
     * MBR を構成する始点(左下隅) の X 座標値を返します。
     *
     * @return	始点の X 座標値
     */
    public float getSX() { return x1; }

    /**
     * MBR を構成する始点(左下隅) の Y 座標値を返します。
     *
     * @return	始点の Y 座標値
     */
    public float getSY() { return y1; }

    /**
     * MBR を構成する終点(右上隅) の X 座標値を返します。
     *
     * @return	始点の X 座標値
     */
    public float getEX() { return x2; }

    /**
     * MBR を構成する終点(右上隅) の Y 座標値を返します。
     *
     * @return	始点の Y 座標値
     */
    public float getEY() { return y2; }

    /**
     * merge() は、引数の MBR と自身を合体させた、より大きな
     * MBR へと自身を変化させます。
     *
     * @param		合併対象となる MBR
     * @exception	IllegalCoordinateGeometryException
     */
    public void merge(MBR mbr) {
	if (mbr == null)
	    return;
	
	float sx = mbr.getSX();
	float sy = mbr.getSY();
	float ex = mbr.getEX();
	float ey = mbr.getEY();

	merge(sx, sy, ex, ey);
    }

    /**
     * merge() は、引数の MBR と自身を合体させた、より大きな
     * MBR へと自身を変化させます。
     *
     * @param		sx (左下隅 X 座標値)
     * @param		sy (左下隅 Y 座標値)
     * @param		ex (右上隅 X 座標値)
     * @param		ey (右上隅 Y 座標値)
     */
    public void merge(float sx, float sy, float ex, float ey) {
	if (sx < x1) x1 = sx;
	if (sy < y1) y1 = sy;
	if (x2 < ex) x2 = ex;
	if (y2 < ey) y2 = ey;
    }

    /*
     * isIn() による検索は、自身が引数の MBR に包含されている場合に 
     * true を返します。
     *
     * @param	mbr	比較対象の MBR
     * @return	引数 MBR の中に自身が包含されていれば true を返す
     */
    public boolean isIn(MBR mbr) {
	boolean retCode = false;

	if (mbr == null)
	    return retCode;
	
	float dx1 = mbr.getSX();
	float dx2 = mbr.getEX();
	float dy1 = mbr.getSY();
	float dy2 = mbr.getEY();

	if (dx1 <= x1 && dx2 >= x2 && dy1 <= y1 && dy2 >= y2) {
	    retCode = true;
	}

	return retCode;
    }

    /**
     * contains() による検索は、自身が引数の MBR の
     * 一部に含まれている場合に true を返します。
     * contains() は、(意味的に) isIn() を包含するものです。
     *
     * @param	mbr	比較対象の MBR
     * @return	引数 MBR の一部に自身が含まれていれば true を返す
     */
    public boolean contains(MBR mbr) {
	if (mbr == null)
	    return false;

	int pattern = 0;
	
	float dx1 = mbr.getSX();
	float dx2 = mbr.getEX();
	float dy1 = mbr.getSY();
	float dy2 = mbr.getEY();

	if (dx1 <= x1 && dx2 >= x2 && dy1 <= y1 && dy2 >= y2)
	    // (1) ... this means isIn() method.
	    pattern = 1;
	else if (dx2 >= x1 && dx1 <= x2 && dy1 >= y1 && dy2 <= y2) 
	    //else if (dx2 >= x1 && dx2 <= x2 && dy1 >= y1 && dy2 <= y2):bug?
	    // include pattern
	    // (2)
	    pattern = 2;
	else if (dx1 <= x2 && dx2 >= x1 && dy2 >= y1 && dy2 <= y2)
	    // (3)
	    pattern = 3;
	else if (dx1 <= x2 && dx2 >= x1 && dy1 >= y1 && dy1 <= y2)
	    // (4)
	    pattern = 4;
	else if (dx2 >= x1 && dx2 <= x2 && dy1 <= y2 && dy2 >= y1)
	    // (5)
	    pattern = 5;
	else if (dx1 >= x1 && dx1 <= x2 && dy1 <= y2 && dy2 >= y1)
	    // (6)
	    pattern = 6;

	return (pattern > 0) ? true : false;
    }

    /**
     * 引数の MBR と重なった部分を算出し、重なった部分を新たな MBR
     * として生成して返すメソッドです。重なった部分がない場合は null
     * を返します。
     *
     * @param mbr MBR
     * @return 自身と引数 MBR との重なり部分を新たな MBR として返す。
     */
    public MBR getOverlapMBR(MBR mbr) {
	if (mbr == null || !contains(mbr))
	    return null;

	float dx1 = mbr.getSX();
	float dx2 = mbr.getEX();
	float dy1 = mbr.getSY();
	float dy2 = mbr.getEY();

	float nx1 = dx1;
	float nx2 = dx2;
	float ny1 = dy1;
	float ny2 = dy2;

	if (dx1 < x1)
	    nx1 = x1;
	if (dx2 > x2)
	    nx2 = x2;
	if (dy1 < y1)
	    ny1 = y1;
	if (dy2 > y2)
	    ny2 = y2;

	return new MBR(nx1, ny1, nx2, ny2); 
    }

    /**
     * MBR の面積を返します。
     * ここでは、単純に四辺形の面積を算出するのみです。
     * (緯度経度値が含まれていた場合、それを平面直角座標系に変換
     *  するなどの処理は含まれない。)
     */
    public float getArea() {
	float area = (x2 - x1) * (y2 - y1);
	return area;
    }

    /**
     * この MBR が有効な値かどうかを返します。
     *
     * @return この MBR が有効であれば<code>true</code>を返します。
     * そうでなければ<code>false</code>を返します。
     */
    public boolean isValid() {
	if ((x1 == Float.POSITIVE_INFINITY || x1 == 0.0f) && 
	    (y1 == Float.POSITIVE_INFINITY || y1 == 0.0f) && 
	    x2 == 0.0f && y2 == 0.0f)
	    return false;
	return true;
    }

    /**
     * equals() は引数の MBR と同じ値であれば true を返します。
     *
     * @param		Object (MBR 型を想定)
     */
    public boolean equals(Object obj) {
	boolean retCode = false;
	if (obj != null && obj instanceof MBR) {
	    MBR mbr = (MBR)obj;
	    if (x1 == mbr.getSX() && y1 == mbr.getSY() &&
		x2 == mbr.getEX() && y2 == mbr.getEY())
		retCode = true;
	}
	return retCode;
    }

    /**
     * clone() は、自身のコピーを返します。
     *
     * @return	Object
     */
    public Object clone() {
	return new MBR(x1, y1, x2, y2);
    }

    /**
     * toString() はデバッグ情報出力用です。
     *
     * @return デバッグ情報
     */
    public String toString() {
	StringBuffer sb = new StringBuffer();
	sb.append("<MBR ");
	sb.append("x1=\""+x1+"\" ");
	sb.append("y1=\""+y1+"\" ");
	sb.append("x2=\""+x2+"\" ");
	sb.append("y2=\""+y2+"\">");
	return sb.toString();
    }

    /**
     * テスト用のメソッド
     */
    public static void test() {
	/*
	 * 単純テスト
	 *
	MBR mbr = new MBR(0,0,100,100);
	System.out.println("mbr:"+mbr);
	float area = mbr.getArea();
	System.out.println("Area:"+area);
	*/
	
	/*
	 * オーバーラップテスト
	 *
	MBR mbr = new MBR(100,100,200,200);
	MBR r_mbr = new MBR(0,150,150,250);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(150,150,160,250);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(150,150,250,250);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(0,150,150,160);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(110,110,190,190);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(150,150,250,160);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(0,0,150,150);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(150,0,160,150);
	System.out.println(mbr.getOverlapMBR(r_mbr));

	r_mbr = new MBR(150,0,250,150);
	System.out.println(mbr.getOverlapMBR(r_mbr));
	*/

	/*
	 * isInvalid テスト
	 */
	/*
	MBR mbr = new MBR();
	System.out.println(mbr+":"+mbr.isValid());
	mbr = new MBR(0.0f, 0.0f, 0.0f, 0.0f);
	System.out.println(mbr+":"+mbr.isValid());
	mbr = new MBR(0.0f, 0.0f, 100.0f, 100.0f);
	System.out.println(mbr+":"+mbr.isValid());
	mbr = new MBR(100.0f, 100.0f, 0.0f, 0.0f);
	System.out.println(mbr+":"+mbr.isValid());
	*/
    }

    /**
     * テストメソッド駆動ルーチン
     */
    public static void main(String[] args) {
	MBR.test();
    }
}
