/*
 * $Id: Converter.java,v 1.3 2002/03/17 14:00:10 nie Exp $
 */

package jp.jasminesoft.gcat.scalc;

import jp.jasminesoft.xmlcat.MBR;

/** 
 * 緯度経度と平面直角座標を変換するユーティリティクラス。
 * 
 * @version $Revision: 1.3 $ $Date: 2002/03/17 14:00:10 $
 * @author  Yoshinori Nie
 */

public class Converter {
    /**
     * 引数の MBR の座標値を平面直角座標系に変換した、新しい MBR を返す。
     *
     * @param mbr ソースとなる MBR (緯度経度値)
     * @return 平面直角座標系の値をもった MBR
     */
    public static MBR latlong2xy(MBR mbr) {
	return Converter.latlong2xy(mbr, -1);
    }

    /**
     * 引数の MBR の座標値を平面直角座標系に変換した、新しい MBR を返す。
     *
     * @param mbr ソースとなる MBR (緯度経度値)
     * @param kei 平面直角座標系の系番号。-1 の場合は、自動算出される。
     * @return 平面直角座標系の値をもった MBR
     */
    public static MBR latlong2xy(MBR mbr, int kei) {
	LatLong2XY conv = new LatLong2XY(kei);
	conv.setLongitude(mbr.getSX());
	conv.setLatitude(mbr.getSY());
	float sx = (float)conv.getX();
	float sy = (float)conv.getY();
	conv.setLongitude(mbr.getEX());
	conv.setLatitude(mbr.getEY());
	float ex = (float)conv.getX();
	float ey = (float)conv.getY();
	return new MBR(sx, sy, ex, ey);
    }

    /**
     * 引数の MBR の座標値を緯度経度に変換した、新しい MBR を返す。
     *
     * @param mbr ソースとなる MBR (平面直角座標系の X,Y 値)
     * @param kei 平面直角座標系の系番号。
     * この値が 1<= kei <=19 の範囲でない場合は null を返す。
     * @return 平面直角座標系の値をもった MBR
     */
    public static MBR xy2latlong(MBR mbr, int kei) {
	if (kei < 1 || kei > 19)
	    return null;

	XY2LatLong conv = new XY2LatLong(kei);
	conv.setX(mbr.getSX());
	conv.setY(mbr.getSY());
	float sx = (float)conv.getLongitude();
	float sy = (float)conv.getLatitude();
	conv.setX(mbr.getEX());
	conv.setY(mbr.getEY());
	float ex = (float)conv.getLongitude();
	float ey = (float)conv.getLatitude();
	return new MBR(sx, sy, ex, ey);
    }

    /**
     * dddmmss.s 形式の経緯度を ddd.mmss に変換します。
     *
     * @param mbr ソースとなる MBR (dddmmss.s 形式の経緯度)
     * @returm ddd.mmss 形式の経緯度を有した MBR
     */
    public static MBR toAngle(MBR mbr) {
	return new MBR(DMSconv.deg2dms(mbr.getSX()),
		       DMSconv.deg2dms(mbr.getSY()),
		       DMSconv.deg2dms(mbr.getEX()),
		       DMSconv.deg2dms(mbr.getEY()));
    }

    /**
     * テストメソッド駆動ルーチン
     */
    public static void main(String[] args) {
	/*
	 * 座標変換テスト
	 */
	MBR mbr = new MBR(0,0,10000,10000);
	System.out.println(mbr);
	MBR mbr2 = Converter.xy2latlong(mbr, 1);
	System.out.println(mbr2);
	MBR mbr3 = Converter.toAngle(mbr2);
	System.out.println(mbr3);
	MBR mbr4 = Converter.latlong2xy(mbr3, 1);
	System.out.println(mbr4);
    }
}

