/* $Id: scmos.c,v 1.1.1.1 2003/08/24 04:53:26 rtakano Exp $
 * Scheme OS (OSKit + TinyScheme)
 * Copyright (c) 2003 TAKANO Ryousei <rtakano@users.sourceforge.net>
 */
/*
 * Copyright (c) 1996, 1998, 1999, 2000 University of Utah and the Flux Group.
 * All rights reserved.
 * 
 * This file is part of the Flux OSKit.  The OSKit is free software, also known
 * as "open source;" you can redistribute it and/or modify it under the terms
 * of the GNU General Public License (GPL), version 2, as published by the Free
 * Software Foundation (FSF).  To explore alternate licensing terms, contact
 * the University of Utah at csl-dist@cs.utah.edu or +1-801-585-3271.
 * 
 * The OSKit is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GPL for more details.  You should have
 * received a copy of the GPL along with the OSKit; see the file COPYING.  If
 * not, write to the FSF, 59 Temple Place #330, Boston, MA 02111-1307, USA.
 */

/* a default root partition */
/* /dev/hda1 -> hda / s1 */
#define DISK_NAME	"hda"
#define PARTITION_NAME	"s1"

#include <oskit/dev/dev.h>
#include <oskit/dev/osenv.h>
#include <oskit/dev/linux.h>
#include <oskit/io/blkio.h>
#include <oskit/fs/filesystem.h>
#include <oskit/fs/dir.h>
#include <oskit/fs/openfile.h>
#include <oskit/fs/linux.h>
#include <oskit/diskpart/diskpart.h>
#include <oskit/principal.h>
#include <oskit/clientos.h>
#include <oskit/startup.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

/* Partition array, filled in by the diskpart library. */
#define MAX_PARTS 10
static diskpart_t part_array[MAX_PARTS];

/* Identity of current client process. */
static oskit_principal_t *cur_principal;

#define CHECK(err, f, args) ({			\
	(err) = f args;				\
	if (err)				\
		panic(#f" failed 0x%lx", (err));\
})
#define HMM(err, f, args) ({			\
	(err) = f args;				\
	if (err)				\
		printf("** " #f " failed 0x%x\n", (err));\
})


oskit_error_t oskit_get_call_context(const struct oskit_guid *iid,
				     void **out_if)
{
  if (memcmp(iid, &oskit_iunknown_iid, sizeof(*iid)) == 0 ||
      memcmp(iid, &oskit_principal_iid, sizeof(*iid)) == 0) {
    *out_if = cur_principal;
    oskit_principal_addref(cur_principal);
    return 0;
  }

  *out_if = 0;
  return OSKIT_E_NOINTERFACE;
}

static void create_identity(void)
{
  oskit_identity_t id;
  oskit_error_t err;

  id.uid = 0;
  id.gid = 0;
  id.ngroups = 0;
  id.groups = 0;

  CHECK(err, oskit_principal_create, (&id, &cur_principal));
}

static void mount_disk(const char *rootdev)
{
  oskit_blkio_t *disk;
  oskit_blkio_t *part;
  oskit_error_t err;
  oskit_filesystem_t *rootfs;
  oskit_fsnamespace_t *fsns;
  oskit_dir_t *dir;
  char diskname[8];
  char partname[8];
  int numparts;

  /* a very dirty command line parser... */
  if (rootdev[0] != '\0') {
    strncpy(diskname, rootdev,   3); diskname[3] = '\0';
    strncpy(partname, rootdev+3, 2); partname[2] = '\0';
  } else {
    strncpy(diskname, DISK_NAME, strlen(DISK_NAME));
    strncpy(partname, PARTITION_NAME, strlen(PARTITION_NAME));
  }
  printf("%s:%s\n", diskname, partname);

  printf(">>>Initializing filesystem...\n");
  CHECK(err, fs_linux_init, ());

  printf(">>>Establishing client identity\n");
  create_identity();

  printf(">>>Opening the disk %s\n", diskname);
  err = oskit_linux_block_open(diskname, 
			       OSKIT_DEV_OPEN_READ|OSKIT_DEV_OPEN_WRITE,
			       &disk);
  if (err)
    panic("error 0x%x opening disk%s", err,
	  err == OSKIT_E_DEV_NOSUCH_DEV ? ": no such device" : "");

  printf(">>>Reading partition table and looking for partition %s\n",
	 partname);
  numparts = diskpart_blkio_get_partition(disk, part_array, MAX_PARTS);
  if (numparts == 0)
    panic("No partitions found");
  if (diskpart_blkio_lookup_bsd_string(part_array, partname,
				       disk, &part) == 0)
    panic("Couldn't find partition %s", partname);

  /* Don't need the disk anymore, the partition has a ref. */
  oskit_blkio_release(disk);

  printf(">>>Mounting partition %s\n", partname);
  CHECK(err, fs_linux_mount, (part, OSKIT_FS_RDONLY, &rootfs));

  /* Don't need the part anymore, the filesystem has a ref. */
  oskit_blkio_release(part);

  printf(">>>Getting a handle to the root dir\n");
  CHECK(err, oskit_filesystem_getroot, (rootfs, &dir));

  printf(">>>Creating a file name space\n");
  CHECK(err, oskit_create_fsnamespace, (dir, dir, &fsns));
  CHECK(err, oskit_clientos_setfsnamespace, (fsns));

  oskit_dir_release(dir);
  oskit_filesystem_release(rootfs);
  return;
}

extern int scheme_main(int argc, char **argv);

int main(int argc, char **argv)
{
  oskit_osenv_t* osenv;
  char rootdev[16] = {0};
  int i;

  oskit_clientos_init();
  for (i = 1; i < argc; i++) {
    if (strncmp(argv[i], "root=", 5) == 0) {
      strncpy(rootdev, &argv[i][5], 16);
    }
  }

  start_clock();  /* so gettimeofday works */
#if 1
  osenv = start_osenv();
#else
  osenv = oskit_osenv_create_default();
  oskit_register(&oskit_osenv_iid, (void*)osenv);
#endif

  printf(">>>Initializing device drivers...\n");
  oskit_dev_init(osenv);
  oskit_linux_init_osenv(osenv);
  oskit_linux_init_ide();

  printf(">>>Probing devices...\n");
  oskit_dev_probe();

  mount_disk(rootdev);

  { char *arg[2];
  arg[0] = strdup("scheme");
  arg[1] = NULL;
  scheme_main(1, arg);
  free(arg[0]);
  }

  return 0;
}
