<?php
/*
Plugin Name: Head Cleaner
Version: 1.3.2
Plugin URI: http://wppluginsj.sourceforge.jp/head-cleaner/
Description: Cleaning tags from your WordPress header and footer.
Author: wokamoto
Author URI: http://dogmap.jp/
Text Domain: head-cleaner
Domain Path: /languages/

License:
 Released under the GPL license
  http://www.gnu.org/copyleft/gpl.html

  Copyright 2009 - 2010 wokamoto (email : wokamoto1973@gmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

Includes:
 PHP Simple HTML DOM Parser Ver.1.11
  http://sourceforge.net/projects/simplehtmldom/
  Licensed under The MIT License

 jsmin.php - PHP implementation of Douglas Crockford's JSMin. Ver.1.1.1 (2008-03-02)
  Copyright (c) 2002 Douglas Crockford <douglas@crockford.com> (jsmin.c)
  Copyright (c) 2008 Ryan Grove <ryan@wonko.com> (PHP port)
  License http://opensource.org/licenses/mit-license.php MIT License

 CSSTidy - CSS Parser and Optimiser Ver.1.3
  @author Florian Schmitz (floele at gmail dot com) 2005-2006
  @license http://opensource.org/licenses/gpl-license.php GNU Public License
*/

if (version_compare(phpversion(), "5.0.0", "<"))
	return false;

//**************************************************************************************
// Defines
//**************************************************************************************
define('HC_PRIORITY', 10000);
define('HC_ANALYZE_EXPIRED', 604800);	// 60 * 60 * 24 * 7 [sec.]
define('HC_XMLNS', 'http://www.w3.org/1999/xhtml');
define('HC_CACHE_DIR', 'cache/head-cleaner');


//**************************************************************************************
if (!defined('ABSPATH') && strstr($_SERVER['PHP_SELF'], '/head-cleaner.php')) {
	define('HC_EXPIRED_JS_CSS', 2592000);	// 60 * 60 * 24 * 30 [sec.]

	if (isset($_GET['f']) && isset($_GET['t'])) {
		$cache_dir = realpath(dirname(__FILE__) . '/../../' . HC_CACHE_DIR) . '/';
		$filename_hash = trim(stripslashes($_GET['f']));
		$type = trim(stripslashes($_GET['t']));

		if (strlen($filename_hash) == 32 && ($type == 'js' || $type == 'css')) {
			$is_gzip  = (strpos(strtolower($_SERVER['HTTP_ACCEPT_ENCODING']), 'gzip') !== FALSE);
			$filename = "{$cache_dir}{$type}/{$filename_hash}.{$type}" . ($is_gzip ? '.gz' : '');
			if (file_exists($filename)) {
				$offset = (!defined('HC_EXPIRED_JS_CSS') ? HC_EXPIRED_JS_CSS : 60 * 60 * 24 * 30);
				$content_type = 'text/' . ($type == 'js' ? 'javascript' : $type);

				header('Last-Modified: '.gmdate('D, d M Y H:i:s', filemtime($filename)).' GMT');
				header('Expires: '.gmdate('D, d M Y H:i:s', time() + $offset).' GMT');
				header("Content-type: $content_type");

				if ($is_gzip)
					header('Content-Encoding: gzip');

				readfile($filename);
				exit();
			} else {
				header('HTTP/1.1 404 Not Found');
				echo "404 : Not Found";
				die();
			}
		} else {
			header("HTTP/1.0 403 Forbidden");
			echo "403 : Forbidden";
			die();
		}
	} else {
		header("HTTP/1.0 403 Forbidden");
		echo "403 : Forbidden";
		die();
	}
}

//**************************************************************************************
// Require
//**************************************************************************************
if (!class_exists('wokController') || !class_exists('wokScriptManager'))
	require(dirname(__FILE__).'/includes/common-controller.php');

//**************************************************************************************
// Head Cleaner
//**************************************************************************************
class HeadCleanerController extends wokController {
	public $plugin_name  = 'head-cleaner';
	public $plugin_ver   = '1.3.2';

	// Deafault Options
	private $options_default = array(
		'foot_js'        => false ,
		'dynamic'        => false ,
		'js_move_foot'   => false ,
		'cache_enabled'  => false ,
		'combined_css'   => false ,
		'combined_js'    => false ,
		'js_minify'      => false ,
		'css_optimise'   => false ,
		'csstidy_option' => array(
			'optimise_shorthands'  => 1 ,
			'compress_colors'      => true ,
			'compress_font_weight' => true ,
			'remove_bslash'        => true ,
			'template'             => 1 ,
			) ,
		'default_media'  => 'all' ,
		'debug_mode'     => false ,
		'filters'        => array('wp_head' => array(), 'wp_footer' => array()) ,
		'priority'       => array('wp_head' => array(), 'wp_footer' => array()) ,
		'head_js'        => array() ,
		'analyze_expired'=> 0 ,
		'xml_declaration'=> false ,
		'ie_conditional' => false ,
		'canonical_tag'  => true ,
		'gzip_on'        => false ,
		'use_ajax_libs'  => false ,
		);
	private $csstidy_template = array(
		'low_compression' ,
		'default' ,
		'high_compression' ,
		'highest_compression' ,
		);

	private $wp_url     = '';
	private $self_url   = '';
	private $cache_path = '';
	private $cache_url  = '';

	private $lang;
	private $foot_js_src;

	private $mtime_start;

	private $filters;
	private $head_js;
	private $no_conflict = array(
		'comment_quicktags' ,       // Comment Quicktags
		'stats_footer' ,            // WordPress.com Stats
		'uga_wp_footer_track' ,     // Ultimate Google Analytics
		'tam_google_analytics::insert_tracking_code' ,  // TaM Google Analytics
		);

	private $default_head_filters = array(
		'HeadCleanerController::.*' ,
		'noindex' ,
		'lambda_[\d]+' ,
		'rsd_link' ,
		'wlwmanifest_link' ,
		'wp_generator' ,
		);

	/**********************************************************
	* Constructor
	***********************************************************/
	public function HeadCleanerController() {
		$this->__construct();
	}
	public function __construct() {
		$this->init(__FILE__);
		$this->options = $this->_init_options($this->getOptions());
		$this->filters = $this->options['filters'];
		$this->head_js = $this->options['head_js'];

		$this->wp_url   = trailingslashit(get_bloginfo('wpurl'));
		$this->self_url = str_replace(ABSPATH, $this->wp_url, __FILE__);
		$this->lang     = (defined('WPLANG') ? WPLANG : 'ja');
		$this->charset  = get_option('blog_charset');
		$this->_get_filters('wp_head');
		$this->_get_filters('wp_footer');

		// Create Directory for Cache
		if ( $this->options['cache_enabled'] ) {
			$this->cache_path = $this->_create_cache_dir();
			if ($this->cache_path !== FALSE) {
				$this->cache_url = str_replace(ABSPATH, $this->wp_url, $this->cache_path);
			} else {
				$this->options['cache_enabled'] = false;

				$this->options['combined_css']  = false;
				$this->options['combined_js']   = false;

				$this->options['js_minify']     = false;
				$this->options['css_optimise']  = false;
			}
		}

		// Require PHP Libraries
		if (!is_admin())
			$this->_require_libraries();
	}

	/**********************************************************
	* Init Options
	***********************************************************/
	private function _init_options($wk_options = '') {
		if (!is_array($wk_options))
			$wk_options = array();

		foreach ($this->options_default as $key => $val) {
			$wk_options[$key] = (isset($wk_options[$key]) ? $wk_options[$key] : $val);
		}

		if (time() > $wk_options['analyze_expired']) {
			$filters  = $this->options_default['filters'];

			$priority = $this->options_default['priority'];
			foreach ($wk_options['priority'] as $tag => $filters) {
				foreach ((array) $filters as $key => $val) {
					if ($val <= 0 || $val > HC_PRIORITY)
						$priority[$tag][$key] = $val;
				}
			}
			unset($filters);

			$head_js  = $this->options_default['head_js'];
			foreach ((array) $wk_options['head_js'] as $key => $val) {
				if ($val === FALSE)
					$head_js[$key] = $val;
			}

			$wk_options['filters']  = $filters;
			$wk_options['priority'] = $priority;
			$wk_options['head_js']  = $head_js;
			$wk_options['analyze_expired'] = time() + HC_ANALYZE_EXPIRED;
		}
		return $wk_options;
	}

	/**********************************************************
	* Require Libraries
	***********************************************************/
	private function _require_libraries(){
		$includes = dirname(__FILE__) . '/includes/';

		// PHP Simple HTML DOM Parser
		if (!function_exists('str_get_html'))
			require($includes . 'simple_html_dom.php' );

		// jsmin.php - PHP implementation of Douglas Crockford's JSMin.
		if ($this->options['js_minify']) {
			require($includes . 'jsmin-1.1.1.php');
			$this->options['js_minify'] = class_exists('JSMin') && $this->options['js_minify'];
		}

		// CSSTidy - CSS Parser and Optimiser
		if ($this->options['css_optimise']) {
			require($includes . 'csstidy-1.3/class.csstidy.php');
			$this->options['css_optimise'] = class_exists('csstidy') && $this->options['css_optimise'];
		}

		// Use Google Ajax Libraries
		if ($this->options['use_ajax_libs'])
			require($includes . 'regist_ajax_libs.php');
	}

	//**************************************************************************************
	// plugin activation
	//**************************************************************************************
	public function activation(){
		$cache_dir = $this->_create_cache_dir();
		if ( $cache_dir !== FALSE )
			$this->_create_htaccess($cache_dir);
	}

	//**************************************************************************************
	// plugin deactivation
	//**************************************************************************************
	public function deactivation(){
//		$this->_remove_cache_file();
	}

	//**************************************************************************************
	// ob_start for Header
	//**************************************************************************************
	public function head_start(){
		if (! $this->_is_ktai() ) {
			ob_start(array(&$this, 'head_cleaner'));
			$this->mtime_start = microtime();
			add_action('wp_head', array(&$this, 'end'), HC_PRIORITY);
			$this->_get_filters('wp_head');
			$this->_change_filters_priority('wp_head');
			if ($this->options['foot_js'])
				add_action('wp_footer', array(&$this, 'footer_start'), 1);
		}
	}

	//**************************************************************************************
	// ob_start for footer
	//**************************************************************************************
	public function footer_start(){
		if (! $this->_is_ktai() && $this->options['foot_js'] ) {
			ob_start(array(&$this, 'footer_cleaner'));
			$this->mtime_start = microtime();
			add_action('wp_footer', array(&$this, 'end'), HC_PRIORITY);
			$this->_get_filters('wp_footer');
			$this->_change_filters_priority('wp_footer');
		}
	}

	//**************************************************************************************
	// ob_end_flush
	//**************************************************************************************
	public function end(){
		ob_end_flush();
	}

	//**************************************************************************************
	// filters info save
	//**************************************************************************************
	public function filters_save(){
		if ($this->_chk_filters_update()) {
			$this->options['filters'] = $this->filters;
			$this->options['head_js'] = $this->head_js;
			if (time() > $this->options['analyze_expired'])
				$this->options['analyze_expired'] = time() + HC_ANALYZE_EXPIRED;
			$this->updateOptions();
		}
	}

	//**************************************************************************************
	// head cleaner
	//**************************************************************************************
	public function head_cleaner($buffer) {
		if (!function_exists('str_get_html'))
			return trim($buffer) . "\n";

		$url = $this->_get_permalink();

		$xml_head  = "<?xml version=\"1.0\" encoding=\"{$this->charset}\"?>\n";
		$html_tag  = "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"{$this->lang}\" lang=\"{$this->lang}\">\n";
		$head_tag  = "<head profile=\"http://gmpg.org/xfn/11\">\n";
		$head_txt  = trim($buffer);

		$ret_val   = '';
		$meta_tag  = ''; $title_tag  = ''; $base_tag  = '';
		$link_tag  = ''; $object_tag = ''; $other_tag = '';
		$css_tag   = ''; $inline_css = '';
		$script_tag = ''; $inline_js = '';

		// for IE conditional tag
		$IE_conditional_tag_pattern = '/<\!-+[ \t]*\[if[ \t]+%sIE%s[ \t]*?\][ \t]*>(.*?)<\![ \t]*\[endif\][ \t]*-+>/ism';
		$IE_conditional_tags = array();
		if ($this->options['ie_conditional']) {
			$ua  = trim(strtolower($_SERVER['HTTP_USER_AGENT']));
			$ie  = (strpos($ua, 'msie') !== false)
				&& (! preg_match('/(gecko|applewebkit|opera|sleipnir|chrome)/i', $ua));
			$ie_ver = ($ie ? preg_replace('/^.*msie +([\d\.]+);.*$/i', '$1', $ua) : false);
			$ie6 = ($ie ? version_compare($ie_ver, '7.0', '<') : false);
			if ($ie) {
				$head_txt = preg_replace(sprintf($IE_conditional_tag_pattern, '', ''), '$1', $head_txt);

				$replace_patterns = array();
				if (version_compare($ie_ver, '5.5', '<')) {						// IE 5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*|[ \t]*)', ' 5\.?0?');	// >= 5, <= 5, = 5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(lte?[ \t]*)', ' 5\.5');		// <  5.5, <= 5.5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(lte?[ \t]*)', ' [678]\.?0?');		// <  6 - 8, <= 6 - 8
				} elseif (version_compare($ie_ver, '6.0', '<')) {					// IE 5.5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*|[ \t]*)', ' 5\.5');	// >= 5.5, <= 5.5, = 5.5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(lte?[ \t]*)', ' [678]\.?0?');		// <  6 - 8, <= 6 - 8
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' 5\.?0?');		// >  5, >= 5
				} elseif (version_compare($ie_ver, '7.0', '<')) {					// IE 6
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*|[ \t]*)', ' 6\.?0?');	// >= 6, <= 6, = 6
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(lte?[ \t]*)', ' [78]\.?0?');		// <  7 - 8, <= 7 - 8
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' 5\.?0?');		// >  5, >= 5
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' 5\.5');		// >  5.5, >= 5.5
				} elseif (version_compare($ie_ver, '8.0', '<')) {					// IE 7
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*|[ \t]*)', ' 7\.?0?');	// >= 7, <= 7, = 7
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(lte?[ \t]*)', ' 8\.?0?');		// <  8, <= 8
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' [56]\.?0?');		// >  5 - 6, >= 5 - 6
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' 5\.5');		// >  5.5, >= 5.5
				} elseif (version_compare($ie_ver, '9.0', '<')) {					// IE 8
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*|[ \t]*)', ' 8\.?0?');	// >= 8, <= 8, = 8
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' [567]\.?0?');		// >  5 - 7, >= 5 - 7
					$replace_patterns[] = sprintf($IE_conditional_tag_pattern, '(gte?[ \t]*)', ' 5\.5');		// >  5.5, >= 5.5
				}

				if (count($replace_patterns) > 0)
					$head_txt = preg_replace($replace_patterns, '$2', $head_txt);
				unset($replace_patterns);
			}
		} else {
			$search_pattern = sprintf($IE_conditional_tag_pattern, '([lg]te[ \t]*)?', '([ \t]+[05678\.]*)?');
			preg_match_all($search_pattern, $head_txt, $IE_conditional_tags, PREG_PATTERN_ORDER);
			$head_txt = preg_replace($search_pattern, '', $head_txt);
		}

		if (preg_match('/^(.*)(<html[^>]*>[^<]*)(<head[^>]*>[^<]*)(.*)$/ism', $head_txt, $matches)) {
			$ret_val  = trim($matches[1]) . "\n";
			$html_tag = trim($matches[2]) . "\n";
			$head_tag = trim($matches[3]) . "\n";
			$head_txt = trim($matches[4]) . "\n";
		}
		unset($matches);
		$html_txt = $xml_head . $html_tag . $head_tag
			  . $head_txt
			  . '</head><body></body></html>';

		// Get Simple DOM Object
		$dom = str_get_html($html_txt);
		if ($dom === false)
			return (trim($buffer)."\n");

		$xmlns    = (defined('HC_XMLNS') ? HC_XMLNS : 'http://www.w3.org/1999/xhtml');
		$xml_lang = $this->lang;
		$lang     = $this->lang;
		if (preg_match_all('/ +([^ ]*)=[\'"]([^\'"]*)[\'"]/', $html_tag, $mathes, PREG_SET_ORDER)) {
			foreach ((array) $matches as $match) {
				switch ($match[1]){
				case 'xmlns':    $xmlns = $match[2]; break;
				case 'xml:lang': $xml_lang = $match[2]; break;
				case 'lang':     $lang = $match[2]; break;
				}
			}
			unset($match);
		}
		unset($matches);
		$html_tag  = "<html xmlns=\"{$xmlns}\" xml:lang=\"{$xml_lang}\" lang=\"{$lang}\">\n";

		$meta_tag  = $this->_dom_to_html($dom->find("meta"));
		$title_tag = $this->_dom_to_html($dom->find("title"), 1);
		$base_tag  = $this->_dom_to_html($dom->find("base"), 1);
		$link_tag  = $this->_dom_to_html($dom->find("link[rel!='stylesheet']"));
		if (count($dom->find("link[rel='canonical']")) <= 0 && $this->options['canonical_tag'])
			$link_tag .= '<link rel="canonical" href="' . htmlspecialchars($url, ENT_QUOTES) . '" />'."\n";

		$css_tag = ''; $inner_css  = ''; $css_src = array();
		$elements = $dom->find("link[rel='stylesheet']");
		foreach ((array) $elements as $element) {
			$tag = trim($element->outertext);
			if (strpos($css_tag, $tag) === FALSE) {
				if (strpos($element->href, $this->wp_url) === FALSE) {
					$css_tag .= $tag . "\n";
				} else {
					$media = trim( isset($element->media) ? $element->media : $this->options['default_media'] );
					$media = ( empty($media) || is_null($media) || $media === FALSE ? $this->options['default_media'] : $media );
					$inner_css .= trim($tag) . "\n";
					$css_src[$media][] = $element->href;
				}
			}
		}
		unset($element); unset($elements);

		$elements = $dom->find("style");
		$wk_inline_css = array();
		foreach ((array) $elements as $element) {
			$media = trim( isset($element->media) ? $element->media : $this->options['default_media'] );
			$media = ( empty($media) || is_null($media) || $media === FALSE ? $this->options['default_media'] : $media );
			$wk_text = $this->_remove_comment($element->innertext, 'css');
			if ( preg_match_all('/@import[\s]+url[\s]*\([\s\'"]*([^\)\'"]*?)[\s\'"]*\);/i', $wk_text, $wk_matches, PREG_SET_ORDER) ) {
				foreach ($wk_matches as $val) {
					$wk_text = trim(str_replace( $val[0], '', $wk_text));
					$href  = trim($val[1]);
					$tag = "<link rel=\"stylesheet\" href=\"{$href}\" type=\"text/css\" media=\"{$media}\" />";
					if (strpos( $href, $this->wp_url) === FALSE) {
						$css_tag .= $tag . "\n";
					} else {
						$inner_css .= $tag . "\n";
						$css_src[$media][] = $href;
					}
				}
			}
			unset($wk_matches);
			if ( !empty($wk_text) )
				$wk_inline_css[$media] .= trim($wk_text) . "\n";
		}
		$inline_css = '';
		if ($this->options['cache_enabled']) {
			if ($this->options['combined_css']) {
				$css_tag    = trim($css_tag) . "\n";
				foreach ($css_src as $key => $val) {
					$inner_css  = $this->_combined_css($val, trim(isset($wk_inline_css[$key]) ? $wk_inline_css[$key] : '' ), $key);
					$css_tag   .= trim($inner_css) . "\n";
					if (isset($wk_inline_css[$key])) $wk_inline_css[$key] = '';
				}
				foreach ($wk_inline_css as $key => $val) {
					$val = trim($val);
					if (!empty($val)) {
						$inner_css  = $this->_combined_css(array(), $val, $key);
						$css_tag   .= trim($inner_css) . "\n";
					}
				}
			} else {
				$css_tag    = trim(trim($css_tag) . "\n" . trim($inner_css)) . "\n";
				foreach ($wk_inline_css as $key => $val) {
					$val = trim($val);
					if (!empty($val)) {
						$inner_css   = $this->_combined_inline_css(trim($val), $key);
						$inline_css .= trim($inner_css) . "\n";
					}
				}
			}
		} else {
			$css_tag    = trim(trim($css_tag) . "\n" . trim($inner_css)) . "\n";
			foreach ($wk_inline_css as $key => $val) {
				$val = trim($val);
				if (!empty($val)) {
					$inline_css .= "<style type=\"text/css\" media=\"{$media}\">/*<![CDATA[ */\n"
						. $val
						. "\n/* ]]>*/</style>\n";
				}
			}
		}
		$css_tag    = trim($css_tag)    . (!empty($css_tag)    ? "\n" : '');
		$inner_css  = trim($inner_css)  . (!empty($inner_css)  ? "\n" : '');
		$inline_css = trim($inline_css) . (!empty($inline_css) ? "\n" : '');
		unset($wk_inline_css);
		unset($element); unset($elements);

		$inner_js = '';
		$foot_js  = '';
		$js_src   = array();
		$js_libs  = array();
		$elements = $dom->find("script");
		foreach ((array) $elements as $element) {
			if (!isset($element->src)) {
				$inline_js .= $this->_remove_comment($element->innertext, 'js');
			} else {
				$src = trim($element->src);
				if (!isset($this->head_js[$src]))
					$this->head_js[$src] = true;

				$find = FALSE;
				if (preg_match('/\/((prototype|jquery|mootools)\.js)\?ver=([\.\d]+)[^\?]*$/i', $src, $matches)) {
					list($find, $filename, $product, $version) = $matches;
				} elseif (preg_match('/\/((prototype|jquery|mootools)[\-\.](min|[\.\d]+)[^\/]*\.js)\?[^\?]*$/i', $src, $matches)) {
					list($find, $filename, $product, $version) = $matches;
				} elseif (preg_match('/\/scriptaculous\/((builder|controls|dragdrop|effects|wp\-scriptaculous|scriptaculous|slider|sound|unittest)\.js)\?ver\=([\.\d]+)[^\?]*$/i', $src, $matches)) {
					list($find, $filename, $product, $version) = $matches;
					$product = (strpos($product, 'scriptaculous') === FALSE
						? 'scriptaculous/' . $product
						: 'scriptaculous/scriptaculous');
				}
				unset($matches);

				if ($find !== FALSE) {
					$version = trim(substr($version, -1) === '.' ? substr($version, 0, -1) : $version);
					if (empty($version))
						$version = preg_replace('/^.*\/([\.\d]+)\/.*$/', '$1', $src);
					if (!preg_match('/^[\.\d]*\d$/', $version))
						$version = '1';
					$js_libs[$product][$version] = $src;
				} elseif ($this->head_js[$src] === FALSE) {
					$foot_js .= trim($element->outertext) . "\n";
				} elseif (strpos($src, $this->wp_url) === FALSE) {
					$script_tag .= trim($element->outertext) . "\n";
				} else {
					$inner_js .= trim($element->outertext) . "\n";
					$js_src[] = $element->src;
				}
			}
		}
		unset($element); unset($elements);

		// JavaScript FrameWork (Prototype.js > jQuery > mootools)
		if (count($js_libs) > 0) {
			list($js_libs_src, $wk_inner_js, $wk_outer_js) = $this->_js_framework($js_libs);

			if (count($js_libs_src) > 0)
				$js_src = array_merge($js_libs_src, $js_src);

			$script_tag = trim($script_tag) . "\n" . $wk_outer_js;
			$inner_js   = $wk_inner_js . $inner_js;

			unset($js_libs_src); unset($js_libs);
		}

		$inline_js = trim($inline_js);
		if ($this->options['cache_enabled']) {
			if ($this->options['combined_js']) {
				$inner_js  = $this->_combined_javascript($js_src, trim($inline_js));
				$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
				$inline_js = '';
			} else {
				$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
				$inline_js = $this->_combined_inline_javascript(trim($inline_js));
			}
		} else {
			$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
			$inline_js = (!empty($inline_js) ? "<script type=\"text/javascript\">//<![CDATA[\n{$inline_js}\n//]]></script>\n" : '');
		}
		$script_tag = trim($script_tag) . (!empty($script_tag) ? "\n" : '');
		$inner_js   = trim($inner_js)   . (!empty($inner_js)   ? "\n" : '');
		$inline_js  = trim($inline_js)  . (!empty($inline_js)  ? "\n" : '');
		$foot_js    = trim($foot_js)    . (!empty($foot_js)    ? "\n" : '');

		$object_tag = $this->_dom_to_html($dom->find("object"));

		$dom->clear(); unset($dom);

		// for IE conditional tag
		if (!$this->options['ie_conditional'] && count($IE_conditional_tags) > 0) {
			$IE_conditional_tag_pattern = '/(<\!-+[ \t]*\[if[ \t]+.*IE.*[ \t]*?\][ \t]*>)[ \t\r\n]*(.*?)[ \t\r\n]*(<\![ \t]*\[endif\][ \t]*-+>)/ism';
			foreach ((array) $IE_conditional_tags as $IE_conditional_tag) {
				if (preg_match($IE_conditional_tag_pattern, $IE_conditional_tag[0])) {
					$IE_tag = trim(preg_replace($IE_conditional_tag_pattern, "$1\n$2\n$3", $IE_conditional_tag[0])) . "\n";
					if ( strpos(strtolower($IE_tag), '<link') !== false )
						$css_tag    = trim($css_tag) . "\n" . $IE_tag;
					elseif ( strpos(strtolower($IE_tag), '<style') !== false )
						$inline_css = trim($inline_css) . "\n" . $IE_tag;
					elseif ( strpos(strtolower($IE_tag), '<script') !== false )
						$inline_js  = trim($inline_js) . "\n" . $IE_tag;
					else
						$object_tag = trim($object_tag) . "\n" . $IE_tag;
				}
			}
		}
		unset($IE_conditional_tag);

		$ret_val .=
			  $html_tag
			. $head_tag
			. $meta_tag
			. $title_tag
			. $base_tag
			. $link_tag
			. $css_tag
			. $inline_css
			. (!$this->options['js_move_foot'] ? $script_tag : '')
			. (!$this->options['js_move_foot'] ? $inline_js  : '')
			. $object_tag
			;
		//$ret_val = str_replace('\'', '"', $ret_val);
		$ret_val = preg_replace(
			array( "/[\s]+([^\=]+)\='([^']*)'/i", '/[\n\r]+/i', '/(<\/[^>]+>)[ \t]*(<[^>]+)/i' ) ,
			array( ' $1="$2"', "\n", "$1\n$2" ),
			$ret_val
			);

		if ($this->options['js_move_foot'] || !empty($foot_js))
			$this->foot_js_src = trim(
				  $foot_js
				. ($this->options['js_move_foot'] ? $script_tag : '')
				. ($this->options['js_move_foot'] ? $inline_js  : '')
				);
		if (!empty($this->foot_js_src)) {
			$this->foot_js_src .= "\n";
			add_action('wp_footer', array(&$this, 'footer'), 9);
		}

		$ret_val = ($ie6 || !$this->options['xml_declaration']
			? preg_replace('/^<\?xml[^>]*>/i', '', $ret_val)
			: (strpos($ret_val, '<?xml') === false ? $xml_head : '') . $ret_val
			);
		$ret_val = trim($ret_val) . "\n";

		if ($this->options['debug_mode'])
			$ret_val .= $this->_get_debug_info($buffer);

		return $ret_val;
	}

	//**************************************************************************************
	// JavaScript Moved bottom
	//**************************************************************************************
	public function footer(){
		echo $this->foot_js_src;
	}

	//**************************************************************************************
	// footer cleaner
	//**************************************************************************************
	public function footer_cleaner($buffer) {
		if (!function_exists('str_get_html'))
			return trim($buffer) . "\n";

		$ret_val    = '';
		$script_tag = ''; $inline_js = '';
		$other_tag  = '';

		$html_txt = '<html><head></head><body>'
			  . '<div id="footer">' . trim($buffer) . '</div>'
			  . '</body></html>';

		// Get Simple DOM Object
		$dom = str_get_html($html_txt);
		if ($dom === false)
			return (trim($buffer)."\n");

		$inner_js  = ''; $js_src = array();
		$elements = $dom->find("div#footer *");
		foreach ((array) $elements as $element) {
			switch ($element->tag) {
			case 'script':
				if (!isset($element->src)) {
					$inline_js .= $this->_remove_comment($element->innertext, 'js');
				} else {
					if (strpos($element->src, $this->wp_url) === FALSE) {
						$script_tag .= trim($element->outertext) . "\n";
					} else {
						$inner_js .= trim($element->outertext) . "\n";
						$js_src[] = preg_replace('/\.gz$/i', '', $element->src);
					}
				}
				break;
			default:
				$tag = trim($element->outertext);
				if (strpos($other_tag, $tag) === FALSE && ! preg_match('/^<\!\-+/', $tag))
					$other_tag .= $tag . "\n";
				break;
			}
		}
		unset($element); unset($elements);

		$inline_js = trim($inline_js);
		if ($this->options['cache_enabled']) {
			if ($this->options['combined_js']) {
				$inner_js  = $this->_combined_javascript($js_src, trim($inline_js));
				$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
				$inline_js = '';
			} else {
				$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
				$inline_js = $this->_combined_inline_javascript(trim($inline_js));
			}
		} else {
			$script_tag = trim(trim($script_tag) . "\n" . $inner_js) . "\n";
			$inline_js = (!empty($inline_js) ? "<script type=\"text/javascript\">//<![CDATA[\n{$inline_js}\n//]]></script>\n" : '');
		}

		$dom->clear(); unset($dom);

		$ret_val .=
			  $other_tag
			. $script_tag
			. $inline_js
			;
		$ret_val = str_replace('\'', '"', $ret_val);
		$ret_val = trim($ret_val) . "\n";

		if ($this->options['debug_mode'])
			$ret_val .= $this->_get_debug_info($buffer);

		return $ret_val;
	}

	//**************************************************************************************
	// WP_CONTENT_DIR
	//**************************************************************************************
	private function _wp_content_dir($path = '') {
		return (!defined('WP_CONTENT_DIR')
			? WP_CONTENT_DIR
			: ABSPATH . 'wp-content'
			) . $path;
	}

	//**************************************************************************************
	// is login?
	//**************************************************************************************
	private function _is_user_logged_in() {
		if (function_exists('is_user_logged_in')) {
			return is_user_logged_in();
		} else {
			global $user;
			if (!isset($user)) $user = wp_get_current_user();
			return (!empty($user->ID));
		}
	}

	//**************************************************************************************
	// Is mobile ?
	//**************************************************************************************
	private function _is_ktai() {
		return ($this->isKtai());
	}

	//**************************************************************************************
	// Get permalink
	//**************************************************************************************
	private function _get_permalink() {
		$url = get_bloginfo('url');
		if (! preg_match('|^(https?://[^/]*)|', $url, $host))
			$host[1] = (empty($_SERVER['HTTPS']) || $_SERVER['HTTPS'] == 'off') ? 'http://' : 'https://' . $_SERVER['SERVER_NAME'];
		$url = preg_replace( '/\?[^s].*$/i', '', $host[1] . $_SERVER['REQUEST_URI']);
		unset($host);
		return ($url);
	}

	//**************************************************************************************
	// DOM Element -> html tag
	//**************************************************************************************
	private function _dom_to_html($elements, $limit = false) {
		$html = ''; $count = 0;
		$type = '';
		$tags = array();
		foreach ((array) $elements as $element) {
			$tag  = trim($element->outertext);
			$type = strtolower($element->tag);
			if (strpos($html, $tag) === FALSE) {
				switch($type) {
				case 'meta':
					if (isset($element->name)) {
						$name    = strtolower($element->name);

						$content = trim(isset($tags[$type][$name]) && isset($tags[$type][$name]['content'])
							 ? $tags[$type][$name]['content']
							 : '');
						$content = trim($content
							. (!empty($content) ? ', ' : '')
							. (isset($element->content) ? $element->content : '')
							);

						$tags[$type][$name] = array(
							 'name'    => $name
							,'content' => $content
							);
					} else {
						$html .= $tag . "\n";
					}
					break;
				default:
					$html .= $tag . "\n";
					break;
				}
			}
			if ($limit !== false && $count++ >= $limit) break;
		}
		unset($element); unset($elements);

		foreach ((array) $tags as $tag_types) {
			foreach ((array) $tag_types as $tag) {
				$html .= "<$type";
				foreach((array) $tag as $key => $val) {
					$html .= " $key=\"$val\"";
				}
				$html .= " />\n";
			}
			unset($tag);
		}
		unset($tags); unset($tag_types);

		$html = trim($html);

		return $html . (!empty($html) ? "\n" : '');
	}

	//**************************************************************************************
	// Get absolute url
	//**************************************************************************************
	private function _abs_url($path, $base_path = ''){
		if (preg_match('/^https?:\/\//i', $base_path))
			$base_path = str_replace($this->wp_url, ABSPATH, $base_path);
		$absolute_path = realpath($base_path . '/' . $path);
	        if(!$absolute_path)
	                $absolute_path = $base_path . '/' . $path;

		$url = str_replace(ABSPATH, $this->wp_url, $absolute_path);
		return $url;
	}

	private function _css_url_edit($content, $filename) {
		if (preg_match_all('/url[ \t]*\([\'"]?([^\)]*)[\'"]?\)/i', $content, $matches, PREG_SET_ORDER)) {
			$base_path = dirname($filename);
			$search = array(); $replace = array();
			foreach ((array) $matches as $match) {
				if (! preg_match('/^https?:\/\//i', trim($match[1]))) {
					$abs_url   = $this->_abs_url(trim($match[1]), $base_path);
					$search[]  = $match[0];
					$replace[] = str_replace(trim($match[1]), $abs_url, $match[0]);
				}
			}
			$content = str_replace($search, $replace, $content);
			unset($match); unset($search); unset($replace);
		}
		unset ($matches);
		return $content;
	}

	//**************************************************************************************
	// Read file
	//**************************************************************************************
	private function _file_read($filename) {
		if (preg_match('/^https?:\/\//i', $filename)) {
			$content = file_get_contents($filename);
		} else {
			if (file_exists($filename)) {
				$handle = @fopen($filename, 'r');
				$content = trim(@fread($handle, filesize($filename)));
				@fclose($handle);
			} else {
				$content = false;
			}
		}

		return $content;
	}

	//**************************************************************************************
	// Read files
	//**************************************************************************************
	private function _files_read($files, $type = 'js') {
		$text = '';
		foreach ((array) $files as $filename) {
			$content = trim($this->_file_read($filename));
			switch ($type) {
			case 'css':
				$content = $this->_css_url_edit($content, $filename);
				break;
			case 'js':
				$content = 'try{'
					 . trim($content . (substr($content, -1) !== ';' ? ';' : '')) . "\n"
					 . ($this->options['debug_mode']
						? '}catch(e){alert("error(' . basename($filename) .  '): " + e.toString());}'
						: '}finally{};');
				break;
			}
			$text .= $content . "\n\n";
		}
		unset($filename);
		$text = trim($text);

		return $text . (!empty($text) ? "\n" : '');
	}

	//**************************************************************************************
	// Write file
	//**************************************************************************************
	private function _file_write($filename, $content = '', $gzip = true) {
		if (!empty($content)) {
			$handle = @fopen($filename, 'w');
			@fwrite($handle, $content);
			@fclose($handle);

			return ($gzip
				? $this->_file_gzip($filename, $content)
				: file_exists($filename)
				);
		} else {
			return false;
		}
	}

	//**************************************************************************************
	// Write gzip file
	//**************************************************************************************
	private function _file_gzip($filename, $content = '') {
		if (file_exists($filename) && file_exists($filename . '.gz'))
			if (filemtime($filename) < filemtime($filename . '.gz'))
				return true;

		if (function_exists('gzopen')) {
			if (empty($content)) {
				$handle  = @fopen($filename, 'r');
				$content = @fread($handle, filesize($filename));
				@fclose($handle);
			}

			if (!empty($content)) {
				$handle = @gzopen($filename . '.gz', 'w9');
				@gzwrite($handle, $content);
				@gzclose($handle);
			}

			return (file_exists($filename . '.gz'));
		} else {
			return false;
		}
	}

	//**************************************************************************************
	// JavaScript FrameWork (Prototype.js > scriptaculous > jQuery > jQuery.noConflict > mootools)
	//**************************************************************************************
	private function _js_framework($js_libs) {
		$prototype = isset($js_libs['prototype']);
		$jquery    = isset($js_libs['jquery']);
		$mootools  = isset($js_libs['mootools']);
		$js_libs_src = array();
		$wk_inner_js = '';
		$wk_outer_js = '';

		// Prototype.js 1.6.0.3
		if ($prototype) {
			list($src, $ver) = $this->_newer_version_src($js_libs['prototype']);
			if (!empty($src)) {
				if (version_compare($ver, '1.6.0.3', '<='))
					$src = $this->plugin_url . 'includes/js/prototype-1.6.0.3.min.js';

				$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
			}
		}

		// scriptaculous 1.8.2
		if (isset($js_libs['scriptaculous/scriptaculous'])) {
			if (!$prototype) {
				$prototype = true;
				$src = $this->plugin_url . 'includes/js/prototype-1.6.0.3.min.js';
				$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
			}
			$scriptaculous = array(
				  'scriptaculous/scriptaculous'
				, 'scriptaculous/controls'
				, 'scriptaculous/dragdrop'
				, 'scriptaculous/effects'
				, 'scriptaculous/slider'
				, 'scriptaculous/sound'
				, 'scriptaculous/unittest'
				);
			foreach ($scriptaculous as $product) {
				if (isset($js_libs[$product])) {
					list($src, $ver) = $this->_newer_version_src($js_libs[$product]);
					if (!empty($src)) {
						if (version_compare($ver, '1.8.2', '<='))
							$src = $this->plugin_url . 'includes/js/' . $product . '.min.js';

						$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
					}
				}
			}
			unset ($scriptaculous);
		}

		// jQuery 1.2.6 or 1.3.2
		if ($jquery) {
			list($src, $ver) = $this->_newer_version_src($js_libs['jquery']);
			if (!empty($src)) {
				if ($prototype)
					$src = $this->plugin_url . 'includes/js/jquery-1.2.6.min.js';
				elseif (version_compare($ver, '1.3.2', '<='))
					$src = $this->plugin_url . 'includes/js/jquery-1.3.2.min.js';

				if ($prototype || $mootools || strpos($src, $this->wp_url) === FALSE) {
					$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
				} else {
					$js_libs_src[] = $src;
					$wk_inner_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
				}

				// jQuery noConflict
				if ($prototype || $mootools) {
					$src = $this->plugin_url . 'includes/js/jquery.noconflict.js';
					if (strpos($src, $this->wp_url) === FALSE) {
						$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
					} else {
						$js_libs_src[] = $src;
						$wk_inner_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
					}
				}
			}
		}

		// mootools 1.2.1
		if ($mootools) {
			list($src, $ver) = $this->_newer_version_src($js_libs['mootools']);
			if (!empty($src)) {
				if (version_compare($ver, '1.2.1', '<='))
					$src = $this->plugin_url . 'includes/js/mootools-1.2.1-core-yc.js';

				if ($prototype || $jquery || strpos($src, $this->wp_url) === FALSE) {
					$wk_outer_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
				} else {
					$js_libs_src[] = $src;
					$wk_inner_js .= "<script type=\"text/javascript\" src=\"$src\"></script>\n";
				}
			}
		}

		return array($js_libs_src, $wk_inner_js, $wk_outer_js);
	}

	//**************************************************************************************
	// Combined CSS
	//**************************************************************************************
	private function _combined_css($styles, $css = '', $media = 'all') {
		$html = '';
		$longfilename = ''; $files = array();

		if (empty($media))
			$media = 'all';

		foreach ((array) $styles as $style) {
			$src = trim(preg_replace('/(\.css|\.php)(\?[^\?]*)$/i', '$1', str_replace($this->wp_url, ABSPATH, $style)));
			if (file_exists($src)) {
				$filename = (preg_match('/\.css$/i', $src) ? $src : $style);
				$longfilename .= $filename . filemtime($src);
				$files[] = $filename;
			} else {
				$html .= "<link rel=\"stylesheet\" type=\"text/css\" href=\"{$style}\" media=\"{$media}\" />\n";
			}
		}

		$md5_filename = md5($longfilename . $css);
		$longfilename = 'css/' . $md5_filename . '.css';
		$filename = $this->cache_path . $longfilename;
		if (! file_exists($filename) ) {
			if (count($files) > 0 && !empty($longfilename))
				$css = $this->_files_read($files, 'css') . "\n\n" . $css;

			// Optimise CSS
			$css = $this->_css_optimise($css);

			if (!empty($css))
				$this->_file_write($filename, $css, $this->options['gzip_on']);
		}

		$fileurl = ($this->options['dynamic']
			? $this->self_url . '?f=' . $md5_filename . '&amp;t=css'
			: $this->cache_url . $longfilename
			);
		if (file_exists($filename))
			$html .= "<link rel=\"stylesheet\" type=\"text/css\" href=\"{$fileurl}\" media=\"{$media}\" />\n";

		return $html;
	}

	//**************************************************************************************
	// Combined inline CSS
	//**************************************************************************************
	private function _combined_inline_css($css, $media = 'all') {
		if (empty($css))
			return '';

		if (empty($media))
			$media = 'all';

		$html = '';
		$md5_filename = md5($css);
		$longfilename = 'css/'. $md5_filename . '.css';

		// Optimise CSS
		$css = $this->_css_optimise($css);

		$filename = $this->cache_path . $longfilename;
		if (!file_exists($filename) && !empty($css))
			$this->_file_write($filename, $css, $this->options['gzip_on']);

		$fileurl = ($this->options['dynamic']
			? $this->self_url . '?f=' . $md5_filename . '&amp;t=css'
			: $this->cache_url . $longfilename
			);
		if (file_exists($filename))
			$html .= "<link rel=\"stylesheet\" type=\"text/css\" href=\"{$fileurl}\" media=\"{$media}\" />\n";

		return $html;
	}

	//**************************************************************************************
	// Combined JavaScript
	//**************************************************************************************
	private function _combined_javascript($javascripts, $script = '') {
		$html = '';
		$longfilename = ''; $files = array();

		foreach ((array) $javascripts as $javascript) {
			$src = trim(preg_replace('/(\.js|\.php)(\?[^\?]*)$/i', '$1', str_replace($this->wp_url, ABSPATH, $javascript)));
			if (file_exists($src)) {
				$filename = (preg_match('/\.js$/i', $src) ? $src : $javascript);
				$longfilename .= $filename . filemtime($src);
				$files[] = $filename;
			} else {
				$html .= "<script type=\"text/javascript\" src=\"{$javascript}\"></script>\n";
			}
		}

		$md5_filename = md5($longfilename . $script);
		$longfilename = 'js/' . $md5_filename . '.js';
		$filename = $this->cache_path . $longfilename;
		if (! file_exists($filename) ) {
			if (count($files) > 0 && !empty($longfilename))
				$script = $this->_files_read($files, 'js') . "\n\n" . $script;

			// Minified JavaScript
			$script = $this->_js_minify($script);

			if (!empty($script))
				$this->_file_write($filename, $script, $this->options['gzip_on']);
		}

		$fileurl = ($this->options['dynamic']
			? $this->self_url . '?f=' . $md5_filename . '&amp;t=js'
			: $this->cache_url . $longfilename
			);
		if (file_exists($filename))
			$html .= "<script type=\"text/javascript\" src=\"{$fileurl}\"></script>\n";

		return $html;
	}

	//**************************************************************************************
	// Combined inline JavaScript
	//**************************************************************************************
	private function _combined_inline_javascript($script) {
		if (empty($script))
			return '';

		$html = '';
		$md5_filename = md5($script);
		$longfilename = 'js/' . $md5_filename . '.js';

		// Minified JavaScript
		$script = $this->_js_minify($script);

		$filename = $this->cache_path . $longfilename;
		if (!file_exists($filename) && !empty($script) )
			$this->_file_write($filename, $script, $this->options['gzip_on']);

		$fileurl = ($this->options['dynamic']
			? $this->self_url . '?f=' . $md5_filename . '&amp;t=js'
			: $this->cache_url . $longfilename
			);
		if (file_exists($filename))
			$html .= "<script type=\"text/javascript\" src=\"{$fileurl}\"></script>\n";

		return $html;
	}

	//**************************************************************************************
	// Remove comment
	//**************************************************************************************
	private function _remove_comment($text, $type) {
		$text = trim($text);

		$comment_pattern = '/^[ \t]*\/(?:\*(?:.|\n)*?\*\/|\/.*)/m';

		switch ($type) {
		case 'css':			// inline CSS
			$text = trim(preg_replace(
				  array($comment_pattern, '/^[ \t]+/m', '/[ \t]+$/m')
				, array('', '', '')
				, $text));
			break;

		case 'js':			// inline JavaScript
			$text = trim(preg_replace(
				  array($comment_pattern, '/^[ \t]+/m', '/[ \t]+$/m', '/^<\!\-+/', '/-+>$/')
				, array('', '', '', '', '')
				, $text));
			$text = 'try{'
				. $text . (substr($text, -1) !== ';' ? ';' : '') . "\n"
				. ($this->options['debug_mode']
					? '}catch(e){alert("error: " + e.toString());}'
					: '}finally{};')
				;
			break;
		}

		return ($text . "\n");
	}

	//**************************************************************************************
	// Get Newer version
	//**************************************************************************************
	private function _newer_version_src($js_libs, $limit_ver = '') {
		$src = ''; $ver = '0.0';
		foreach ((array) $js_libs as $key => $val) {
			if ( version_compare( $key, $ver, '>') ) {
				$src = $val; $ver = $key;
			}
		}

		return array($src, $ver);
	}

	//**************************************************************************************
	// Minified JavaScript
	//**************************************************************************************
	private function _js_minify($buffer) {
		$js = trim($buffer . "\n");
		if ($this->options['js_minify']) {
			$js = JSMin::minify($js);
		}
		$js = trim($js);

		return $js . (!empty($js) ? "\n" : '');
	}

	//**************************************************************************************
	// Optimise CSS
	//**************************************************************************************
	private function _css_import($css) {
		if (preg_match_all('/@import[ \t]*url[ \t]*\([\'"]?([^\)]*)[\'"]?\);?/i', $css, $matches, PREG_SET_ORDER)) {
			$search = array(); $replace = array();
			foreach ((array) $matches as $match) {
				$filename  = str_replace($this->wp_url, ABSPATH, trim($match[1]));
				$content   = $this->_file_read($filename);
				$content   = $this->_css_url_edit($content, $filename);
				if (preg_match('/@import[ \t]*url[ \t]*\[\'"]?[^\)]*[\'"]?\;?/i', $content))
					$content = $this->_css_import($content);
				$search[]  = $match[0];
				$replace[] = $content;
			}
			$css = str_replace($search, $replace, $css);
			unset($match); unset($search); unset($replace);
		}
		unset($matches);
		return $css;
	}

	private function _css_optimise($buffer) {
		$css = trim($buffer . "\n");

		if ($this->options['css_optimise']) {
			$css = $this->_css_import($css);

			$csstidy = new csstidy();
			$csstidy->set_cfg('optimise_shorthands',  $this->options['csstidy_option']['optimise_shorthands']);
			$csstidy->set_cfg('compress_colors',      $this->options['csstidy_option']['compress_colors']);
			$csstidy->set_cfg('compress_font-weight', $this->options['csstidy_option']['compress_font_weight']);
			$csstidy->set_cfg('remove_bslash',        $this->options['csstidy_option']['remove_bslash']);
			$csstidy->load_template($this->csstidy_template[$this->options['csstidy_option']['template']]);
			$csstidy->parse($css);
			$css = $csstidy->print->plain();
			unset($csstidy);
		}
		$css = trim($css);

		return $css . (!empty($css) ? "\n" : '');
	}

	//**************************************************************************************
	// Create cache dir
	//**************************************************************************************
	private function _create_cache_dir($cache_dir = '') {
		if (empty($cache_dir))  $cache_dir  = HC_CACHE_DIR;
		$cache_dir = $this->_wp_content_dir('/' . trailingslashit($cache_dir) );

		$mode = 0777;
		if( !file_exists(dirname($cache_dir)) )
			@mkdir(dirname($cache_dir), $mode);
		if( !file_exists($cache_dir) )
			@mkdir($cache_dir, $mode);
		if( !file_exists($cache_dir) . '/css/' )
			@mkdir($cache_dir . '/css/', $mode);
		if( !file_exists($cache_dir) . '/js/' )
			@mkdir($cache_dir . '/js/', $mode);

		return (file_exists($cache_dir) ? $cache_dir : FALSE);
	}

	//**************************************************************************************
	// Remove cache file
	//**************************************************************************************
	private function _remove_cache_file($cache = 'cache', $plugin = 'head-cleaner') {
		$cache_dir = ( !empty($this->cache_path)
			? $this->cache_path
			: $this->_wp_content_dir("/$cache/$plugin/")
			);
		$this->_remove_all_file($cache_dir . 'css');
		$this->_remove_all_file($cache_dir . 'js');
	}

	//**************************************************************************************
	// Remove files
	//**************************************************************************************
	private function _remove_all_file($dir, $rmdir = false) {
		if(file_exists($dir)) {
			if($objs = glob($dir."/*")) {
				foreach((array) $objs as $obj) {
					is_dir($obj)
					? $this->_remove_all_file($obj, true)
					: @unlink($obj);
				}
				unset($objs);
			}
			if ($rmdir) rmdir($dir);
		}
	}

	//**************************************************************************************
	// Create .htaccess
	//**************************************************************************************
	private function _create_htaccess($dir) {
		if (!file_exists($dir))
			return FALSE;

		$rewrite_base = trailingslashit(str_replace(ABS_PATH, '/', $dir));

		$text   = '# BEGIN Head Cleaner' . "\n"
			. '<IfModule mod_rewrite.c>' . "\n"
			. 'RewriteEngine On' . "\n"
			. 'RewriteBase ' . $rewrite_base . "\n"
			. 'RewriteCond %{HTTP:Accept-Encoding} gzip' . "\n"
			. 'RewriteCond %{REQUEST_FILENAME} "\.(css|js)$"' . "\n"
			. 'RewriteCond %{REQUEST_FILENAME} !"\.gz$"' . "\n"
			. 'RewriteCond %{REQUEST_FILENAME}.gz -s' . "\n"
			. 'RewriteRule .+ %{REQUEST_URI}.gz [L]' . "\n"
			. '</IfModule>' . "\n"
			. '# END Head Cleaner' . "\n";
		$filename = trailingslashit($dir) . '.htaccess';

		if ( $this->options['gzip_on'] ) {
			if (!file_exists($filename)) {
				return $this->_file_write($filename, $text, false);
			} else {
				$content = $this->_file_read($filename);
				if ($content !== FALSE) {
					if (strpos($content, '# BEGIN Head Cleaner') === FALSE && strpos($content, 'RewriteRule .+ %{REQUEST_URI}.gz') === FALSE) {
						$text = $content . "\n" . $text;
						return $this->_file_write($filename, $text, false);
					} else {
						return TRUE;
					}
				} else {
					return FALSE;
				}
			}
		} else {
			if ( file_exists($filename) ) {
				$content = trim($this->_file_read($filename));
				if ($content !== FALSE) {
					$content = trim(preg_replace('/# BEGIN Head Cleaner.*# END Head Cleaner/ism', '', $content));
					if ( $text === $content || $content === '') {
						@unlink($filename);
						return TRUE;
					} else {
						return $this->_file_write($filename, $content . "\n", false);
					}
				} else {
					return FALSE;
				}
			} else {
				return TRUE;
			}
		}
	}

	//**************************************************************************************
	// Add Admin Menu
	//**************************************************************************************
	public function admin_menu() {
		$this->addOptionPage(__('Head Cleaner'), array($this, 'option_page'));
	}

	public function plugin_setting_links($links, $file) {
		if (method_exists($this, 'addPluginSettingLinks')) {
			$links = $this->addPluginSettingLinks($links, $file);
		} else {
			$this_plugin = plugin_basename(__FILE__);
			if ($file == $this_plugin) {
				$settings_link = '<a href="' . $this->admin_action . '">' . __('Settings') . '</a>';
				array_unshift($links, $settings_link); // before other links
			}
		}
		return $links;
	}

	//**************************************************************************************
	// Show Option Page
	//**************************************************************************************
	public function option_page() {
		if ($this->_chk_filters_update()) {
			$this->options['filters'] = $this->filters;
			$this->options['head_js'] = $this->head_js;
		}

		if (isset($_POST['options_update'])) {
			if ($this->wp25)
				check_admin_referer("update_options", "_wpnonce_update_options");

			// strip slashes array
			$head_js      = $this->stripArray(isset($_POST['head_js'])      ? $_POST['head_js']      : array());
			$head_filters = $this->stripArray(isset($_POST['head_filters']) ? $_POST['head_filters'] : array());
			$head_remove  = $this->stripArray(isset($_POST['head_remove'])  ? $_POST['head_remove']  : array());
			$foot_filters = $this->stripArray(isset($_POST['foot_filters']) ? $_POST['foot_filters'] : array());
			$foot_remove  = $this->stripArray(isset($_POST['foot_remove'])  ? $_POST['foot_remove']  : array());
			$_POST = $this->stripArray($_POST);

			// get options
			$this->options['xml_declaration'] = (isset($_POST['xml_declaration']) && $_POST['xml_declaration'] == 'on' ? true : false);
			$this->options['ie_conditional']= (isset($_POST['ie_conditional']) && $_POST['ie_conditional'] == 'on' ? true : false);
			$this->options['canonical_tag'] = (isset($_POST['canonical_tag']) && $_POST['canonical_tag'] == 'on' ? true : false);
			$this->options['foot_js']       = (isset($_POST['foot_js']) && $_POST['foot_js'] == 'on' ? true : false);
			$this->options['dynamic']       = (isset($_POST['dynamic']) && $_POST['dynamic'] == 'on' ? true : false);
			$this->options['js_move_foot']  = (isset($_POST['js_move_foot']) && $_POST['js_move_foot'] == 'on' ? true : false);
			$this->options['cache_enabled'] = (isset($_POST['cache_enabled']) && $_POST['cache_enabled'] == 'on' ? true : false);
			$this->options['combined_css']  = (isset($_POST['combined_css']) && $_POST['combined_css'] == 'on' ? true : false);
			$this->options['combined_js']   = (isset($_POST['combined_js']) && $_POST['combined_js'] == 'on' ? true : false);
			$this->options['js_minify']     = (isset($_POST['js_minify']) && $_POST['js_minify'] == 'on' ? true : false);
			$this->options['css_optimise']  = (isset($_POST['css_optimise']) && $_POST['css_optimise'] == 'on' ? true : false);
			$this->options['default_media'] = trim($_POST['default_media']);
			$this->options['gzip_on']       = (isset($_POST['gzip_on']) && $_POST['gzip_on'] == 'on' ? true : false);
			$this->options['use_ajax_libs'] = (isset($_POST['use_ajax_libs']) && $_POST['use_ajax_libs'] == 'on' ? true : false);
			$this->options['debug_mode']    = (isset($_POST['debug_mode']) && $_POST['debug_mode'] == 'on' ? true : false);
			$this->options['csstidy_option']['template']             = (int) $_POST['template'];
			$this->options['csstidy_option']['optimise_shorthands']  = (int) $_POST['optimise_shorthands'];
			$this->options['csstidy_option']['compress_colors']      = (isset($_POST['compress_colors']) && $_POST['compress_colors'] == 'on' ? true : false);
			$this->options['csstidy_option']['compress_font_weight'] = (isset($_POST['compress_font_weight']) && $_POST['compress_font_weight'] == 'on' ? true : false);
			$this->options['csstidy_option']['remove_bslash']        = (isset($_POST['remove_bslash']) && $_POST['remove_bslash'] == 'on' ? true : false);

			$rm_generator = (isset($_POST['rm_generator']) && $_POST['rm_generator'] == 'on' ? true : false);
			$rm_rsd_link  = (isset($_POST['rm_rsd_link'])  && $_POST['rm_rsd_link']  == 'on' ? true : false);
			$rm_manifest  = (isset($_POST['rm_manifest'])  && $_POST['rm_manifest']  == 'on' ? true : false);

			foreach ((array) $this->options['head_js'] as $javascript => $value) {
				if (count($head_js) > 0 && array_search($javascript, $head_js) !== FALSE)
					$this->options['head_js'][$javascript] = FALSE;
				else
					$this->options['head_js'][$javascript] = TRUE;
			}
			unset($head_js);

			$tag = 'wp_head';

			foreach ((array) $this->options['filters'][$tag] as $function_name => $priority) {
				if (count($head_filters) > 0 && array_search($function_name, $head_filters) !== FALSE)
					$this->options['priority'][$tag][$function_name] = HC_PRIORITY + 1;
				elseif (count($head_remove) > 0 && array_search($function_name, $head_remove) !== FALSE)
					$this->options['priority'][$tag][$function_name] = -1;
				elseif (isset($this->options['priority'][$tag][$function_name]))
					$this->options['priority'][$tag][$function_name] = $priority;
			}
			$this->options['priority'][$tag]['wp_generator']     = ($rm_generator ? -1 : $this->options['filters'][$tag]['wp_generator']);
			$this->options['priority'][$tag]['rsd_link']         = ($rm_rsd_link  ? -1 : $this->options['filters'][$tag]['rsd_link']);
			$this->options['priority'][$tag]['wlwmanifest_link'] = ($rm_manifest  ? -1 : $this->options['filters'][$tag]['wlwmanifest_link']);
			unset($head_filters);

			$tag = 'wp_footer';
			foreach ((array) $this->options['filters'][$tag] as $function_name => $priority) {
				if (count($foot_filters) > 0 && array_search($function_name, $foot_filters) !== FALSE)
					$this->options['priority'][$tag][$function_name] = HC_PRIORITY + 1;
				elseif (count($foot_remove) > 0 && array_search($function_name, $foot_remove) !== FALSE)
					$this->options['priority'][$tag][$function_name] = -1;
				elseif (isset($this->options['priority'][$tag][$function_name]))
					$this->options['priority'][$tag][$function_name] = $priority;
			}
			unset($foot_filters);

			// options update
			$this->updateOptions();

			// create .htaccess file
			$cache_dir = $this->_create_cache_dir();
			if ( $cache_dir !== FALSE )
				$this->_create_htaccess($cache_dir);

			// Done!
			$this->note .= "<strong>".__('Done!', $this->textdomain_name)."</strong>";

		} elseif(isset($_POST['remove_cache'])) {
			if ($this->wp25)
				check_admin_referer("remove_cache", "_wpnonce_remove_cache");

			// Remove all cache files
			$this->_remove_cache_file();

			// Done!
			$this->note .= "<strong>".__('Done!', $this->textdomain_name)."</strong>";

		} elseif(isset($_POST['options_delete'])) {
			if ($this->wp25)
				check_admin_referer("delete_options", "_wpnonce_delete_options");

			// options delete
			$this->_delete_settings();

			// Done!
			$this->note .= "<strong>".__('Done!', $this->textdomain_name)."</strong>";
			$this->error++;

		} else {
			$this->activation();
		}

		$out  = '';

		// Add Options
		$out .= "<div class=\"wrap\">\n";
		$out .= "<form method=\"post\" id=\"update_options\" action=\"".$this->admin_action."\">\n";
		$out .= "<h2>".__('Head Cleaner Options', $this->textdomain_name)."</h2><br />\n";
		if ($this->wp25) $out .= $this->makeNonceField("update_options", "_wpnonce_update_options", true, false);

		$out .= "<table class=\"optiontable form-table\" style=\"margin-top:0;\"><tbody>\n";

		$out .= "<tr>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"cache_enabled\" id=\"cache_enabled\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['cache_enabled'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('CSS and JavaScript are cached on the server.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"js_move_foot\" id=\"js_move_foot\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['js_move_foot'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Put JavaScripts at the Bottom.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"dynamic\" id=\"dynamic\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['dynamic'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('CSS and JS are dynamically generated.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "</tr>\n";

		if ($this->options['cache_enabled']) {
			$out .= "<tr>";
			$out .= "<td>";
			$out .= "<input type=\"checkbox\" name=\"combined_css\" id=\"combined_css\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['combined_css'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Two or more CSS is combined.', $this->textdomain_name);
			$out .= "</td>";
			$out .= "<td>";
			$out .= "<input type=\"checkbox\" name=\"css_optimise\" id=\"css_optimise\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['css_optimise'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('CSS is optimized.', $this->textdomain_name);
			$out .= "</td>";
			$out .= "<td>";
			$out .= __('Default media attribute applied to CSS.', $this->textdomain_name);
			$out .= "<input type=\"text\" name=\"default_media\" id=\"default_media\" value=\"{$this->options['default_media']}\" style=\"margin-left:0.5em;\" />";
			$out .= "</td>";
			$out .= "<td>";
			$out .= "</tr>\n";

			$out .= "<tr>";
			$out .= "<td>";
			$out .= "<input type=\"checkbox\" name=\"combined_js\" id=\"combined_js\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['combined_js'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Two or more JavaScript is combined.', $this->textdomain_name);
			$out .= "</td>";
			$out .= "<td>";
			$out .= "<input type=\"checkbox\" name=\"js_minify\" id=\"js_minify\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['js_minify'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('JavaScript is minified.', $this->textdomain_name);
			$out .= "</td>";
			$out .= "<td>";
			$out .= "<input type=\"checkbox\" name=\"foot_js\" id=\"foot_js\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['foot_js'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Bottom JavaScript is combined, too.', $this->textdomain_name);
			$out .= "</td>";
			$out .= "</tr>\n";
		}
		$out .= "<tr>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"gzip_on\" id=\"gzip_on\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['gzip_on'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('gzip compress to CSS and JS.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"use_ajax_libs\" id=\"use_ajax_libs\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['use_ajax_libs'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Use Google Ajax Libraries.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "</td>";
		$out .= "</tr>\n";

		$out .= "<tr>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"xml_declaration\" id=\"xml_declaration\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['xml_declaration'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Add XML Declaration.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"canonical_tag\" id=\"canonical_tag\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['canonical_tag'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Add canonical tag.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"ie_conditional\" id=\"ie_conditional\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['ie_conditional'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Remove IE Conditional Tag.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "</tr>\n";

		$out .= "<tr>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"rm_generator\" id=\"rm_generator\" value=\"on\" style=\"margin-right:0.5em;\"".(isset($this->options['priority']['wp_head']['wp_generator']) && $this->options['priority']['wp_head']['wp_generator'] <= 0 ? " checked=\"true\"" : "")." />";
		$out .= __('Remove generator tag.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"rm_rsd_link\" id=\"rm_rsd_link\" value=\"on\" style=\"margin-right:0.5em;\"".(isset($this->options['priority']['wp_head']['rsd_link']) && $this->options['priority']['wp_head']['rsd_link'] <= 0 ? " checked=\"true\"" : "")." />";
		$out .= __('Remove RSD link tag.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"rm_manifest\" id=\"rm_manifest\" value=\"on\" style=\"margin-right:0.5em;\"".(isset($this->options['priority']['wp_head']['wlwmanifest_link']) && $this->options['priority']['wp_head']['wlwmanifest_link'] <= 0 ? " checked=\"true\"" : "")." />";
		$out .= __('Remove wlwmanifest link tag.', $this->textdomain_name);
		$out .= "</td>";
		$out .= "</tr>\n";

		$out .= "<tr>";
		$out .= "<td>";
		$out .= "<input type=\"checkbox\" name=\"debug_mode\" id=\"debug_mode\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['debug_mode'] === true ? " checked=\"true\"" : "")." />";
		$out .= __('Debug mode', $this->textdomain_name);
		$out .= "</td>";
		$out .= "<td>";
		$out .= "</td>";
		$out .= "<td>";
		$out .= "</td>";
		$out .= "</tr>\n";

		$out .= "</tbody></table>";

		// CSS Tidy Options
		if ($this->options['css_optimise']) {
			$out .= "<div style=\"margin-top:2em;\" id=\"csstidy_option\">\n";
			$out .= "<h3>" . __('The CSS optimization settings', $this->textdomain_name) . "</h3>" ;

			$out .= __('Compression (code layout):', $this->textdomain_name) . '&nbsp;&nbsp;';
			$out .= '<select name="template" id="template">';
			$out .= '<option value="3"' . ($this->options['csstidy_option']['template']==3 ? ' selected="selected"' : '') . '>' . __('Highest (no readability, smallest size)', $this->textdomain_name) .'</option>';
			$out .= '<option value="2"' . ($this->options['csstidy_option']['template']==2 ? ' selected="selected"' : '') . '>' . __('High (moderate readability, smaller size)', $this->textdomain_name) .')</option>';
			$out .= '<option value="1"' . ($this->options['csstidy_option']['template']==1 ? ' selected="selected"' : '') . '>' . __('Standard (balance between readability and size)', $this->textdomain_name) .'</option>';
			$out .= '<option value="0"' . ($this->options['csstidy_option']['template']==0 ? ' selected="selected"' : '') . '>' . __('Low (higher readability)', $this->textdomain_name) .'</option>';
			$out .= '</select>';
			$out .= '&nbsp;&nbsp;&nbsp;&nbsp;';
			$out .= __('Optimise shorthands', $this->textdomain_name) . '&nbsp;&nbsp;';
			$out .= '<select name="optimise_shorthands" id="optimise_shorthands">';
			$out .= '<option value="2"' . ($this->options['csstidy_option']['optimise_shorthands']==2 ? ' selected="selected"' : '') . '>' . __('All optimisations', $this->textdomain_name) .')</option>';
			$out .= '<option value="1"' . ($this->options['csstidy_option']['optimise_shorthands']==1 ? ' selected="selected"' : '') . '>' . __('Safe optimisations', $this->textdomain_name) .'</option>';
			$out .= '<option value="0"' . ($this->options['csstidy_option']['optimise_shorthands']==0 ? ' selected="selected"' : '') . '>' . __("Don't optimise", $this->textdomain_name) .'</option>';
			$out .= '</select>';
			$out .= "<br />\n";

			$out .= "<input type=\"checkbox\" name=\"compress_colors\" id=\"compress_colors\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['csstidy_option']['compress_colors'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Compress colors', $this->textdomain_name);
			$out .= '&nbsp;&nbsp;&nbsp;&nbsp;';
			$out .= "<input type=\"checkbox\" name=\"compress_font_weight\" id=\"compress_font_weight\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['csstidy_option']['compress_font_weight'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Compress font-weight', $this->textdomain_name);
			$out .= '&nbsp;&nbsp;&nbsp;&nbsp;';
			$out .= "<input type=\"checkbox\" name=\"remove_bslash\" id=\"remove_bslash\" value=\"on\" style=\"margin-right:0.5em;\"".($this->options['csstidy_option']['remove_bslash'] === true ? " checked=\"true\"" : "")." />";
			$out .= __('Remove unnecessary backslashes', $this->textdomain_name);
			$out .= "<br />\n";

			$out .= "</div>\n";
		}

		// Active Filters
		$out .= "<div style=\"margin-top:2em;\" id=\"active_filters\">\n";
		$out .= "<h3>" . __('Active Filters', $this->textdomain_name) . "</h3>" ;
		$out .= "<table><tbody>\n";
		$out .= "<tr>";
		$out .= "<th>" . __('Off the subject', $this->textdomain_name) . "</th>\n";
		$out .= "<th>" . __('Remove', $this->textdomain_name) . "</th>\n";
		$out .= "<th>" . __('Head filters', $this->textdomain_name) . "</th>\n";
		if ($this->options['debug_mode'])
			$out .= "<th>" . __('Priority', $this->textdomain_name) . "</th>\n";
		$out .= "</tr>\n";
		$head_filters = array();
		foreach ((array) $this->options['filters']['wp_head'] as $function_name => $priority) {
			if ($priority < HC_PRIORITY) {
				if (isset($this->options['priority']['wp_head'][$function_name]))
					$priority = (int) $this->options['priority']['wp_head'][$function_name];
				if (!isset($head_filters[$priority]))
					$head_filters[$priority] = array();
				$head_filters[$priority][] = $function_name;
			}
		}
		ksort($head_filters, SORT_NUMERIC);
		$i = 0;
		foreach ($head_filters as $priority => $filters) {
			foreach ($filters as $function_name){
				if ( ! preg_match('/^(' . implode('|', $this->default_head_filters) . ')$/i', $function_name)) {
					$out .= "<tr>";
					$out .= "<th><input type=\"checkbox\" name=\"head_filters[$i]\" value=\"$function_name\"".($priority > HC_PRIORITY ? " checked=\"true\"" : "")." /></th>";
					$out .= "<th><input type=\"checkbox\" name=\"head_remove[$i]\" value=\"$function_name\"".($priority <= 0 ? " checked=\"true\"" : "")." /></th>";
					$out .= "<td>$function_name</td>";
					if ($this->options['debug_mode'])
						$out .= "<td>( $priority )</td>";
					$out .= "</tr>\n";
					$i++;
				}
			}
		}
		unset($filters); unset($head_filters);

		if ($this->options['foot_js'] === true) {
			$out .= "<tr><td colspan='3'>&nbsp;</td></tr>\n";
			$out .= "<tr>";
			$out .= "<th>" . __('Off the subject', $this->textdomain_name) . "</th>\n";
			$out .= "<th>" . __('Remove', $this->textdomain_name) . "</th>\n";
			$out .= "<th>" . __('Bottom filters', $this->textdomain_name) . "</th>\n";
			if ($this->options['debug_mode'])
				$out .= "<th>" . __('Priority', $this->textdomain_name) . "</th>\n";
			$out .= "</tr>\n";
			$footer_filters = array();
			foreach ((array) $this->options['filters']['wp_footer'] as $function_name => $priority) {
				if ($priority < HC_PRIORITY) {
					if (isset($this->options['priority']['wp_footer'][$function_name]))
						$priority = (int) $this->options['priority']['wp_footer'][$function_name];
					if (!isset($footer_filters[$priority]))
						$footer_filters[$priority] = array();
					$footer_filters[$priority][] = $function_name;
				}
			}
			ksort($footer_filters, SORT_NUMERIC);
			$i = 0;
			foreach ($footer_filters as $priority => $filters) {
				foreach ($filters as $function_name){
					if ( ! preg_match('/^(' . implode('|', $this->default_head_filters) . ')$/i', $function_name)) {
						$out .= "<tr>";
						$out .= "<th><input type=\"checkbox\" name=\"foot_filters[$i]\" value=\"$function_name\"".($priority > HC_PRIORITY ? " checked=\"true\"" : "")." /></th>";
						$out .= "<th><input type=\"checkbox\" name=\"foot_remove[$i]\" value=\"$function_name\"".($priority <= 0 ? " checked=\"true\"" : "")." /></th>";
						$out .= "<td>$function_name</td>";
						if ($this->options['debug_mode'])
							$out .= "<td>( $priority )</td>";
						$out .= "</tr>\n";
						$i++;
					}
				}
			}
			unset($filters); unset($footer_filters);
		}
		$out .= "</tbody></table>";
		$out .= "</div>\n";

		// Active JavaScripts
		$out .= "<div style=\"margin-top:2em;\" id=\"active_javascripts\">\n";
		$out .= "<h3>" . __('Active JavaScripts', $this->textdomain_name) . "</h3>" ;
		$out .= "<table><tbody>\n";
		$out .= "<tr>";
		$out .= "<th>" . __('Move to footer', $this->textdomain_name) . "</th>\n";
		$out .= "<th>" . __('JavaScripts', $this->textdomain_name) . "</th>\n";
		foreach ($this->options['head_js'] as $javascript => $value) {
			$out .= "<tr>";
			$out .= "<th><input type=\"checkbox\" name=\"head_js[$i]\" value=\"$javascript\"".($value === FALSE ? " checked=\"true\"" : "")." /></th>";
			$out .= "<td>$javascript</td>";
			$i++;
		}
		$out .= "</tbody></table>";
		$out .= "</div>\n";

		// Add Update Button
		$out .= "<p style=\"margin-top:1em\"><input type=\"submit\" name=\"options_update\" class=\"button-primary\" value=\"".__('Update Options', $this->textdomain_name)." &raquo;\" class=\"button\" /></p>";
		$out .= "</form></div>\n";

		// Cache Delete
		$out .= "<div class=\"wrap\" style=\"margin-top:2em;\">\n";
		$out .= "<h2>" . __('Remove all cache files', $this->textdomain_name) . "</h2><br />\n";
		$out .= "<form method=\"post\" id=\"remove_cache\" action=\"".$this->admin_action."\">\n";
		if ($this->wp25) $out .= $this->makeNonceField("remove_cache", "_wpnonce_remove_cache", true, false);
		$out .= "<p>" . __('All cache files are removed.', $this->textdomain_name) . "</p>";
		$out .= "<input type=\"submit\" name=\"remove_cache\" class=\"button-primary\" value=\"".__('Remove All Cache Files', $this->textdomain_name)." &raquo;\" class=\"button\" />";
		$out .= "</form></div>\n";

		// Options Delete
		$out .= "<div class=\"wrap\" style=\"margin-top:2em;\">\n";
		$out .= "<h2>" . __('Uninstall', $this->textdomain_name) . "</h2><br />\n";
		$out .= "<form method=\"post\" id=\"delete_options\" action=\"".$this->admin_action."\">\n";
		if ($this->wp25) $out .= $this->makeNonceField("delete_options", "_wpnonce_delete_options", true, false);
		$out .= "<p>" . __('All the settings of &quot;Head Cleaner&quot; are deleted.', $this->textdomain_name) . "</p>";
		$out .= "<input type=\"submit\" name=\"options_delete\" class=\"button-primary\" value=\"".__('Delete Options', $this->textdomain_name)." &raquo;\" class=\"button\" />";
		$out .= "</form></div>\n";

		// Output
		echo (!empty($this->note) ? "<div id=\"message\" class=\"updated fade\"><p>{$this->note}</p></div>\n" : '') . "\n";
		echo ($this->error == 0 ? $out : '') . "\n";

	}

	//**************************************************************************************
	// Delete Settings
	//**************************************************************************************
	private function _delete_settings() {
		$this->deleteOptions();
		$this->_remove_cache_file();
		$this->options = $this->_init_options();
	}

	//**************************************************************************************
	// Get function name
	//**************************************************************************************
	private function _get_function_name($function) {
		return (is_array($function)
			? (get_class($function[0]) !== FALSE ? get_class($function[0]) : $function[0]) . '::' . $function[1]
			: $function
			);
	}

	//**************************************************************************************
	// Get Filters
	//**************************************************************************************
	private function _get_filters($tag = '') {
		global $wp_filter;

		if (empty($tag) && function_exists('current_filter'))
			$tag = current_filter();

		if (!isset($this->filters[$tag]))
			$this->filters[$tag] = array();

		$active_filters = (isset($wp_filter[$tag])
			? (array) $wp_filter[$tag]
			: array());
		foreach ($active_filters as $priority => $filters) {
			foreach ($filters as $filter) {
				$function_name = $this->_get_function_name($filter['function']);
				$this->filters[$tag][$function_name] = $priority;
			}
		}
		unset($active_filters);

		return $this->filters;
	}

	//**************************************************************************************
	// Filters update Check
	//**************************************************************************************
	private function _chk_filters_update(){
		$retval = false;
		foreach ( $this->filters as $tag => $filters ) {
			if ( isset($this->options['filters'][$tag]) ) {
				foreach ( $filters as $function_name => $priority) {
					$retval = ( !isset($this->options['filters'][$tag][$function_name]) );
					if ($retval) break;
				}
			} else {
				$retval = true; break;
			}
		}
		unset ($filters);

		foreach ( $this->head_js as $key => $val ) {
			if ( !isset($this->options['head_js'][$key]) ) {
				$retval = true; break;
			}
		}

		return $retval;
	}

	//**************************************************************************************
	// Filters priority change
	//**************************************************************************************
	private function _change_filters_priority($tag = ''){
		global $wp_filter;

		if (empty($tag) && function_exists('current_filter'))
			$tag = current_filter();

		$active_filters = (isset($wp_filter[$tag])
			? $wp_filter[$tag]
			: array());
		$custom_priority = (isset($this->options['priority'][$tag])
			? $this->options['priority'][$tag]
			: array());
		foreach ($this->no_conflict as $function_name) {
			$custom_priority[$function_name] = HC_PRIORITY + 1;
		}
		foreach ($active_filters as $priority => $filters) {
			foreach ($filters as $filter) {
				$function_name = $this->_get_function_name($filter['function']);
				if ( isset($custom_priority[$function_name]) && $custom_priority[$function_name] != $priority) {
					remove_filter( $tag, $filter['function'], $priority);
					if ($custom_priority[$function_name] > 0)
						add_filter( $tag, $filter['function'], $custom_priority[$function_name]);
				}
			}
		}
		unset($custom_priority);
		unset($active_filters);
	}

	//**************************************************************************************
	// Debug Information
	//**************************************************************************************
	private function _microtime_diff($start, $end=NULL) {
		if( !$end )
			$end= microtime();
		list($start_usec, $start_sec) = explode(" ", $start);
		list($end_usec, $end_sec) = explode(" ", $end);
		$diff_sec= intval($end_sec) - intval($start_sec);
		$diff_usec= floatval($end_usec) - floatval($start_usec);
		return floatval( $diff_sec ) + $diff_usec;
	}

	private function _get_debug_info($buffer, $tag = '') {
		$ret_val = "<!--\n";

		$ret_val .= "***** Processing time ****************************\n"
			 .  $this->_microtime_diff($this->mtime_start) . " seconds\n"
			 .  "**************************************************\n\n";
		$this->mtime_start = microtime();

		$ret_val .= "***** Original ***********************************\n"
			 .  str_replace(array('<', '>'), array('＜', '＞'), $buffer) . "\n"
			 .  "**************************************************\n\n";

		$ret_val .= "***** Filters ************************************\n";
		if (empty($tag) && function_exists('current_filter'))
			$tag = current_filter();
		$active_filters = (isset($this->filters[$tag])
			? $this->filters[$tag]
			: array());
		foreach ($active_filters as $function_name => $priority) {
			if (strpos($function_name, 'HeadCleanerController') === FALSE && strpos($function_name, 'noindex') === FALSE)
				$ret_val .= "    ($priority) : $function_name\n";
		}
		$ret_val .= "**************************************************\n\n";

		$ret_val .= "***** ob_list_handlers ***************************\n";
		if (function_exists('ob_list_handlers')) {
			$active_handlers = ob_list_handlers();
			foreach ($active_handlers as $handler) {
				$ret_val .= "   $handler\n";
			}
			unset($active_handlers);
		}
		$ret_val .= "**************************************************\n\n";

		$ret_val .= "--> \n";

		return $ret_val;
	}
}

global $head_cleaner;

$head_cleaner = new HeadCleanerController();

if (is_admin()) {
	add_action('admin_menu', array(&$head_cleaner, 'admin_menu'));
	add_filter('plugin_action_links', array(&$head_cleaner, 'plugin_setting_links'), 10, 2 );

	if (function_exists('register_activation_hook'))
		register_activation_hook(__FILE__, array(&$head_cleaner, 'activation'));
	if (function_exists('register_deactivation_hook'))
		register_deactivation_hook(__FILE__, array(&$head_cleaner, 'deactivation'));

} else {
	add_action('get_header', array(&$head_cleaner, 'head_start'));
	add_action('wp_footer',  array(&$head_cleaner, 'filters_save'), 11);

}
?>