/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.bool.BooleanState;
import woolpack.fn.Fn;

/**
 * 値検証と値変換のユーティリティです。
 * 
 * @author nakamura
 * 
 */
public final class ValidatorUtils {
	
	/**
	 * メッセージ情報からメッセージを抽出した一覧を生成する関数です。
	 * <br/>適用しているデザインパターン：Strategy。
	 */
	public static final Fn<Iterable<AddressedMessage>, List<String>, RuntimeException> TO_MESSAGE =
		new Fn<Iterable<AddressedMessage>, List<String>, RuntimeException>() {
			public List<String> exec(final Iterable<AddressedMessage> c) throws RuntimeException {
				final List<String> list = new ArrayList<String>();
				for (final AddressedMessage message : c) {
					list.add(message.getMessage());
				}
				return list;
			}
		};
	
	/**
	 * {@link ValidatorContext#getValue()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final Fn<ValidatorContext, Object, RuntimeException> VALUE =
		new Fn<ValidatorContext, Object, RuntimeException>() {
		public Object exec(final ValidatorContext c) {
			return c.getValue();
		}
	};
	
	/**
	 * {@link ValidatorContext#getInputMap()}を返す関数です。
	 * <br/>適用しているデザインパターン：Accessor。
	 */
	public static final Fn<ValidatorContext, Map<String, List<Object>>, RuntimeException> MAP =
		new Fn<ValidatorContext, Map<String, List<Object>>, RuntimeException>() {
		public Map<String, List<Object>> exec(final ValidatorContext c) {
			return c.getInputMap();
		}
	};
	
	/**
	 * {@link ValidatorContext#getInputMap()}の
	 * {@link ValidatorContext#getKey()}を
	 * キーとした値一覧を返す関数です。
	 */
	public static final Fn<ValidatorContext, List<Object>, RuntimeException> VALUES =
		new Fn<ValidatorContext, List<Object>, RuntimeException>() {
		public List<Object> exec(final ValidatorContext c) {
			return c.getInputMap().get(c.getKey());
		}
	};

	private ValidatorUtils() {
	}
	
	/**
	 * プロパティ名で並べ替えたメッセージの一覧を生成する関数を生成します。
	 * サーブレットAPIの request.getParameterMap() はキーの出現順が不確定なためこのメソッドを定義しています。
	 * <br/>適用しているデザインパターン：Strategy。
	 * @param iterable キーの一覧。ここで指定しないキーに関しては順序を保持したまま一覧に追加します。
	 * @return 関数。
	 */
	public static final Fn<Iterable<AddressedMessage>, List<String>, RuntimeException> toOrderedMessage(
			final Iterable<String> iterable) {
		return new Fn<Iterable<AddressedMessage>, List<String>, RuntimeException>() {
			public List<String> exec(final Iterable<AddressedMessage> c) throws RuntimeException {
				final List<String> list = new ArrayList<String>();
				for (final String param : iterable) {
					for(final Iterator<AddressedMessage> it = c.iterator(); it.hasNext();) {
						final AddressedMessage message = it.next();
						if (param.equals(message.getKey())) {
							list.add(message.getMessage());
							it.remove();
						}
					}
				}
				for (final AddressedMessage message : c) {
					list.add(message.getMessage());
				}
				return list;
			}
		};
	}
	
	/**
	 * プロパティ名で委譲先を分岐する関数を生成します。
	 * プロパティ名に対応する値が存在しない場合は委譲しません。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <E>
	 * @param operator 値検証一覧の呼び出し方。
	 * @param map プロパティ名と委譲先の対応表。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> branchByNameIfExists(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranch<E>(operator, map) {
			@Override
			protected Collection<String> keySet(final ValidatorContext context) {
				return context.getInputMap().keySet();
			}
		};
	}
	
	/**
	 * プロパティ名で委譲先を分岐する関数を生成します。
	 * プロパティ名に対応する値がからの場合は委譲しないため、
	 * ブラウザから送信されない項目を無視する際に使用します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <E>
	 * @param operator 値検証一覧の呼び出し方。
	 * @param map プロパティ名と委譲先の対応表。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> branchByNameIfNotEmpty(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranch<E>(operator, map) {
			@Override
			protected Collection<String> keySet(final ValidatorContext context) {
				final List<String> list = new ArrayList<String>();
				myloop:for (final Entry<String, List<Object>> entry : context.getInputMap().entrySet()) {
					for (final Object object : entry.getValue()) {
						if (object != null && !"".equals(object)) {
							list.add(entry.getKey());
							continue myloop;
						}
					}
				}
				return list;
			}
		};
	}
	
	/**
	 * プロパティ名で委譲先を分岐する関数を生成します。
	 * プロパティ名に対応する値が存在しない場合も委譲するため、
	 * ブラウザから送信されない項目を必須を検証する際に使用することができます。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <E>
	 * @param operator 値検証一覧の呼び出し方。
	 * @param map プロパティ名と委譲先の対応表。
	 * @return 関数。
	 * @see NameBranch
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> branchByName(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator,
			final Map<String, ? extends Fn<? super ValidatorContext, Boolean, ? extends E>> map) {
		return new NameBranch<E>(operator, map);
	}
	
	/**
	 * 値の変換を委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：参照透過と副作用のAdapter。
	 * @param <E>
	 * @param fn 値を変換する委譲先。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> convertValue(final Fn<Object, ?, ? extends E> fn) {
		return new Fn<ValidatorContext, Boolean, E>() {
			public Boolean exec(final ValidatorContext c) throws E {
				c.setValue(fn.exec(c.getValue()));
				return true;
			}
		};
	}
	
	/**
	 * メッセージを追加する関数です。
	 * @param message メッセージ。
	 * @return 関数。
	 */
	public static Fn<ValidatorContext, Boolean, RuntimeException> message(final String message) {
		return new Fn<ValidatorContext, Boolean, RuntimeException>() {
			public Boolean exec(final ValidatorContext context) {
				context.add(message);
				return false;
			}
		};
	}
	
	/**
	 * インデックス値を一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化する関数を生成します。
	 * <br/>適用しているデザインパターン：副作用のSand Box。
	 * @param <E>
	 * @param index 一時的に設定するインデックス。
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> localIndex(
			final int index,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new Fn<ValidatorContext, Boolean, E>() {
			public Boolean exec(final ValidatorContext context) throws E {
				final int baseIndex = context.getIndex();
				try {
					context.setIndex(index);
					return fn.exec(context);
				} finally {
					context.setIndex(baseIndex);
				}
			}
		};
	}
	
	/**
	 * キー値を一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化する関数を生成します。
	 * <br/>適用しているデザインパターン：副作用のSand Box。
	 * @param <E>
	 * @param key 一時的に設定するキー。
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> localKey(
			final String key,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new Fn<ValidatorContext, Boolean, E>() {
			public Boolean exec(final ValidatorContext context) throws E {
				final String baseKey = context.getKey();
				try {
					context.setKey(key);
					return fn.exec(context);
				} finally {
					context.setKey(baseKey);
				}
			}
		};
	}
	
	/**
	 * {@link ValidatorContext#getInputMap()}の
	 * {@link ValidatorContext#getKey()}をキーとした
	 * 値一覧に対し順次委譲先を実行する関数を生成します。
	 * <br/>適用しているデザインパターン：Proxy。
	 * @param <E>
	 * @param operator 値検証一覧の呼び出し方。
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<ValidatorContext, Boolean, E> loopValue(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		return new Fn<ValidatorContext, Boolean, E>() {
			public Boolean exec(final ValidatorContext context) throws E {
				final int length;
				{
					final List<Object> list = context.getInputMap().get(context.getKey());
					if (list == null) {
						length = 1;
					} else {
						length = list.size();
					}
				}

				final Iterable<Fn<ValidatorContext, Boolean, ? extends E>> iterable = 
					new Iterable<Fn<ValidatorContext, Boolean, ? extends E>>() {
						public Iterator<Fn<ValidatorContext, Boolean, ? extends E>> iterator() {
							return new Iterator<Fn<ValidatorContext, Boolean, ? extends E>>() {
								private int i = 0;

								public boolean hasNext() {
									return i < length;
								}

								public Fn<ValidatorContext, Boolean, ? extends E> next() {
									return new Fn<ValidatorContext, Boolean, E>() {
										public Boolean exec(
												final ValidatorContext context) throws E {
											final int baseIndex = context.getIndex();
											try {
												context.setIndex(i++);
												return fn.exec(context);
											} finally {
												context.setIndex(baseIndex);
											}
										}
									};
								}
								public void remove() {
									throw new UnsupportedOperationException();
								}
							};
						}
					};
				
				final BooleanState state = operator.exec(null);
				for (final Fn<? super ValidatorContext, Boolean, ? extends E> e : iterable) {
					state.in(e.exec(context));
					if (state.isStopped()) {
						break;
					}
				}
				return state.isCurrent();
			}
		};
	}
}
