/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.typeconvert;

import java.util.Map;

import woolpack.fn.Delegator;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

/**
 * プリミティブ型をラッパー型に置き換えて委譲する関数です。
 * <br/>適用しているデザインパターン：コンテキストの状態を変えて委譲するProxy。
 * @author nakamura
 *
 */
public class ToPrimitiveConverter extends Delegator<ConvertContext, Void, RuntimeException> {
	private static final Map<Class, Object> DEDAULT_VALUE_MAP = Utils.<Class, Object>
	map(boolean.class, Boolean.FALSE)
	.map(char.class, (char) 0)
	.map(byte.class, (byte) 0)
	.map(short.class, (short) 0)
	.map(int.class, (int) 0)
	.map(long.class, (long) 0)
	.map(float.class, (float) 0)
	.map(double.class, (double) 0);

	public ToPrimitiveConverter(
			final Fn<? super ConvertContext, Void, ? extends RuntimeException> fn) {
		super(fn);
	}

	@Override
	public Void exec(final ConvertContext c) {
		final Class clazz = c.getToType();
		c.setToType(FnUtils.TO_WRAPPER.exec(clazz));
		super.exec(c);
		if (c.getValue() == null) {
			c.setValue(DEDAULT_VALUE_MAP.get(clazz));
		}
		return null;
	}
}
