// compile:
// g++ -o eq2xml eq2xml.cpp general.cpp
// sample:
// eq2xml -g 0.2 -x \-1 -X 1 -z \-1 -Z 1 "exp(-1 * (\$x*\$x+\$z*\$z))"
#include <iostream>
#include <cstdlib>
#include <unistd.h>
#include <vector>
#include <string>
#include <sstream>
#include <algorithm>

#include "parse_eq.hpp"
#include "general.hpp"

void usage( const int &ret = 0 );
void printTri( std::ostream &out,
               const std::string &tag,
               const double &x,
               const double &y,
               const double &z );

void usage( const int &ret ) {
  std::cerr << "usage: eq2xml [options] [function(x,z)]\n\n";
  std::cerr << "options: -x [value(Float)]   min x value (default=-10.0)\n";
  std::cerr << "         -X [value(Float)]   max x value (default=10.0)\n";
  std::cerr << "         -z [value(Float)]   min z value (default=10.0)\n";
  std::cerr << "         -Z [value(Float)]   max z value (default=10.0)\n";
  std::cerr << "         -c [color(string)]  color(default=\"0x00FF00\")\n";
  std::cerr << "         -a [alpha(float)]   alpha(default=1.0, opaque)\n";
  std::cerr << "         -g [grid(float)]    gridsize(default=0.5)" << std::endl;
  exit( ret );
}

void printTri( std::ostream &out,
               const std::string &tag,
               const double &x,
               const double &y,
               const double &z ) {
  out.setf( std::ios::fixed );
  out.precision( 3 );
  out << tag << "=\"" << x << " " << y << " " << -z << "\" ";
}

int main( int argc, char* argv[] ) {
  double xmin = -10.0, xmax = 10.0;
  double zmin = -10.0, zmax = 10.0;
  double alpha = 1.0, grid = 0.5;
  std::string color = "0x00FF00";

  if ( argc == 1 ) usage(0);
  int ch;
  extern char *optarg;
  extern int optind, opterr;
  while( ( ch = getopt( argc, argv, "x:X:z:Z:g:c:a:" ) ) != -1 ) {
    if ( ch == 'x' ) {
      xmin = atof( optarg );
    } else if ( ch == 'X' ) {
      xmax = atof( optarg );
    } else if ( ch == 'z' ) {
      zmin = atof( optarg );
    } else if ( ch == 'Z' ) {
      zmax = atof( optarg );
    } else if ( ch == 'g' ) {
      grid = atof( optarg );
    } else if ( ch == 'c' ) {
      color = optarg;
    } else if ( ch == 'a' ) {
      alpha = atof( optarg );
    }
  }
  argc -= optind;
  argv += optind;

  if ( argc < 1 ) {
    std::cerr << "eq2xml: please input equation." << std::endl;
    usage(0);
  }
  std::string eq = argv[0];
  parseeq::Equation< Vector3D > hogehoge;

  const int numx = int(floor( ( xmax - xmin ) / grid )) + 1;
  const int numz = int(floor( ( zmax - zmin ) / grid )) + 1;
  // data container
  std::vector< std::vector< double > > dat( numx, std::vector< double >( numz, 0.0 ) );
  for ( int i = 0; i < numx; ++i ) {
    double curx = xmin + (double)i * grid;
    for ( int j = 0; j < numz; ++j ) {
      double curz = zmin + (double)j * grid;
      hogehoge.registerValue( "x", curx );
      hogehoge.registerValue( "z", curz );
      dat[i][j] = hogehoge.eval( eq ).first;
    }
  }

  std::cout << "<WMXML>" << std::endl;
  std::cout << "  <SHAPE color=\"" << color
            << "\" alpha=\"" << alpha
            << "\">" << std::endl;
  for ( int i = 1; i < numx; ++i ) {
    double curx = xmin + (double)i * grid;
    double prex = xmin + (double)( i - 1 ) * grid;
    for ( int j = 1; j < numz; ++j ) {
      double curz = zmin + (double)j * grid;
      double prez = zmin + (double)( j - 1 ) * grid;
      std::cout << "    <TRI ";
      printTri( std::cout, "p0",
                prex, dat[i-1][j-1], prez );
      printTri( std::cout, "p1",
                curx, dat[i][j-1], prez );
      printTri( std::cout, "p2",
                prex, dat[i-1][j], curz );
      std::cout << "/>" << std::endl;
      std::cout << "    <TRI ";
      printTri( std::cout, "p0",
                curx, dat[i][j-1], prez );
      printTri( std::cout, "p1",
                curx, dat[i][j], curz );
      printTri( std::cout, "p2",
                prex, dat[i-1][j], curz );
      std::cout << "/>" << std::endl;
    }
  }
  std::cout << "  </SHAPE>" << std::endl;
  std::cout << "</WMXML>" << std::endl;

  return(0);
}
