import xpath.XPath;
import xpath.xml.XPathXml;
import xpath.xml.XPathHxXml;

import flash.geom.Matrix3D;
import flash.geom.Vector3D;
import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

class WMSystem {
  public var myString( __getMyString, __setMyString ):String;
  private var atoms( null, null ):Array< WMAtom >;
  private var bonds( null, null ):Array< WMBond >;
  private var chains( null, null ):Array< WMChain >;
  private var shapes( null, null ):Array< WMShape >;

  private var autoScale( null, null ):Bool;
  private var scaleFactor( null, null ):Float;
  private var scaleFactorManual( null, null ):Float;

  private var origin( null, null ):Point3D;

  public function new() {
    // read XML data from resource file attached in compilation phase.
    // -resource (filename):structure
    myString = haxe.Resource.getString( "structure" );
    // if you modify myString after the call, please call gen() to
    // reconstruct the scene.

    autoScale = true;
    scaleFactor = 0.3;
    scaleFactorManual = 10.0;
    origin = null;
  }

  public function clear() {
    atoms = [];
    bonds = [];
    chains = [];
    shapes = [];
    origin = null;
  }

  public function draw( c:Context3D,
                        mpos:Matrix3D,
                        proj:Matrix3D,
                        voffset:Vector3D,
                        light:Vector3D,
                        cpos:Vector3D ):Void {
    for ( atom in atoms ) atom.draw( c, mpos, proj, voffset, light, cpos );
    for ( bond in bonds ) bond.draw( c, mpos, proj, voffset, light, cpos );
    for ( chain in chains ) chain.draw( c, mpos, proj, voffset, light, cpos );
    for ( shape in shapes ) shape.draw( c, mpos, proj, voffset, light, cpos );
  }

  public function gen( ?wm:Watermelon = null,
                       ?c:Context3D = null ):Void {
    if ( wm == null || c == null ) return;
    loadXML( wm );
    initializeCrd();
    for ( atom in atoms ) atom.gen( c );
    for ( bond in bonds ) bond.gen( c );
    for ( chain in chains ) chain.gen( c );
    for ( shape in shapes ) shape.gen( c );
  }

  public function initializeCrd():Void {
    if ( empty() ) return;
    if ( origin == null ) origin = geometricCenter();
    translate( Point3D.getMultiply( origin, -1.0 ) );
    rescaleCoord();
  }

  public function geometricCenter():Point3D {
    var ret:Point3D = new Point3D( 0.0, 0.0, 0.0 );
    for ( atom in atoms ) ret.add( atom.pos );
    for ( bond in bonds ) {
      ret.add( bond.pos0 );
      ret.add( bond.pos1 );
    }
    var totnum = atoms.length + bonds.length * 2;
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) ret.add( p );
      totnum += pos.length;
    }
    for ( shape in shapes ) {
      ret.add( shape.sumPos() );
      totnum += shape.num();
    }
    ret.multiply( 1.0 / totnum );
    return( ret );
  }

  public function translate( p:Point3D ):Void {
    for ( atom in atoms ) atom.pos.add( p );
    for ( bond in bonds ) {
      bond.pos0.add( p );
      bond.pos1.add( p );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( cp in pos ) cp.add( p );
    }
    for ( shape in shapes ) {
      var pos:Array< Point3D > = shape.getPositions0();
      for ( cp in pos ) cp.add( p );
      shape.setPositions0( pos );
      pos = shape.getPositions1();
      for ( cp in pos ) cp.add( p );
      shape.setPositions1( pos );
      pos = shape.getPositions2();
      for ( cp in pos ) cp.add( p );
      shape.setPositions2( pos );
    }
  }

  public function rescaleCoord():Void {
    var swidth = flash.Lib.current.stage.stageWidth * scaleFactor;
    var sheight = flash.Lib.current.stage.stageHeight * scaleFactor;
    if ( !autoScale ) {
      scaleCoord( scaleFactorManual );
      return;
    }

    // origin is assumed to be defined.
    var xmax:Float = 0.0;
    var ymax:Float = 0.0;
    for ( atom in atoms ) {
      var p:Point3D = atom.pos;
      xmax = Math.max( xmax, Math.max( Math.abs( p.x ), Math.abs( p.z ) ) );
      ymax = Math.max( ymax, Math.abs( p.y ) );
    }
    for ( bond in bonds ) {
      var p0:Point3D = bond.pos0;
      var p1:Point3D = bond.pos1;
      xmax = Math.max( xmax, Math.max( Math.abs( p0.x ), Math.abs( p0.z ) ) );
      xmax = Math.max( xmax, Math.max( Math.abs( p1.x ), Math.abs( p1.z ) ) );
      ymax = Math.max( ymax, Math.max( Math.abs( p0.y ), Math.abs( p1.y ) ) );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) {
        xmax = Math.max( xmax, Math.max( Math.abs( p.x ), Math.abs( p.z ) ) );
        ymax = Math.max( ymax, Math.abs( p.y ) );
      }
    }
    for ( shape in shapes ) {
      var pmax:Point3D = shape.absmax();
      xmax = Math.max( xmax, Math.max( pmax.x, pmax.z ) );
      ymax = Math.max( xmax, pmax.y );
    }
    // shapes are tentatively ignored
    xmax = Math.max( 1.0, xmax );
    ymax = Math.max( 1.0, ymax );
    // scale coordinate
    scaleCoord( Math.min( swidth / xmax, sheight / ymax ) );
  }

  public function scaleCoord( scale:Float ):Void {
    for ( atom in atoms ) atom.pos.multiply( scale );
    for ( bond in bonds ) {
      bond.pos0.multiply( scale );
      bond.pos1.multiply( scale );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) p.multiply( scale );
    }
    for ( shape in shapes ) {
      var pos:Array< Point3D > = shape.getPositions0();
      for ( p in pos ) p.multiply( scale );
      shape.setPositions0( pos );
      pos = shape.getPositions1();
      for ( p in pos ) p.multiply( scale );
      shape.setPositions1( pos );
      pos = shape.getPositions2();
      for ( p in pos ) p.multiply( scale );
      shape.setPositions2( pos );
    }
  }

  public function loadXML( ?wm:Watermelon = null ):Void {
    clear();
    if ( wm == null ) return;

    // alias will be processed here
    // string is converted in cpp macro-like fashion
    // ex.
    // alias MYBOND BOND radius="x" ...
    // <MYBOND pos0="yyy" pos1="zzz" />
    // is converted to <BOND radius="x" pos0="yyy" pos1="zzz" />

    // split string with LF
    var lines:Array< String > = myString.split( String.fromCharCode( 10 ) );
    var myLines:Array< String > = [];
    for ( l in lines ) {
      // split string with CR
      var ls:Array< String > = l.split( String.fromCharCode( 13 ) );
      for ( ll in ls ) {
        if ( ll.length > 0 ) myLines.push( ll );
      }
    }
    lines = [];
    // search alias and replace strings
    var newString:String = myString;
    for ( l in myLines ) {
      var ls:Array< String > = l.split( " " );
      if ( ls.length < 3 ) continue;
      if ( ls[0].toLowerCase() == "alias" ) {
        var newname:String = ls[1];
        ls.shift();
        ls.shift();
        newString = StringTools.replace( newString, newname, ls.join( " " ) );
      }
    }

    var myxml:Xml = Xml.parse( newString );

    // read settings
    var xml_setting:XPath = new XPath( "WMXML/SETTING" );
    __loadXMLSettings( wm, xml_setting.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read chains and their elements
    var xml_chain:XPath = new XPath( "WMXML/CHAIN" );
    __loadXMLChains( wm, xml_chain.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read chains and their elements
    var xml_shape:XPath = new XPath( "WMXML/SHAPE" );
    __loadXMLShapes( wm, xml_shape.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read ATOMs
    var xml_atom:XPath = new XPath( "WMXML/ATOM" );
    __loadXMLAtoms( wm, xml_atom.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read BONDs
    var xml_bond:XPath = new XPath( "WMXML/BOND" );
    __loadXMLBonds( wm, xml_bond.selectNodes( XPathHxXml.wrapNode( myxml ) ) );
  }

  private function __parseSettingAttributes( wm:Watermelon,
                                             nd:Xml ):Void {
    if ( nd.exists( "light" ) ) { // <LIGHT> in former versions
      wm.setLightDirection( Point3D.fromString( nd.get( "light" ) ) );
    }
    if ( nd.exists( "arrate" ) ) { // <AR> in former versions
      wm.setAutoRotationRate( Std.parseFloat( nd.get( "arrate" ) ) );
    }
    if ( nd.exists( "framerate" ) ) { // <FRAMERATE> in former versions
      wm.setFrameRate( Std.parseInt( nd.get( "framerate" ) ) );
    }
    if ( nd.exists( "origin" ) ) {
      origin = Point3D.fromString( nd.get( "origin" ) );
    }
  }

  private function __loadXMLSettings( wm:Watermelon,
                                      ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      __parseSettingAttributes( wm, cast( nd, XPathHxXml ).getWrappedXml() );
      var ndio:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      for ( ndc in ndio.elements() ) {
        var nn:String = ndc.nodeName;
        switch( nn ) {
          case "ATOM":
            WMAtom.defaults.loadFromXml( ndc );
          case "BOND":
            WMBond.defaults.loadFromXml( ndc );
            if ( ndc.exists( "rounded" ) ) {
              WMBond.def_round = ( Std.parseInt( ndc.get( "rounded" ) ) > 0 );
            }
            if ( ndc.exists( "round" ) ) {
              WMBond.def_round = ( Std.parseInt( ndc.get( "round" ) ) > 0 );
            }
            if ( ndc.exists( "exclude" ) ) {
              WMBond.def_exclude = ( Std.parseInt( ndc.get( "exclude" ) ) > 0 );
            }
          case "RIBBON":
            WMRibbon.defaultsRibbon.loadFromXml( ndc );
          case "COIL":
            WMRibbon.defaultsCoil.loadFromXml( ndc );
          case "SHAPE":
            WMShape.defaults.loadFromXml( ndc );
          case "CAMERA":
            if ( ndc.exists( "z" ) ) {
              wm.setCameraPosZ( Std.parseFloat( ndc.get( "z" ) ) );
            }
          case "PROTECT":
            wm.setProtectData();
          case "RADIUS":
            if ( ndc.exists( "method" ) ) {
              var met:String = ndc.get( "method" );
              if ( met.toLowerCase() == "absolute" ) {
                WMBase.useRelative = false;
              } else if ( met.toLowerCase() == "relative" ) {
                WMBase.useRelative = true;
              }
            }
            if ( ndc.exists( "scale" ) ) {
              WMBase.characteristicSize = Std.parseFloat( ndc.get( "scale" ) );
            }
          case "WHEEL":
            if ( ndc.exists( "scale" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "scale" ) ) );
            }
            if ( ndc.exists( "depth" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "depth" ) ) );
            }
          case "MATRIX":
            if ( ndc.exists( "data" ) ) {
              var m:Matrix4 = Matrix4.fromString( ndc.get( "data" ) );
              wm.setInitialView( m );
            }
          case "AUTOSCALE":
            if ( ndc.exists( "manual" ) ) {
              var n:Int = Std.parseInt( ndc.get( "manual" ) );
              if ( n > 0 ) autoScale = false;
            }
            if ( ndc.exists( "autoscale" ) ) {
              scaleFactor = Std.parseFloat( ndc.get( "autoscale" ) );
            }
            if ( ndc.exists( "manualscale" ) ) {
              scaleFactorManual = Std.parseFloat( ndc.get( "manualscale" ) );
            }
        }
      }
    }
  }

  private function __loadXMLChains( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ni:Int = 1;
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      if ( ndi.exists( "N" ) ) ni = Std.parseInt( ndi.get( "N" ) );
      // get child elements describing control points
      var children:Array< Dynamic > = new Array< Dynamic >();
      for ( child in ndi.elementsNamed( "POINT" ) ) {
        var anon = { pos:child.get( "pos" ), index:Std.parseInt( child.get( "index" ) ), dir:null, n:-1 };
        if ( child.exists( "N" ) ) anon.n = Std.parseInt( child.get( "N" ) );
        if ( child.exists( "dir" ) ) anon.dir = child.get( "dir" );
        children.push( anon );
      }
      // ignore too short chains, since catmull-rom interpolation is not
      // available for such short chains
      if ( children.length < 4 ) {
        trace( "__loadXMLChains: ignoring very short chain... " );
        trace( "__loadXMLChains: A <CHAIN> must contain at least 4 <POINT>s" );
        continue;
      }
      children.sort( __hasSmallIndex );
      var chain:WMChain = new WMChain();
      chain.setPositions( children, ni );
      // read ribbons and coils
      for ( child in ndi.elementsNamed( "RIBBON" ) ) {
        var rib:WMRibbon = new WMRibbon( true );
        rib.loadFromXml( child );
        chain.register( rib );
      }
      for ( child in ndi.elementsNamed( "COIL" ) ) {
        var coi:WMRibbon = new WMRibbon( false );
        coi.loadFromXml( child );
        chain.register( coi );
      }
      chains.push( chain );
    }
  }

  private function __hasSmallIndex( o0:Dynamic,
                                    o1:Dynamic ):Int {
    if ( o0.index == o1.index ) return(0);
    if ( o0.index < o1.index ) return(-1);
    return(1);
  }

  private function __loadXMLShapes( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var shape:WMShape = new WMShape();
      shape.loadFromXml( ndi );
      shapes.push( shape );
    }
  }

  private function __loadXMLAtoms( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var at:WMAtom = new WMAtom();
      at.loadFromXml( ndi );
      atoms.push( at );
    }
  }

  private function __loadXMLBonds( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var bd:WMBond = new WMBond();
      bd.loadFromXml( ndi );
      bonds.push( bd );
    }
  }

  public function empty():Bool {
    if ( numAtoms() + numBonds() + numChains() + numShapes() == 0 ) return( true );
    return( false );
  }
  public function numAtoms():Int { return( atoms.length ); }
  public function numBonds():Int { return( bonds.length ); }
  public function numChains():Int { return( chains.length ); }
  public function numShapes():Int { return( shapes.length ); }
  public function __getMyString():String { return( myString ); }
  public function __setMyString( s:String ):String {
    myString = s;
    return( myString );
  }
}
