/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"


static LRESULT WINAPI
w32stc_DefWindowProc (HWND   hWnd,
                      UINT   Msg,
                      WPARAM wParam,
                      LPARAM lParam)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!wd)
    return 0;
  switch (Msg)
    {
      case STM_SETICON:
        return w32stc_DefWindowProc (hWnd, STM_SETIMAGE, IMAGE_ICON, wParam);
      case STM_GETICON:
        return w32stc_DefWindowProc (hWnd, STM_GETIMAGE, IMAGE_ICON, 0);
      case STM_SETIMAGE:
        if (GTK_IS_IMAGE (hWnd) && wParam == IMAGE_BITMAP)
          {
            /* ja:ビットマップ */
            GdkPixbuf *pixbuf;

            pixbuf = gtk_image_get_pixbuf (GTK_IMAGE (hWnd));
            if (pixbuf)
              g_object_ref (pixbuf);
            gtk_image_new_from_pixbuf (GINT_TO_POINTER (lParam));
            return GPOINTER_TO_INT (pixbuf);
          }
        return GPOINTER_TO_INT (NULL);
      case STM_GETIMAGE:
        if (GTK_IS_IMAGE (hWnd) && wParam == IMAGE_BITMAP)/* ja:ビットマップ */
          return GPOINTER_TO_INT (gtk_image_get_pixbuf (GTK_IMAGE (hWnd)));
        return GPOINTER_TO_INT (NULL);
      case WM_GETTEXT:
        if (GTK_IS_LABEL (hWnd))
          {
            const gchar *utf8str;

            utf8str = gtk_label_get_text (GTK_LABEL (hWnd));
            if (wParam > 0 && lParam && utf8str)
              {
                if (wd->widechar)
                  {
                    gunichar2 *wc;

                    wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                    lstrcpynW (GINT_TO_POINTER (lParam), wc, wParam);
                    g_free (wc);
                    return lstrlenW (GINT_TO_POINTER (lParam));
                  }
                else
                  {
                    gchar *mb;

                    mb = w32ldr_utf8_to_mb (utf8str);
                    g_strncpy (GINT_TO_POINTER (lParam), mb, wParam);
                    g_free (mb);
                    return g_strlen (GINT_TO_POINTER (lParam));
                  }
              }
          }
        return 0;
      case WM_GETTEXTLENGTH:
        if (GTK_IS_LABEL (hWnd))
          {
            const gchar *utf8str;

            utf8str = gtk_label_get_text (GTK_LABEL (hWnd));
            if (utf8str)
              {
                LRESULT lResult;

                if (wd->widechar)
                  {
                    gunichar2 *wc;

                    wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                    lResult = lstrlenW (wc);
                    g_free (wc);
                  }
                else
                  {
                    gchar *mb;

                    mb = w32ldr_utf8_to_mb (utf8str);
                    lResult = g_strlen (mb);
                    g_free (mb);
                  }
                return lResult;
              }
          }
        return 0;
      case WM_SETTEXT:
        if (GTK_IS_LABEL (hWnd))
          {
            gchar *utf8str;

            utf8str = wd->widechar ? g_utf16_to_utf8 (GINT_TO_POINTER (lParam),
                                                        -1, NULL, NULL, NULL)
                            : w32ldr_utf8_from_mb (GINT_TO_POINTER (lParam));
            if (wd->dwStyle & SS_NOPREFIX)
              {
                gtk_label_set_text (GTK_LABEL (hWnd), utf8str);
              }
            else
              {
                gchar *title;
                gint i, j = 0;

                title = g_malloc ((g_strlen (utf8str) * 2 + 1)
                                                            * sizeof (gchar));
                for (i = 0; utf8str[i] != '\0'; i++)
                  switch (utf8str[i])
                    {
                      case '-': title[j++] = '-';
                      case '&': title[j++] = '_'; break;
                      default:  title[j++] = utf8str[i];
                    }
                title[j] = '\0';
                gtk_label_set_text_with_mnemonic (GTK_LABEL (hWnd), title);
                g_free (title);
              }
            g_free (utf8str);
          }
        return TRUE;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}


/* ja:ラベルの状態が変化したとき */
static void
w32stc_state_changed (GtkWidget    *widget,
                      GtkStateType  state,
                      gpointer      user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (state == GTK_STATE_NORMAL)
    {
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_ENABLE), GPOINTER_TO_INT (widget));
      else
        SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_ENABLE), GPOINTER_TO_INT (widget));
    }
  else if (state == GTK_STATE_INSENSITIVE)
    {
      if (wd->widechar)
        SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_DISABLE), GPOINTER_TO_INT (widget));
      else
        SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_DISABLE), GPOINTER_TO_INT (widget));
    }
}


/* ja:ラベルがクリックされたとき */
static gboolean
w32stc_button_press_event (GtkWidget      *widget,
                           GdkEventButton *event,
                           gpointer        user_data)
{
  if (event->button == 1)
    {
      W32LdrWindowData *wd;

      wd = g_object_get_data (G_OBJECT (widget), "user_data");
      if (event->type == GDK_BUTTON_PRESS)
        {
          if (wd->widechar)
            SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_CLICKED), GPOINTER_TO_INT (widget));
          else
            SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_CLICKED), GPOINTER_TO_INT (widget));
        }
      else if (event->type == GDK_2BUTTON_PRESS)
        {
          if (wd->widechar)
            SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_DBLCLK), GPOINTER_TO_INT (widget));
          else
            SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, STN_DBLCLK), GPOINTER_TO_INT (widget));
        }
    }
  return FALSE;
}


GtkWidget *
w32ldr_control_create_static (const gchar    *windowname,
                              const guint32   style,
                              const guint32   exstyle,
                              const gint      width,
                              const gint      height,
                              const guint16   id,
                              HWND            hWndParent,
                              HINSTANCE       hInstance,
                              const gboolean  widechar)
{
  GtkWidget *widget;
  W32LdrWindowData *wd;

  switch (style & SS_TYPEMASK)
    {
      case SS_LEFT:
      case SS_CENTER:
      case SS_RIGHT:
      case SS_LEFTNOWORDWRAP:
        widget = gtk_label_new ("");
        switch (style & SS_TYPEMASK)
          {
            case SS_CENTER:
              gtk_label_set_justify (GTK_LABEL (widget), GTK_JUSTIFY_CENTER);
              break;
            case SS_RIGHT:
              gtk_label_set_justify (GTK_LABEL (widget), GTK_JUSTIFY_RIGHT);
              break;
            default:
              gtk_label_set_justify (GTK_LABEL (widget), GTK_JUSTIFY_LEFT);
          }
        gtk_label_set_line_wrap (GTK_LABEL (widget),
                                (style & SS_TYPEMASK) != SS_LEFTNOWORDWRAP);
        break;
      case SS_ICON:
      case SS_BITMAP:
        widget = gtk_image_new ();
      default:
        g_print ("Unsupported Control Static.\n");
        return NULL;
    }
  if (style & SS_NOTIFY)
    {
      g_signal_connect (G_OBJECT (widget), "state-changed",
                                G_CALLBACK (w32stc_state_changed), NULL);
      g_signal_connect (G_OBJECT (widget), "button-press-event",
                                G_CALLBACK (w32stc_button_press_event), NULL);
      gtk_widget_add_events (widget, GDK_BUTTON_PRESS_MASK);
    }
  wd = g_malloc (sizeof (W32LdrWindowData));
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32stc_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = FALSE;
  wd->classname = g_strdup ("STATIC");
  wd->focus = NULL;
  wd->child = NULL;
  g_object_set_data (G_OBJECT (widget), "user_data", wd);
  if ((style & SS_TYPEMASK) == SS_ICON)
    {
      /* ja:アイコン */
      if (width != CW_USEDEFAULT && height != CW_USEDEFAULT)
        gtk_widget_set_size_request (widget, width, height);
    }
  else if ((style & SS_TYPEMASK) == SS_BITMAP)
    {
      /* ja:ビットマップ */
      GdkPixbuf *pixbuf;

      if (GPOINTER_TO_UINT (windowname) > G_MAXUINT16)
        if (widechar)
          {
            gunichar2 *wc;

            wc = g_utf8_to_utf16 (windowname, -1, NULL, NULL, NULL);
            pixbuf = LoadBitmapW (hInstance, wc);
            g_free (wc);
          }
        else
          {
            gchar *mb;

            mb = w32ldr_utf8_to_mb (windowname);
            pixbuf = LoadBitmapA (hInstance, mb);
            g_free (mb);
          }
      else
        pixbuf = widechar ? LoadBitmapW (hInstance,
                            MAKEINTRESOURCEW (GPOINTER_TO_INT (windowname)))
                          : LoadBitmapA (hInstance,
                            MAKEINTRESOURCEA (GPOINTER_TO_INT (windowname)));
      if (pixbuf)
        {
          gtk_image_set_from_pixbuf (GTK_IMAGE (widget), pixbuf);
          g_object_unref (pixbuf);
        }
    }
  else if (GTK_IS_LABEL (widget))
    {
      /* ja:テキスト */
      LPARAM lParam;

      lParam = widechar
                    ? GPOINTER_TO_INT (w32ldr_atom_get_string_wc (windowname))
                    : GPOINTER_TO_INT (w32ldr_atom_get_string_mb (windowname));
      w32stc_DefWindowProc (widget, WM_SETTEXT, 0, lParam);
      g_free (GINT_TO_POINTER (lParam));
      wd->resizable = TRUE;
    }
  return widget;
}
