/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "pixbuf.h"


/******************************************************************************
*                                                                             *
* ja:GdkPixbufビットマップ変換関数                                            *
*                                                                             *
******************************************************************************/
/*  ja:ビットマップをGdkPixbufに変換する
    bmih,BitmapInfoHeader構造体へのポインタ
    RET,GdkPixbuf                                                           */
GdkPixbuf *
pixbuf_from_bitmap (const BitmapInfoHeader *bmih)
{
  GdkPixbuf *pixbuf = NULL;

  if (bmih)
    {
      guchar *pixels;
      guint8 *bits;
      gint x, y, n_channels, width, height, rowstride;
      BitmapInfoHeader *bmih0;

      width = bmih_get_width (bmih);
      height = bmih_get_height (bmih);
      /* ja:圧縮 */
      if (bmih_get_compression (bmih) == BI_COMP_RLE4
                                || bmih_get_compression (bmih) == BI_COMP_RLE8)
        {
          bmih0 = g_malloc (bx_all_bytes (width, height,
                                        bmih_get_bit_count (bmih), BI_COMP_RGB,
                                                bmih_get_color_used (bmih)));
          bitmap_expand (bmih, bmih0);
        }
      else
        {
          bmih0 = (BitmapInfoHeader *)bmih;
        }
      /* ja:24/32ビット以外もしくはビットフィールドは32ビットに変換 */
      if ((bmih_get_bit_count (bmih0) != 24
                                        && bmih_get_bit_count (bmih0) != 32)
                        || bmih_get_compression (bmih0) == BI_COMP_BITFIELDS)
        {
          BitmapInfoHeader *bmih1;

          bmih1 = g_malloc (bx_all_bytes (width, height, 32, BI_COMP_RGB, 0));
          bitmap_convert_32 (bmih0, bmih1);
          if (bmih != bmih0)
            g_free (bmih0);
          bmih0 = bmih1;
        }
      bits = (guint8 *)bmih0 + bm_all_bytes (bmih0) - bm_line_bytes (bmih0);
      pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
                        bmih_get_bit_count (bmih0) == 32, 8, width, height);
      n_channels = gdk_pixbuf_get_n_channels (pixbuf);
      pixels = gdk_pixbuf_get_pixels (pixbuf);
      rowstride = gdk_pixbuf_get_rowstride (pixbuf);
      if (bmih_get_bit_count (bmih0) == 24 && n_channels == 3)
        {
          for (y = 0; y < height; y++)
            {
              for (x = 0; x < width; x++)
                {
                  *pixels++ = bits[x * 3 + 2];
                  *pixels++ = bits[x * 3 + 1];
                  *pixels++ = bits[x * 3    ];
                }
              pixels += rowstride - width * 3;
              bits -= bm_line_bytes (bmih0);
            }
        }
      else if (bmih_get_bit_count (bmih0) == 32 && n_channels == 4)
        {
          for (y = 0; y < height; y++)
            {
              for (x = 0; x < width; x++)
                {
                  *pixels++ = bits[x * 4 + 2];
                  *pixels++ = bits[x * 4 + 1];
                  *pixels++ = bits[x * 4    ];
                  *pixels++ = bits[x * 4 + 3];
                }
              pixels += rowstride - width * 4;
              bits -= bm_line_bytes (bmih0);
            }
        }
      else
        {
          g_object_unref (pixbuf);
          pixbuf = NULL;
        }
      if (bmih != bmih0)
        g_free (bmih0);
    }
  return pixbuf;
}


/*  ja:GdkPixbufをビットマップに変換する
    pixbuf,GdkPixbuf
       RET,BitmapInfoHeader構造体へのポインタ                               */
BitmapInfoHeader *
pixbuf_to_bitmap (GdkPixbuf *pixbuf)
{
  BitmapInfoHeader *bmih = NULL;

  if (pixbuf)
    {
      guchar *pixels;
      gint n_channels, width, height, rowstride;

      n_channels = gdk_pixbuf_get_n_channels (pixbuf);
      pixels = gdk_pixbuf_get_pixels (pixbuf);
      width = gdk_pixbuf_get_width (pixbuf);
      height = gdk_pixbuf_get_height (pixbuf);
      rowstride = gdk_pixbuf_get_rowstride (pixbuf);
      if (gdk_pixbuf_get_colorspace (pixbuf) == GDK_COLORSPACE_RGB && pixels
                && (n_channels == 3 || n_channels == 4)
                && gdk_pixbuf_get_bits_per_sample (pixbuf) == 8
                && width > 0 && height > 0 && rowstride > 0)
        {
          guint8 *bits;
          gint x, y;

          bmih = g_malloc0
                (bx_all_bytes (width, height, n_channels * 8, BI_COMP_RGB, 0));
          bmih_set_size (bmih, BMIH_SIZE);
          bmih_set_width (bmih, width);
          bmih_set_height (bmih, height);
          bmih_set_planes (bmih, 1);
          bmih_set_bit_count (bmih, n_channels * 8);
          bmih_set_compression (bmih, BI_COMP_RGB);
          bmih_set_size_image (bmih, bm_image_bytes (bmih));
          bits = (guint8 *)bmih + bm_all_bytes (bmih) - bm_line_bytes (bmih);
          if (n_channels == 3)
            for (y = 0; y < height; y++)
              {
                for (x = 0; x < width; x++)
                  {
                    bits[x * 3 + 2] = *pixels++;
                    bits[x * 3 + 1] = *pixels++;
                    bits[x * 3    ] = *pixels++;
                  }
                pixels += rowstride - width * 3;
                bits -= bm_line_bytes (bmih);
              }
          else
            for (y = 0; y < height; y++)
              {
                for (x = 0; x < width; x++)
                  {
                    bits[x * 4 + 2] = *pixels++;
                    bits[x * 4 + 1] = *pixels++;
                    bits[x * 4    ] = *pixels++;
                    bits[x * 4 + 3] = *pixels++;
                  }
                pixels += rowstride - width * 4;
                bits -= bm_line_bytes (bmih);
              }
        }
    }
  return bmih;
}
