/*
    avicore
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"
#include "avifmt.h"
#include "chunk.h"
#include "gsrfmt.h"
#include "misc/pixbuf.h"


/******************************************************************************
*                                                                             *
* ja:AVI入力関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:ファイルを判別する
    file,ファイル名
     RET,AVI_FILE_*                                                         */
gint
avi_main_type (const gchar *file)
{
  guint8 buffer[12];            /* ja:ファイル判別用のバッファ */
  FileIO *fio;
#if ! GTK_CHECK_VERSION(2,4,0)
  GdkPixbuf *pixbuf;
  GError *error = NULL;
#endif /* not GTK_CHECK_VERSION(2,4,0) */

  /* ja:ファイルを開く */
  if (!file || !(fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING)))
    return AVI_FILE_UNKNOWN;
  /* ja:ヘッタ部分の読み込み */
  if (fileio_read (fio, buffer, sizeof (guint8) * 12) != sizeof (guint8) * 12)
    {
      fileio_close (fio);
      return AVI_FILE_UNKNOWN;
    }
  if (!fileio_close (fio))
    return AVI_FILE_UNKNOWN;
  /* ja:ファイルの形式毎に分岐 */
  if (buffer[0] == 'R' && buffer[1] == 'I'
                                    && buffer[2] == 'F' && buffer[3] == 'F'
                                    && buffer[8] == 'A' && buffer[9] == 'V'
                                    && buffer[10] == 'I' && buffer[11] == ' ')
    return AVI_FILE_AVI;/* en:AVI */
  if (buffer[0] == 'G' && buffer[1] == 'S'
                                    && buffer[2] == 'R' && buffer[3] == '\0')
    return AVI_FILE_GSR;/* en:GtkShot Raw */
  else if (buffer[0] == 'B' && buffer[1] == 'M')
    return AVI_FILE_BITMAP;/* ja:ビットマップ */
  else if (buffer[0] == 'R' && buffer[1] == 'I'
                                    && buffer[2] == 'F' && buffer[3] == 'F'
                                    && buffer[8] == 'W' && buffer[9] == 'A'
                                    && buffer[10 ]== 'V' && buffer[11] == 'E')
    return AVI_FILE_WAVE;/* ja:WAVEファイル */
  else if (buffer[0] == '#' && buffer[1] == 'V' && buffer[2] == 'M'
                    && buffer[3] == 'A' && buffer[4] == 'I' && buffer[5] == 'D'
                                && (buffer[6] == '\n' || buffer[6] == '\r'))
    return AVI_FILE_SCENARIO;/* ja:シナリオオブジェクト */
#if GTK_CHECK_VERSION(2,4,0)
  if (gdk_pixbuf_get_file_info (file, NULL, NULL))
    return AVI_FILE_PIXBUF;
#else /* not GTK_CHECK_VERSION(2,4,0) */
  pixbuf = gdk_pixbuf_new_from_file (file, &error);
  if (error)
    g_error_free (error);
  if (pixbuf)
    {
      g_object_unref (pixbuf);
      return AVI_FILE_PIXBUF;
    }
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  return AVI_FILE_UNKNOWN;
}


/*  ja:ストリームの数を取得する
    file,ファイル名
     RET,ストリームの数,0:エラー                                            */
gint
avi_main_streams (const gchar *file)
{
  switch (avi_main_type (file))
    {
      case AVI_FILE_AVI:
        {
          FileIO *fio;
          AviMainHeader amh;
          Chunk *ck;

          /* ja:ファイルを開く */
          if (!file || !(fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING)))
            return 0;
          /* ja:ファイルディスクリターからチャンクポインタを作る */
          ck = chunk_open (fio);
          /* ja:RIFFチャンクに入る */
          if (!chunk_in (fio, ck))
            {
              chunk_free (ck);
              fileio_close (fio);
              return 0;
            }
          /* ja:フォームがhdrlのLISTチャンクを探す */
          while (chunk_form (fio) != CK_LIST_AVIHEADER)
            if (!chunk_next (fio, ck))
              {
                chunk_free (ck);
                fileio_close (fio);
                return 0;
              }
          /* ja:フォームがhdrlのLISTチャンクに入る(RIFF'AVI ' -> LIST'hdrl') */
          if (!chunk_in (fio, ck))
            {
              chunk_free (ck);
              fileio_close (fio);
              return 0;
            }
          /* ja:avihチャンクを探す(AviMainHeader) */
          while (chunk_id (fio) != CK_ID_AVIMAINHDR)
            if (!chunk_next (fio, ck))
              {
                chunk_free (ck);
                fileio_close (fio);
                return 0;
              }
          /* ja:avihチャンクのサイズ(AviMainHeader) */
          /* ja:avihチャンクに入る(AviMainHeader)
                                        (RIFF'AVI ' -> LIST'hdrl' -> avih) */
          /* ja:AviMainHeaderを読み込む */
          if (chunk_size (fio) != AMH_SIZE || !chunk_in (fio, ck)
                            || fileio_read (fio, &amh, AMH_SIZE) != AMH_SIZE)
            {
              chunk_free (ck);
              fileio_close (fio);
              return 0;
            }
          /* ja:チャンクポインタを解放する */
          return chunk_free (ck) & fileio_close (fio)
                                                ? amh_get_streams (&amh) : 0;
        }
      case AVI_FILE_GSR:
        {
          FileIO *fio;
          GsrMainHeader gmh;

          /* ja:ファイルを開く */
          return file && (fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING))
                    && ((fileio_read (fio, &gmh, GMH_SIZE) == GMH_SIZE)
                                                        & fileio_close (fio))
                                                ? gmh_get_streams (&gmh) : 0;
        }
      case AVI_FILE_BITMAP:
      case AVI_FILE_PIXBUF:
      case AVI_FILE_WAVE: return 1;
    }
  return 0;
}


/*  ja:AVIをファイルから開く
     file,ファイル名
    param,ストリーム番号
      RET,AVI編集ハンドル,NULL:エラー                                       */
static AviEdit *
avi_edit_open_with_param (const gchar *file,
                          const gint   param)
{
  AviEdit *avi_edit;

  if (!file)
    return NULL;
  avi_edit = g_malloc0 (sizeof (AviEdit));      /* ja:メモリ確保 */
  avi_edit->file = avi_file_open (file, param,
                    &avi_edit->type, &avi_edit->rate, &avi_edit->scale,
                    &avi_edit->priority, &avi_edit->language, &avi_edit->name);
  if (!avi_edit->file)
    {
      g_free (avi_edit);
      return NULL;
    }
  /* ja:展開後のフォーマット */
  if (avi_edit_type (avi_edit) == AVI_TYPE_VIDEO)
    {
      if (avi_edit->file->handler == AVI_STREAM_COMP_DIB
                            || avi_edit->file->handler == BI_COMP_RGB
                            || avi_edit->file->handler == BI_COMP_RLE8
                            || avi_edit->file->handler == BI_COMP_RLE4
                            || avi_edit->file->handler == BI_COMP_BITFIELDS)
        {
          avi_edit->file->handler = AVI_STREAM_COMP_DIB;
          avi_edit->bmih = g_memdup (avi_edit->file->bmih,
                                    bm_header_bytes (avi_edit->file->bmih));
          if (bmih_get_compression (avi_edit->bmih) == AVI_STREAM_COMP_DIB)
            bmih_set_compression (avi_edit->bmih, BI_COMP_RGB);
          if (bmih_get_size_image (avi_edit->bmih) <= 0)
            bmih_set_size_image (avi_edit->bmih,
                                            bm_image_bytes (avi_edit->bmih));
        }
      else
        {
          IcmObject *icm_object;

          /* ja:圧縮 */
          icm_object = icm_open (avi_edit->file->handler, ICM_MODE_DECOMPRESS);
          if (!icm_object
                || !(avi_edit->bmih = g_malloc (icm_decompress_get_format_size
                                        (icm_object, avi_edit->file->bmih)))
                || (!icm_decompress_get_format (icm_object,
                                        avi_edit->file->bmih, avi_edit->bmih)
                                                    | !icm_close (icm_object)))
            {
              avi_file_close (avi_edit->file);
              g_free (avi_edit->bmih);
              g_free (avi_edit);
              return NULL;
            }
          avi_edit->bmih = g_realloc (avi_edit->bmih,
                                            bm_header_bytes (avi_edit->bmih));
        }
    }
  else
    {
      if (wfx_get_format_tag (avi_edit->file->wfx) == WAVE_FMT_PCM)
        {
          avi_edit->wfx = g_memdup (avi_edit->file->wfx,
                                        wf_header_bytes (avi_edit->file->wfx));
        }
      else
        {
          AcmObject *acm_object;

          /* ja:圧縮 */
          acm_object = acm_open (wfx_get_format_tag (avi_edit->file->wfx));
          if (!acm_object
                || !(avi_edit->wfx = g_malloc (acm_decompress_get_format_size
                                            (acm_object, avi_edit->file->wfx)))
                || (!acm_decompress_get_format (acm_object,
                                            avi_edit->file->wfx, avi_edit->wfx)
                                                    | !acm_close (acm_object)))
            {
              avi_file_close (avi_edit->file);
              g_free (avi_edit->wfx);
              g_free (avi_edit);
              return NULL;
            }
          avi_edit->rate = wfx_get_average_bytes_per_sec (avi_edit->wfx);
          avi_edit->scale = wfx_get_block_align (avi_edit->wfx);
        }
    }
  /* ja:情報 */
  avi_edit->length = avi_edit->file->length;
  return avi_edit;
}


/*  ja:AVIをファイルから開く
    file,ファイル名
     RET,AVI編集ハンドル,NULL:エラー                                        */
AviEdit **
avi_edit_open (const gchar *file)
{
  AviEdit **avi_edit = NULL;

  if (avi_main_type (file) == AVI_FILE_SCENARIO)
    {
      gchar *scenario = NULL;
      goffset offset;
      FileIO *fio;

      /* ja:ファイルを開く */
      fio = fileio_open (file, FILEIO_ACCESS_READ,
                                FILEIO_SHARE_READ, FILEIO_MODE_OPEN_EXISTING);
      if (!fio)
        return NULL;
      /* ja:読み込み */
      offset = fileio_seek (fio, 0, SEEK_END);
      if (offset == -1 || fileio_seek (fio, 0, FILEIO_SEEK_SET) == -1
                                || !(scenario = g_malloc ((gssize)offset + 1)))
        {
          fileio_close (fio);
          return NULL;
        }
      scenario[offset] = '\0';
      if (fileio_read (fio, scenario, (gssize)offset) != offset)
        {
          fileio_close (fio);
          g_free (scenario);
          return NULL;
        }
      if (!fileio_close (fio))
        {
          g_free (scenario);
          return NULL;
        }
      avi_edit = avi_edit_from_scenario (scenario);
      g_free (scenario);
    }
  else
    {
      gint streams;

      streams = avi_main_streams (file);
      if (streams > 0)
        {
          gint i, j = 0;

          avi_edit = g_malloc ((streams + 1) * sizeof (AviEdit *));
          for (i = 0; i < streams; i++)
            {
              avi_edit[j] = avi_edit_open_with_param (file, i);
              if (avi_edit[j])
                j++;
            }
          avi_edit[j] = NULL;
          avi_edit = g_realloc (avi_edit,
                                    (j > 0 ? j + 1 : 0) * sizeof (AviEdit *));
        }
    }
  return avi_edit;
}


/*  ja:AVIをファイルから開く
    data,データ
     RET,AVI編集ハンドル,NULL:エラー                                        */
AviEdit *
avi_edit_open_from_memory (gconstpointer data)
{
  gint type;
  AviEdit *avi_edit = NULL;
  AviFile *avi_file;

  avi_file = avi_file_open_memory (data, &type);
  if (avi_file)
    {
      /* ja:AVI編集ハンドル */
      avi_edit = g_malloc0 (sizeof (AviEdit));
      avi_edit->file = avi_file;
      avi_edit->type = type;
      avi_edit->length = avi_file->length;
      if (type == AVI_TYPE_VIDEO)
        {
          /* ja:ビデオ */
          avi_edit->rate = 1;
          avi_edit->scale = 1;
          avi_edit->bmih = g_memdup (avi_file->bmih,
                                            bm_header_bytes (avi_file->bmih));
          if (bmih_get_size_image (avi_edit->bmih) <= 0)
            bmih_set_size_image (avi_edit->bmih,
                                            bm_image_bytes (avi_edit->bmih));
        }
      else if (type == AVI_TYPE_AUDIO)
        {
          /* ja:オーディオ */
          if (wfx_get_format_tag (avi_file->wfx) == WAVE_FMT_PCM)
            {
              avi_edit->wfx = g_memdup (avi_file->wfx,
                                            wf_header_bytes (avi_file->wfx));
            }
          else
            {
              AcmObject *acm_object;

              /* ja:圧縮 */
              acm_object = acm_open (wfx_get_format_tag (avi_edit->file->wfx));
              if (!acm_object
                    || !(avi_edit->wfx = g_malloc
                                (acm_decompress_get_format_size (acm_object,
                                                        avi_edit->file->wfx)))
                    || (!acm_decompress_get_format (acm_object,
                                            avi_edit->file->wfx, avi_edit->wfx)
                                                    | !acm_close (acm_object)))
                {
                  avi_file_close (avi_edit->file);
                  g_free (avi_edit->wfx);
                  g_free (avi_edit);
                  return NULL;
                }
            }
          avi_edit->rate = wfx_get_average_bytes_per_sec (avi_edit->wfx);
          avi_edit->scale = wfx_get_block_align (avi_edit->wfx);
        }
      else
        {
          /* ja:エラー */
          avi_file_close (avi_file);
          g_free (avi_edit);
          return NULL;
        }
    }
  return avi_edit;
}


/*  ja:AVIをGdkPixbufから開く
    pixbuf,GdkPixbuf
       RET,AVI編集ハンドル,NULL:エラー                                      */
AviEdit *
avi_edit_open_from_pixbuf (GdkPixbuf *pixbuf)
{
  AviEdit *avi_edit = NULL;
  BitmapInfoHeader *bmih;

  bmih = pixbuf_to_bitmap (pixbuf);
  if (bmih)
    {
      avi_edit = avi_edit_open_from_memory (bmih);
      g_free (bmih);
    }
  return avi_edit;
}


/*  ja:AVIを閉じる
    avi_edit,AVI編集ハンドル
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_edit_close (AviEdit *avi_edit)
{
  gint i;

  if (!avi_edit)
    return FALSE;
  while (avi_edit->file)
    {
      AviFile *avi_file;

      avi_file = avi_edit->file->next;
      avi_file_close (avi_edit->file);
      avi_edit->file = avi_file;
    }
  g_free (avi_edit->name);
  g_free (avi_edit->bmih);
  for (i = 0; i < AVI_EDIT_CACHE; i++)
    g_free (avi_edit->buf[i].data);
  g_free (avi_edit->wfx);
  g_free (avi_edit);
  return TRUE;
}
