/*
    Video maid
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "general.h"
#if GTK_CHECK_VERSION(2,8,0)
# include "icons.h"
#endif /* GTK_CHECK_VERSION(2,8,0) */
#include "thread.h"
#include "misc/fileio.h"
#include "orz/orzmdi.h"
#include <time.h>
#if defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H)
# include <sys/time.h>
#endif /* defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H) */
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
# include <time.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:スレッド関数群                                                           *
*                                                                             *
******************************************************************************/
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gint critical = FALSE;
#else /* not USE_THREAD */
/*  ja:メニューを設定する
    task,TRUE:通常,FALSE:終了のみ                                           */
static void
thread_set_menu_bar (gboolean task)
{
  gint i;
  GList *glist;
  GtkWidget *menu_shell;

  menu_shell = misc_find_menu (menu_entries, "/file/");
  glist = gtk_container_children (GTK_CONTAINER (menu_shell));
  for (i = g_list_length (glist) - 2; i >= 0; i--)
    gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)), task);
  g_list_free (glist);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/edit"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/view"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/option"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/window"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/help"), task);
  for (i = 0; toolbar_entries[i].path; i++)
    if (toolbar_entries[i].tool_item)
      gtk_widget_set_sensitive (GTK_WIDGET (toolbar_entries[i].tool_item),
                                                                        task);
  if (task)
    set_menu_bar ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}
#endif /* not USE_THREAD */


/*  ja:リストに加える
      id,スレッドID
    mode,モード(0:終了,1:開く,2:保存,3:フレーム数,4:反転)
    file,ファイル名
     RET,TRUE:正常終了,FALSE:エラー                                         */
#ifdef USE_THREAD
gboolean
thread_insert (GThread     *id,
               const gint   mode,
               const gchar *file)
#else /* not USE_THREAD */
gboolean
thread_insert (gint         id,
               const gint   mode,
               const gchar *file)
#endif /* not USE_THREAD */
{
  const gchar *fmt = "%c";
  gchar str[512], *text_mode, *text_file, *text_time;
  GtkListStore *store;
  GtkStockItem stock_item;
  GtkTreeIter iter;
  ThreadList *tl;
#if GTK_CHECK_VERSION(2,8,0)
  const gchar *stock_id;
  GdkPixbuf *pixbuf_mode, *pixbuf_file;
#endif /* GTK_CHECK_VERSION(2,8,0) */
  time_t t;

  tl = g_malloc (sizeof (ThreadList));
  tl->mode = mode;
  tl->percent = 0;
  tl->userbreak = TRUE;
  tl->id = id;
  /* ja:モード */
  switch (mode)
    {
      case THREAD_MODE_EXIT:
        gtk_stock_lookup (GTK_STOCK_QUIT, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = NULL;
        stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = misc_mnemonic_to_text (stock_item.label);
        break;
      case THREAD_MODE_OPEN:
        gtk_stock_lookup (GTK_STOCK_OPEN, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = NULL;
        stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = misc_mnemonic_to_text (stock_item.label);
        break;
      case THREAD_MODE_SAVE:
        gtk_stock_lookup (GTK_STOCK_SAVE_AS, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = NULL;
        stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = misc_mnemonic_to_text (stock_item.label);
        break;
      case THREAD_MODE_CHANGE:
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = gdk_pixbuf_new_from_xpm_data (change16_xpm);
        stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = g_strdup (_("Change Frames"));
        break;
      case THREAD_MODE_REVERSE:
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = gdk_pixbuf_new_from_xpm_data (reverse16_xpm);
        stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = g_strdup (_("Reverse"));
        break;
      default:
#if GTK_CHECK_VERSION(2,8,0)
        pixbuf_mode = NULL;
        stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
        text_mode = NULL;
    }
  /* ja:ファイル名 */
#if GTK_CHECK_VERSION(2,8,0)
  pixbuf_file = fileio_extract_icon (file, FILEIO_ICON_SIZE_SMALL);
#endif /* GTK_CHECK_VERSION(2,8,0) */
#if GLIB_CHECK_VERSION(2,6,0)
  text_file = file ? g_filename_display_name (file) : NULL;
#else /* not GLIB_CHECK_VERSION(2,6,0) */
  text_file = file ? g_filename_to_utf8 (file, -1, NULL, NULL, NULL) : NULL;
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
  /* ja:時刻 */
  t = time (NULL);
  if (strftime (str, sizeof (str), fmt, localtime (&t)) > 0)
    {
      text_time = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
    }
  else
    {
      gsize length;

      text_time = g_locale_to_utf8 (ctime (&t), -1, NULL, NULL, NULL);
      length = g_strlen (text_time);
      if (text_time[length - 1] == '\n')
        text_time[length - 1] = '\0';
    }
  /* ja:設定 */
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
  gtk_list_store_append (store, &iter);
  gtk_list_store_set (store, &iter,
#if GTK_CHECK_VERSION(2,8,0)
                                        THREAD_TVIEW_MODE_PIXBUF, pixbuf_mode,
                                        THREAD_TVIEW_MODE_STOCK,  stock_id,
#endif /* GTK_CHECK_VERSION(2,8,0) */
                                        THREAD_TVIEW_MODE_TEXT,   text_mode,
#if GTK_CHECK_VERSION(2,8,0)
                                        THREAD_TVIEW_FILE_PIXBUF, pixbuf_file,
#endif /* GTK_CHECK_VERSION(2,8,0) */
                                        THREAD_TVIEW_FILE_TEXT,   text_file,
                                        THREAD_TVIEW_TIME,        text_time,
#if GTK_CHECK_VERSION(2,6,0)
                                        THREAD_TVIEW_STATUS_PROG, 0,
                                        THREAD_TVIEW_STATUS_TEXT, NULL,
#else /* not GTK_CHECK_VERSION(2,6,0) */
                                        THREAD_TVIEW_STATUS_TEXT, "0",
#endif /* not GTK_CHECK_VERSION(2,6,0) */
                                        THREAD_TVIEW_POINTER,     tl, -1);
  gtk_tree_view_columns_autosize (GTK_TREE_VIEW (tview));
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
#if GTK_CHECK_VERSION(2,8,0)
  if (pixbuf_mode)
    g_object_unref (pixbuf_mode);
  if (pixbuf_file)
    g_object_unref (pixbuf_file);
#endif /* GTK_CHECK_VERSION(2,8,0) */
  g_free (text_mode);
  g_free (text_file);
  g_free (text_time);
#ifndef USE_THREAD
  thread_set_menu_bar (FALSE);
  gtk_widget_set_sensitive (mdi, FALSE);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストから削除する
     id,スレッドID
    RET,TRUE:正常終了,FALSE:エラー                                          */
#ifdef USE_THREAD
gboolean
thread_delete (GThread *id)
#else /* not USE_THREAD */
gboolean
thread_delete(gint id)
#endif /* not USE_THREAD */
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  ThreadList *tl;
#ifndef USE_THREAD
  gint id_standard = 0, id_close = 0;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  /* ja:IDに一致するアイテムを探す */
  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
  for (i = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (store), NULL) - 1;
                                                                i >= 0; i--)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store), &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
      if (tl->id == id)
        {
          /* ja:リストにアイテムがあるとき */
          g_free (tl);
          gtk_list_store_remove (store, &iter);
        }
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  /* ja:IDに一致するアイテムを探す */
  for (i = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (store), NULL) - 1;
                                                                i >= 0; i--)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store), &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
      switch (tl->id)
        {
          case THREAD_ID_STANDARD: id_standard++; break;
          case THREAD_ID_CLOSE:    id_close++;    break;
        }
    }
  if (id_standard <= 0)
    {
      thread_set_menu_bar (TRUE);
      gtk_widget_set_sensitive (mdi, TRUE);
    }
  if (id_close > 0)
    gtk_widget_destroy (window);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストを更新する
         id,スレッドID
    percent,進行状況
        RET,TRUE:継続,FALSE:中断                                            */
#ifdef USE_THREAD
gboolean
thread_idling (GThread    *id,
               const gint  percent)
#else /* not USE_THREAD */
gboolean
thread_idling (gint       id,
               const gint percent)
#endif /* not USE_THREAD */
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  ThreadList *tl = NULL;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
  /* ja:IDに一致するアイテムを探す */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
      if (tl->id == id)
        break;
    }
  if (!tl)
    {
#ifdef USE_THREAD
      gdk_threads_leave ();
      critical = FALSE;
      G_UNLOCK (critical);
#endif /* USE_THREAD */
      return FALSE;
    }
  if (tl->userbreak && tl->percent != percent)
    {
#if ! GTK_CHECK_VERSION(2,6,0)
      gchar *text;
#endif /* not GTK_CHECK_VERSION(2,6,0) */

      /* ja:中断ではなくかつ進行状況が異なるとき */
      tl->percent = percent;
#if GTK_CHECK_VERSION(2,6,0)
      if (0 <= percent && percent <= 100)
        gtk_list_store_set (store, &iter,
                                        THREAD_TVIEW_STATUS_TEXT, NULL,
                                        THREAD_TVIEW_STATUS_PROG, percent, -1);
      else
        gtk_list_store_set (store, &iter,
                                        THREAD_TVIEW_STATUS_TEXT, _("Error"),
                                        THREAD_TVIEW_STATUS_PROG, 0, -1);
#else /* not GTK_CHECK_VERSION(2,6,0) */
      text = percent >= 0
                    ? g_strdup_printf ("%d", percent) : g_strdup (_("Error"));
      gtk_list_store_set (store, &iter, THREAD_TVIEW_STATUS_TEXT, text, -1);
      g_free (text);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  while (gtk_events_pending ())
    gtk_main_iteration ();
#endif /* not USE_THREAD */
  return tl->userbreak;
}


/*  ja:スレッドを停止させる
    all,TRUE:すべてのスレッド,FALSE:選択されたスレッドのみ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
thread_break (const gboolean all)
{
  gint i;
  GtkListStore *store;
  GtkTreeIter iter;
  GtkTreeSelection *select;
  ThreadList *tl;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
    if (all || gtk_tree_selection_iter_is_selected (select, &iter))
      {
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
        if (tl->userbreak)
          {
            tl->userbreak = FALSE;
#if GTK_CHECK_VERSION(2,6,0)
            gtk_list_store_set (store, &iter,
                                            THREAD_TVIEW_STATUS_TEXT, "*",
                                            THREAD_TVIEW_STATUS_PROG, 0, -1);
#else /* not GTK_CHECK_VERSION(2,6,0) */
            gtk_list_store_set (store, &iter,
                                            THREAD_TVIEW_STATUS_TEXT, "*", -1);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
          }
      }
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  return TRUE;
}


/*  ja:プログラムの終了(独立スレッド)                                       */
void *
close_program (void)
{
  gboolean result;
  gint i, count, length;
  GtkListStore *store;
  GtkTreeIter iter;
  ThreadList *tl;
#ifdef USE_THREAD
  GThread *id;

  id = g_thread_self ();
#else /* not USE_THREAD */
  gint id = 1;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  thread_insert (id, THREAD_MODE_EXIT, NULL);

  /* ja:既にエラーになっているスレッドを終了させる */
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */

  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
  /* ja:エラーになっているアイテムを探す */
  for (i = 0; gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                        &iter, NULL, i); i++)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
      if (tl->percent < 0 && tl->userbreak) /* ja:リストにアイテムがあるとき */
        {
          /* ja:表示を変更する */
          tl->userbreak = FALSE;
#if GTK_CHECK_VERSION(2,6,0)
          gtk_list_store_set (store, &iter, THREAD_TVIEW_STATUS_TEXT, "*",
                                            THREAD_TVIEW_STATUS_PROG, 0, -1);
#else /* not GTK_CHECK_VERSION(2,6,0) */
          gtk_list_store_set (store, &iter, THREAD_TVIEW_STATUS_TEXT, "*", -1);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
        }
    }

#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);

  do
    {
      g_thread_yield ();
      G_LOCK (critical);
      critical = TRUE;
      gdk_threads_enter ();
#endif /* USE_THREAD */

      /* ja:終了スレッドに一致するアイテムを探す */
      result = FALSE;
      count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (store), NULL);
      for (i = count - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store), &iter,
                                                                    NULL, i);
          gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
          if (tl->mode == THREAD_MODE_EXIT)
            {
              /* ja:リストにアイテムがあるとき */
              result = tl->id != id;
              break;
            }
        }
      /* ja:MDIの子ウインドウ数 */
      length = orz_mdi_get_n_pages (ORZ_MDI (mdi));

#ifdef USE_THREAD
      gdk_threads_leave ();
      critical = FALSE;
      G_UNLOCK (critical);

      /* ja:リストに終了があり、このスレッド以外ならばこのスレッドは破棄 */
      if (result || !thread_idling (id, 0))
        {
          thread_delete (id);
          return NULL;
        }
    }
  while (count > 1 || length > 0);
  thread_delete (id);
  gdk_threads_enter ();
  gtk_widget_destroy (window);
  gdk_threads_leave ();
#else /* not USE_THREAD */
  /* ja:終了する */
  if (length <= 0)
    {
      if (count <= 0)
        gtk_widget_destroy (window);
      else if (i < 0)
        thread_insert (THREAD_ID_CLOSE, THREAD_MODE_EXIT, NULL);
    }
#endif /* USE_THREAD */
  return NULL;
}
