/*
    avicore
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "icm.h"
#include "avifmt.h"
#include <gdk/gdkkeysyms.h>
#include "misc/fileio.h"
#ifdef W32CODECDIR
# include "misc/profile.h"
#endif /* W32CODECDIR */


#ifdef G_OS_WIN32
# define icm_send_driver_message(icm_object,msg,lParam1,lParam2)            \
    SendDriverMessage(((IcmObject *)(icm_object))->hDrvr,msg,lParam1,lParam2)
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
# define icm_send_driver_message(icm_object,msg,lParam1,lParam2)            \
            ((IcmObject *)(icm_object))->icm_driver->DriverProc             \
            (((IcmObject *)(icm_object))->dwDriverId,                       \
            ((IcmObject *)(icm_object))->icm_driver->hLib,msg,lParam1,lParam2)
#endif /* W32CODECDIR */


/******************************************************************************
*                                                                             *
* ja:ICM初期化                                                                *
*                                                                             *
******************************************************************************/
static GHashTable *ghash_handler = NULL, *ghash_path = NULL;


#if defined (G_OS_WIN32) || defined (W32CODECDIR)
static guint32
icm_handler_tolower (const guint32 key)
{
  gint i;
  guint32 a = 0;

  for (i = 24; i >= 0; i -= 8)
    {
      a <<= 8;
      a |= (guchar)g_ascii_tolower ((gchar)(key >> i));
    }
  return a;
}


/*static guint32
icm_handler_toupper (const guint32 key)
{
  gint i;
  guint32 a = 0;

  for (i = 24; i >= 0; i -= 8)
    {
      a <<= 8;
      a |= (guchar)g_ascii_toupper ((gchar)(key >> i));
    }
  return a;
}*/


static guint
icm_hash_handler (gconstpointer key)
{
  return icm_handler_tolower ((const guint32)key);
}


static gboolean
icm_equal_handler (gconstpointer a,
                   gconstpointer b)
{
  return icm_handler_tolower ((const guint32)a)
                                    == icm_handler_tolower ((const guint32)b);
}


static guint
icm_hash_path (gconstpointer key)
{
  const gchar *p;
  guint h = 0;

  for (p = key; *p != '\0'; p++)
# ifdef G_OS_WIN32
    h = (h << 5) - h + g_ascii_tolower (*p);
# else /* not G_OS_WIN32 */
    h = (h << 5) - h + *p;
# endif /* not G_OS_WIN32 */

  return h;
}


static gboolean
icm_equal_path (gconstpointer a,
                gconstpointer b)
{
  return g_strfilecmp (a, b) == 0;
}


static void
icm_init_callback (gpointer key,
                   gpointer value,
                   gpointer ghash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable *)ghash, key,
                                                    &orig_key, &orig_value))
    {
      gchar *path;
      guint32 handler;
      IcmDriver *icm_driver;

      handler = (guint32)orig_key;
      path = orig_value;
      icm_driver = g_hash_table_lookup (ghash_path, path);
      if (!icm_driver)
        {
          /* ja:新規ロード */
# ifdef G_OS_WIN32
          icm_driver = g_malloc0 (sizeof (IcmDriver));
          if (icm_driver)
            {
              icm_driver->lpszName = g_utf8_to_utf16 (path, -1,
                                                            NULL, NULL, NULL);
              g_hash_table_insert (ghash_path, path, icm_driver);
            }
          else
            {
              g_free (path);
            }
# else /* not G_OS_WIN32 */
          if ((icm_driver = g_malloc0 (sizeof (IcmDriver)))
                    && (icm_driver->hLib = peimage_create())
                    && peimage_load (icm_driver->hLib, path)
                    && (icm_driver->DriverProc
                        = peimage_resolve (icm_driver->hLib, "DriverProc")))
            {
              icm_driver->DriverProc (0, icm_driver->hLib, DRV_LOAD, 0, 0);
              icm_driver->DriverProc (0, icm_driver->hLib, DRV_ENABLE, 0, 0);
              icm_driver->lOpenID = icm_driver->DriverProc
                                (0, icm_driver->hLib, DRV_OPEN, (LONG)L"", 0);
              g_hash_table_insert (ghash_path, path, icm_driver);
            }
          else
            {
              if (icm_driver->hLib)
                peimage_destroy (icm_driver->hLib);
              g_free (icm_driver);
              g_free (path);
              icm_driver = NULL;
            }
# endif /* not G_OS_WIN32 */
        }
      if (icm_driver)   /* ja:登録 */
        g_hash_table_insert (ghash_handler, (gpointer)handler, icm_driver);
    }
}
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */


/*  ja:ICMを初期化する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_init (void)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  GHashTable *ghash;
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
#ifdef G_OS_WIN32
  DWORD dwIndex, dwName, dwType, dwValue;
  HKEY hKey;
  TCHAR szName[MAX_PATH], szValue[MAX_PATH];
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
  Profile *profile;
#endif /* W32CODECDIR */

  if (ghash_handler || ghash_path)
    return FALSE;
  ghash = g_hash_table_new (NULL, NULL);
#ifdef G_OS_WIN32
  if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
        _T("Software\\Microsoft\\Windows NT\\CurrentVersion\\Drivers32"), 0,
                                    KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS)
    return FALSE;
  for (dwIndex = 0; dwName = MAX_PATH, dwValue = MAX_PATH,
            RegEnumValue (hKey, dwIndex, szName, &dwName, NULL,
            &dwType, (LPBYTE)szValue, &dwValue) == ERROR_SUCCESS; dwIndex++)
    if (dwType == REG_SZ && (TCHAR)CharUpper ((LPTSTR)szName[0]) == 'V'
                         && (TCHAR)CharUpper ((LPTSTR)szName[1]) == 'I'
                         && (TCHAR)CharUpper ((LPTSTR)szName[2]) == 'D'
                         && (TCHAR)CharUpper ((LPTSTR)szName[3]) == 'C'
                         && (TCHAR)CharUpper ((LPTSTR)szName[4]) == '.')
      {
        gchar *key, *value;
        guint32 handler;

# ifdef UNICODE
        key = g_utf8_from_utf16 (szName + 5, -1, NULL, NULL, NULL);
        value = g_utf8_from_utf16 (szName, -1, NULL, NULL, NULL);
# else /* not UNICODE */
        key = g_locale_to_utf8 (szName + 5, -1, NULL, NULL, NULL);
        value = g_locale_to_utf8 (szName, -1, NULL, NULL, NULL);
# endif /* not UNICODE */
        if (key && value && (handler
                    = icm_handler_tolower (icm_string_to_handler (key))) != 0)
          g_hash_table_insert (ghash, (gpointer)handler, value);
        g_free (key);
      }
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
  /* ja:設定 */
  profile = profile_open (NULL);
  if (profile)
    {
      GList *glist;

      glist = profile_enum_key (profile, "vidc");
      if (glist)
        {
          gint i, length;

          length = g_list_length (glist);
          for (i = 0; i < length; i++)
            {
              gchar *key, *value;
              guint32 handler;

              key = g_list_nth_data (glist, i);
              if ((handler
                    = icm_handler_tolower (icm_string_to_handler (key))) != 0
                        && (value = profile_get_string (profile, "vidc", key)))
                {
                  gchar *path;

                  path = g_hash_table_lookup (ghash, (gconstpointer)handler);
                  g_free (path);
                  if (value[0] == '\0'
                            || value[g_strlen(value) - 1] == G_DIR_SEPARATOR)
                    {
                      if (path)
                        g_hash_table_remove (ghash, (gconstpointer)handler);
                    }
                  else
                    {
                      gchar *tmp;

                      tmp = g_filename_from_utf8 (value, -1, NULL, NULL, NULL);
                      if (g_path_is_absolute (tmp))
                        {
                          path = tmp;
                        }
                      else
                        {
                          path = g_strconcat (W32CODECDIR,
                                                G_DIR_SEPARATOR_S, tmp, NULL);
                          g_free (tmp);
                        }
                      tmp = path;
                      path = fileio_get_full_path (path);
                      g_free (tmp);
                      g_hash_table_insert (ghash, (gpointer)handler, path);
                    }
                  g_free (value);
                }
            }
          g_list_free (glist);
        }
      profile_close (profile);
    }
#endif /* W32CODECDIR */
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  ghash_handler = g_hash_table_new (icm_hash_handler, icm_equal_handler);
  ghash_path = g_hash_table_new (icm_hash_path, icm_equal_path);
  g_hash_table_foreach (ghash, icm_init_callback, ghash);
  g_hash_table_destroy (ghash);
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
#ifdef G_OS_WIN32
  return RegCloseKey (hKey) == ERROR_SUCCESS;
#else /* not G_OS_WIN32 */
  return TRUE;
#endif /* not G_OS_WIN32 */
}


static void
icm_exit_callback (gpointer key,
                   gpointer value,
                   gpointer ghash)
{
  gpointer orig_key, orig_value;

  if (g_hash_table_lookup_extended ((GHashTable *)ghash, key,
                                                    &orig_key, &orig_value))
    {
      IcmDriver *icm_driver;

      icm_driver = orig_value;
#ifdef G_OS_WIN32
      g_free (icm_driver->lpszName);
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
      icm_driver->DriverProc (icm_driver->lOpenID,
                                            icm_driver->hLib, DRV_CLOSE, 0, 0);
      icm_driver->DriverProc (0, icm_driver->hLib, DRV_DISABLE, 0, 0);
      icm_driver->DriverProc (0, icm_driver->hLib, DRV_FREE, 0, 0);
      peimage_destroy (icm_driver->hLib);
#endif /* W32CODECDIR */
      g_free (icm_driver);
      g_free (orig_key);
    }
}


/*  ja:ICMを終了する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
icm_exit (void)
{
  if (ghash_handler || ghash_path)
    {
      g_hash_table_foreach (ghash_path, icm_exit_callback, ghash_path);
      g_hash_table_destroy (ghash_handler);
      g_hash_table_destroy (ghash_path);
      ghash_handler = NULL;
      ghash_path = NULL;
    }
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:ICM4文字コード処理                                                       *
*                                                                             *
******************************************************************************/
/*  ja:4文字コード→文字列
    handler,4文字コード
        RET,文字列                                                          */
gchar *
icm_handler_to_string (const guint32 handler)
{
  gchar c, *text = NULL, *temp0, *temp1;
  gint i;

  for (i = 24; i >= 0; i -= 8)
    {
      c = (handler >> i) & 0xff;
      temp0 = g_strdup_printf (g_ascii_isalnum (c) ? "%c" : "%%%02x",
                                                                    (guint8)c);
      temp1 = g_strconcat (temp0, text, NULL);
      g_free (temp0);
      g_free (text);
      text = temp1;
    }
  return text;
}


/*  ja:文字列→4文字コード
    text,文字列
     RET,4文字コード                                                        */
guint32
icm_string_to_handler (const gchar *text)
{
  gint i, j = 0, k;
  guint32 handler = 0;

  for (i = 0; i < 4; i++)
    {
      handler >>= 8;
      if (text[j] == '%')
        {
          j++;
          for (k = 28; k >= 24; k -= 4)
            {
              if ('0' <= text[j] && text[j] <= '9')
                handler |= (text[j] - '0') << k;
              else if ('A' <= text[j] && text[j] <= 'A')
                handler |= (text[j] - 'A') << k;
              else if ('a' <= text[j] && text[j] <= 'f')
                handler |= (text[j] - 'a')<<k;
              else
                break;
              j++;
            }
          if (k>=24)
            return 0;
        }
      else if (text[j] != '\0')
        {
          handler |= text[j++] << 24;
        }
      else
        {
          return 0;
        }
    }
  return handler;
}


/******************************************************************************
*                                                                             *
* ja:ICM基本                                                                  *
*                                                                             *
******************************************************************************/
static void
icm_get_handler_list_callback (gpointer   key,
                               gpointer   value,
                               GList    **glist)
{
  *glist = g_list_append (*glist, key);
}


/*  ja:登録されている4文字コードを取得する
    RET,4文字コードのリスト,NULL:登録なし                                   */
GList *
icm_get_handler_list (void)
{
  GList *glist = NULL;

  if (ghash_handler)
    g_hash_table_foreach (ghash_handler,
                                (GHFunc)icm_get_handler_list_callback, &glist);
  return glist;
}


/*  ja:Codecを開く
    handler,4文字コード
       mode,モード
        RET,IcmObject,NULL:エラー                                           */
IcmObject *
icm_open (const guint32 handler, const gint mode)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  IcmObject *icm_object;
  ICOPEN icopen;

  icm_object = g_malloc0 (sizeof (IcmObject));
  icm_object->counter = 1;
  icm_object->handler = handler;
  icm_object->mode = mode;
  icm_object->icm_driver = g_hash_table_lookup (ghash_handler,
                                                    (gconstpointer)handler);
  if (icm_object->icm_driver)
    {
      g_memset (&icopen, 0, sizeof (ICOPEN));
      icopen.dwSize = sizeof (ICOPEN);
      icopen.fccType = ICTYPE_VIDEO;
      icopen.fccHandler = handler;
      switch (mode)
        {
          case ICM_MODE_COMPRESS:
            icopen.dwFlags = ICMODE_COMPRESS;
            break;
          case ICM_MODE_DECOMPRESS:
            icopen.dwFlags = ICMODE_DECOMPRESS;
            break;
          case ICM_MODE_FASTDECOMPRESS:
            icopen.dwFlags = ICMODE_FASTDECOMPRESS;
            break;
          case ICM_MODE_QUERY:
            icopen.dwFlags = ICMODE_QUERY;
            break;
          case ICM_MODE_FASTCOMPRESS:
            icopen.dwFlags = ICMODE_FASTCOMPRESS;
            break;
          case ICM_MODE_DRAW:
            icopen.dwFlags = ICMODE_DRAW;
        }
# ifdef G_OS_WIN32
      icm_object->hDrvr = OpenDriver (icm_object->icm_driver->lpszName, NULL,
                                                                (LONG)&icopen);
    }
  if (!icm_object->hDrvr)
# else /* not G_OS_WIN32 */
      icm_object->dwDriverId = icm_object->icm_driver->DriverProc
                (0, icm_object->icm_driver->hLib, DRV_OPEN, 0, (LONG)&icopen);
    }
  if (icm_object->dwDriverId == 0)
# endif /* not G_OS_WIN32 */
    {
      g_free (icm_object);
      icm_object = NULL;
    }
  return icm_object;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return NULL;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:Codecを閉じる
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_close (IcmObject *icm_object)
{
  if (icm_object && --(icm_object->counter) <= 0)
    {
#ifdef G_OS_WIN32
      CloseDriver (icm_object->hDrvr, 0, 0);
#endif /* G_OS_WIN32 */
#ifdef W32CODECDIR
      icm_object->icm_driver->DriverProc (icm_object->dwDriverId,
                                icm_object->icm_driver->hLib, DRV_CLOSE, 0, 0);
#endif /* W32CODECDIR */
      g_free (icm_object->bmih_in);
      g_free (icm_object->bmih_out);
      g_free (icm_object->prev);
      g_free (icm_object);
    }
  return TRUE;
}


/*  ja:Codecの情報を取得する
    icm_object,IcmObject
      icm_info,情報
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_get_info (IcmObject *icm_object,
              IcmInfo   *icm_info)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  ICINFO icinfo;

  if (!icm_object)
    return FALSE;
  g_memset (&icinfo, 0, sizeof (ICINFO));
  icinfo.dwSize = sizeof (ICINFO);
  icm_send_driver_message (icm_object, ICM_GETINFO,
                                            (LONG)&icinfo, sizeof (ICINFO));
  icm_info->name = g_utf16_to_utf8 (icinfo.szDescription, -1,
                                                            NULL, NULL, NULL);
  icm_info->handler = icm_get_handler (icm_object); /* icinfo.fccHandler */
  icm_info->flags = 0;
  if (icinfo.dwFlags & VIDCF_QUALITY)
    icm_info->flags = ICM_FLAG_QUALITY;
  if (icinfo.dwFlags & VIDCF_CRUNCH)
    icm_info->flags |= ICM_FLAG_CRUNCH;
  if (icinfo.dwFlags & VIDCF_TEMPORAL)
    icm_info->flags |= ICM_FLAG_TEMPORAL;
  if (icinfo.dwFlags & VIDCF_COMPRESSFRAMES)
    icm_info->flags |= ICM_FLAG_COMPRESSFRAMES;
  if (icinfo.dwFlags & VIDCF_DRAW)
    icm_info->flags |= ICM_FLAG_DRAW;
  if (icinfo.dwFlags & VIDCF_FASTTEMPORALC)
    icm_info->flags |= ICM_FLAG_FASTTEMPORALC;
  if (icinfo.dwFlags & VIDCF_FASTTEMPORALD)
    icm_info->flags |= ICM_FLAG_FASTTEMPORALD;
# ifdef W32CODECDIR
  if (icinfo.dwFlags & VIDCF_QUALITYTIME)
    icm_info->flags |= ICM_FLAG_QUALITYTIME;
# endif /* W32CODECDIR */
  icm_info->attribute = ICM_ATTR_WIN32;
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:パラメータのサイズを取得する
    icm_object,IcmObject
           RET,サイズ                                                       */
gsize
icm_get_state_size (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object
                ? icm_send_driver_message (icm_object, ICM_GETSTATE, 0, 0) : 0;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:パラメータを取得する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_get_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && param && param_size > 0
                        && icm_send_driver_message (icm_object, ICM_GETSTATE,
                                        (LPARAM)param, param_size) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:パラメータを設定する
    icm_object,IcmObject
         param,パラメータ
    param_size,パラメータのサイズ
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_set_state (IcmObject *icm_object,
               gpointer   param,
               gsize      param_size)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (icm_object && param && param_size > 0)
    {
      icm_send_driver_message (icm_object, ICM_SETSTATE,
                                                    (LPARAM)param, param_size);
      return TRUE;
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
}


/******************************************************************************
*                                                                             *
* ja:ICMダイアログ                                                            *
*                                                                             *
******************************************************************************/
/*  ja:Aboutダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_about (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && icm_send_driver_message (icm_object,
                                                ICM_ABOUT, -1, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:Aboutダイアログを表示する
    icm_object,IcmObject                                                    */
void
icm_dialog_about (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (icm_object)
    icm_send_driver_message (icm_object, ICM_ABOUT, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:設定ダイアログのサポートを確認する
    icm_object,IcmObject
           RET,TRUE:有効,FALSE:無効                                         */
gboolean
icm_is_dialog_configure (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && icm_send_driver_message (icm_object,
                                            ICM_CONFIGURE, -1, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:設定ダイアログを表示する
    icm_object,IcmObject                                                    */
void
icm_dialog_configure (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (icm_object)
    icm_send_driver_message (icm_object, ICM_CONFIGURE, 0, 0);
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/******************************************************************************
*                                                                             *
* en:ICM Compress                                                             *
*                                                                             *
******************************************************************************/
/*  ja:圧縮可能か確認する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_compress_query (IcmObject              *icm_object,
                    const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && bmih && icm_send_driver_message (icm_object,
                                ICM_COMPRESS_QUERY, (LONG)bmih, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮のデフォルトの周期を取得する
    icm_object,IcmObject
           RET,デフォルト(0:未対応)                                         */
gint
icm_compress_get_default_key_frame (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  IcmInfo icm_info;

  if (icm_object && icm_get_info (icm_object, &icm_info))
    {
      g_free (icm_info.name);
      if ((icm_info.flags & ICM_FLAG_TEMPORAL) != 0)
        {
          DWORD dwKeyFrame;

          if (icm_send_driver_message (icm_object,
                ICM_GETDEFAULTKEYFRAMERATE, (LONG)&dwKeyFrame, 0) == ICERR_OK)
            return dwKeyFrame;
        }
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
}


/*  ja:圧縮のデフォルトの品質を取得する
    icm_object,IcmObject
           RET,デフォルト(-1:未対応)                                        */
gint
icm_compress_get_default_quality (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  IcmInfo icm_info;

  if (icm_object && icm_get_info (icm_object, &icm_info)
                                && (icm_info.flags & ICM_FLAG_QUALITY) != 0)
    {
      g_free (icm_info.name);
      if ((icm_info.flags & ICM_FLAG_QUALITY) != 0)
        {
          DWORD dwQuality;

          if (icm_send_driver_message (icm_object,
                    ICM_GETDEFAULTQUALITY, (LONG)&dwQuality, 0) == ICERR_OK)
            return dwQuality;
        }
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return -1;
}


/*  ja:圧縮後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮するビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_format_size (IcmObject              *icm_object,
                              const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (!icm_object || !bmih)
    return 0;
  return icm_send_driver_message (icm_object,
                                    ICM_COMPRESS_GET_FORMAT, (LONG)bmih, 0);
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_get_format (IcmObject              *icm_object,
                         const BitmapInfoHeader *bmih_in,
                         BitmapInfoHeader       *bmih_out)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && bmih_in && bmih_out
        && icm_send_driver_message (icm_object,
        ICM_COMPRESS_GET_FORMAT, (LONG)bmih_in, (LONG)bmih_out) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮後のデータの最大サイズを取得する
    icm_object,IcmObject
           RET,バイト数,0:エラー                                            */
gsize
icm_compress_get_size (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && icm_object->bmih_in && icm_object->bmih_out
                ? icm_send_driver_message (icm_object, ICM_COMPRESS_GET_SIZE,
                    (LONG)icm_object->bmih_in, (LONG)icm_object->bmih_out) : 0;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮を設定する
    icm_object,IcmObject
       bmih_in,圧縮するビットマップのヘッタ
      bmih_out,圧縮されたビットマップのヘッタ
     bmih_size,圧縮されたビットマップのヘッタのサイズ
     key_frame,周期(1:すべてキーフレーム,2～:周期,0:デフォルト)
       quality,品質0～10000,-1:デフォルト
     data_rate,1秒間のサイズ(0:無効)
          rate,レート
         scale,スケール
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_frames_info (IcmObject              *icm_object,
                          const BitmapInfoHeader *bmih_in,
                          const BitmapInfoHeader *bmih_out,
                          const gsize             bmih_size,
                          const gint              key_frame,
                          const gint              quality,
                          const gsize             data_rate,
                          const guint32           rate,
                          const guint32           scale)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  IcmInfo icm_info;

  if (!icm_object || !icm_get_info (icm_object, &icm_info))
    return FALSE;
  g_free (icm_info.name);
  if (bmih_in)
    {
      g_free (icm_object->bmih_in);
      icm_object->bmih_in = g_memdup (bmih_in, bm_header_bytes (bmih_in));
    }
  if (!icm_object->bmih_in)
    return FALSE;
  if (bmih_out && bmih_size > 0)
    {
      g_free (icm_object->bmih_out);
      icm_object->bmih_out = g_memdup (bmih_out, bmih_size);
    }
  else if (!icm_object->bmih_out)
    {
      icm_object->bmih_out
            = g_malloc (icm_compress_get_format_size (icm_object, bmih_in));
      if (!icm_compress_get_format (icm_object, bmih_in, icm_object->bmih_out))
        {
          g_free (icm_object->bmih_out);
          icm_object->bmih_out = NULL;
          return FALSE;
        }
    }
  if (0 <= quality && quality <= 10000)
    icm_object->quality = quality;
  else if (quality == -1)
    icm_object->quality = icm_compress_get_default_quality (icm_object);
  if (icm_object->quality < 0 && 10000 < icm_object->quality)
    icm_object->quality = 0;
  icm_object->key_frame = key_frame >= 1 ? key_frame
                    : MAX (icm_compress_get_default_key_frame (icm_object), 1);
  if (icm_info.flags & ICM_FLAG_COMPRESSFRAMES)
    {
      ICCOMPRESSFRAMES iccompressframes;

      g_memset (&iccompressframes, 0, sizeof (ICCOMPRESSFRAMES));
      (BitmapInfoHeader *)iccompressframes.lpbiOutput = icm_object->bmih_out;
      (BitmapInfoHeader *)iccompressframes.lpbiInput = icm_object->bmih_in;
      iccompressframes.lQuality = icm_object->quality;
      iccompressframes.lDataRate = data_rate > 0 ? data_rate : G_MAXINT;
      iccompressframes.lKeyRate = icm_object->key_frame;
      iccompressframes.dwRate = MAX (rate, 1);
      iccompressframes.dwScale = MAX (scale, 1);
      icm_send_driver_message (icm_object, ICM_COMPRESS_FRAMES_INFO,
                        (LONG)&iccompressframes, sizeof (ICCOMPRESSFRAMES));
    }
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮を開始する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_begin (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (!icm_object || !icm_object->bmih_in || icm_object->prev)
    return FALSE;
  if (!icm_object->bmih_out)
    {
      icm_object->bmih_out = g_malloc (icm_compress_get_format_size
                                            (icm_object, icm_object->bmih_in));
      if (!icm_compress_get_format (icm_object,
                                    icm_object->bmih_in, icm_object->bmih_out))
        {
          g_free (icm_object->bmih_out);
          icm_object->bmih_out = NULL;
          return FALSE;
        }
    }
  if (icm_send_driver_message (icm_object, ICM_COMPRESS_BEGIN,
            (LONG)icm_object->bmih_in, (LONG)icm_object->bmih_out) == ICERR_OK
                && (icm_object->size = icm_compress_get_size (icm_object)) > 0)
    {
      icm_object->frame = 0;
      icm_object->key_point = 0;
      icm_object->prev_size = bm_image_bytes (icm_object->bmih_in);
      icm_object->prev = g_malloc (icm_object->prev_size);
      return TRUE;
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
}


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_compress_end (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (!icm_object || !icm_object->prev)
    return FALSE;
  g_free (icm_object->prev);
  icm_object->prev = NULL;
  return icm_send_driver_message (icm_object,
                                        ICM_COMPRESS_END, 0, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:圧縮する
    icm_object,IcmObject
     key_frame,TRUE:キーフレーム,FALSE:非キーフレーム
         input,圧縮するデータ
        output,圧縮されたデータ
           RET,フレームのサイズ,0:エラー                                    */
gsize
icm_compress (IcmObject *icm_object,
              gboolean  *key_frame,
              gpointer   input,
              gpointer   output)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  DWORD dwFlags,dwID;
  ICCOMPRESS iccompress;

  if (!icm_object || !input || !output
                            || !icm_object->bmih_in || !icm_object->bmih_out
                            || !icm_object->prev || icm_object->prev_size <= 0)
    return 0;
  iccompress.dwFlags = icm_object->key_point % icm_object->key_frame == 0
                                                    ? ICCOMPRESS_KEYFRAME : 0;
  (BitmapInfoHeader *)iccompress.lpbiOutput = icm_object->bmih_out;
  iccompress.lpOutput = output;
  (BitmapInfoHeader *)iccompress.lpbiInput = icm_object->bmih_in;
  iccompress.lpInput = input;
  iccompress.lpckid = &dwID;
  iccompress.lpdwFlags = &dwFlags;
  iccompress.lFrameNum = icm_object->frame;
  iccompress.dwFrameSize = icm_object->size;
  iccompress.dwQuality = icm_object->quality;
  if (icm_object->frame > 0)
    {
      (BitmapInfoHeader *)iccompress.lpbiPrev = icm_object->bmih_in;
      iccompress.lpPrev = icm_object->prev;
    }
  else
    {
      iccompress.lpbiPrev = NULL;
      iccompress.lpPrev = NULL;
    }
  if (icm_send_driver_message (icm_object,
            ICM_COMPRESS, (LONG)&iccompress, sizeof (ICCOMPRESS)) == ICERR_OK
            && bmih_get_size_image (icm_object->bmih_out) <= icm_object->size)
    {
      icm_object->frame++;
      if ((dwFlags & AVI_IF_KEYFRAME) != 0)
        icm_object->key_point = 1;
      else
        icm_object->key_point++;
      g_memmove (icm_object->prev, input, icm_object->prev_size);
      if (key_frame)
        *key_frame = (dwFlags & AVI_IF_KEYFRAME) != 0;
      return bmih_get_size_image (icm_object->bmih_out);
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
}


/******************************************************************************
*                                                                             *
* en:ICM Decompress                                                           *
*                                                                             *
******************************************************************************/
/*  ja:展開可能か確認する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:可能,FALSE:不可能                                       */
gboolean
icm_decompress_query (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  return icm_object && bmih
                && icm_send_driver_message (icm_object,
                            ICM_DECOMPRESS_QUERY, (LONG)bmih, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:展開後のフォーマットのサイズを取得する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,バイト数,0:エラー                                            */
gsize
icm_decompress_get_format_size (IcmObject              *icm_object,
                                const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  gsize bmih_size = 0;

  if (icm_object && bmih)
    {
      bmih_size = icm_send_driver_message (icm_object,
                                    ICM_DECOMPRESS_GET_FORMAT, (LONG)bmih, 0);
      if (bmih_size > 0)
        {
          BitmapInfoHeader *bmih_out;

          bmih_out = g_malloc (bmih_size);
          if (icm_send_driver_message (icm_object, ICM_DECOMPRESS_GET_FORMAT,
                                    (LONG)bmih, (LONG)bmih_out) == ICERR_OK)
            {
              if (bmih_size == bmih_get_size (bmih_out)
                        && bm_color_bytes (bmih) == bm_color_bytes (bmih_out))
                bmih_size += bm_color_bytes (bmih_out);
            }
          else
            {
              bmih_size = 0;
            }
          g_free (bmih_out);
        }
    }
  return bmih_size;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return 0;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:展開後のフォーマットを取得する
    icm_object,IcmObject
       bmih_in,圧縮されたビットマップのヘッタ
      bmih_out,展開されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_get_format (IcmObject              *icm_object,
                           const BitmapInfoHeader *bmih_in,
                           BitmapInfoHeader       *bmih_out)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (icm_object && bmih_in && bmih_out
        && icm_send_driver_message (icm_object,
        ICM_DECOMPRESS_GET_FORMAT, (LONG)bmih_in, (LONG)bmih_out) == ICERR_OK)
    {
      gsize bmih_size;

      bmih_size = icm_send_driver_message (icm_object,
                                ICM_DECOMPRESS_GET_FORMAT, (LONG)bmih_in, 0);
      if (bmih_size > 0)
        {
          if (bmih_size == bmih_get_size (bmih_out)
                    && bm_color_bytes (bmih_in) == bm_color_bytes (bmih_out))
            g_memmove ((guint8 *)bmih_out + bmih_get_size (bmih_out),
                       (guint8 *)bmih_in  + bmih_get_size (bmih_in),
                       bm_color_bytes (bmih_out));
          return TRUE;
        }
    }
#endif /* defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
}


/*  ja:展開を開始する
    icm_object,IcmObject
          bmih,圧縮されたビットマップのヘッタ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_begin (IcmObject              *icm_object,
                      const BitmapInfoHeader *bmih)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (!icm_object || !bmih || icm_object->bmih_in || icm_object->bmih_out)
    return FALSE;
  icm_object->bmih_out
                = g_malloc (icm_decompress_get_format_size (icm_object, bmih));
  if (!icm_decompress_get_format (icm_object,bmih, icm_object->bmih_out)
                || icm_send_driver_message (icm_object,
                        ICM_DECOMPRESS_BEGIN,
                        (LONG)bmih, (LONG)icm_object->bmih_out) != ICERR_OK)
    {
      g_free (icm_object->bmih_out);
      icm_object->bmih_out = NULL;
      return FALSE;
    }
  icm_object->bmih_in = g_memdup (bmih, bm_header_bytes (bmih));
  return TRUE;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:展開を終了する
    icm_object,IcmObject
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress_end (IcmObject *icm_object)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  if (!icm_object || (!icm_object->bmih_in && !icm_object->bmih_out))
    return FALSE;
  g_free (icm_object->bmih_in);
  g_free (icm_object->bmih_out);
  icm_object->bmih_in = NULL;
  icm_object->bmih_out = NULL;
  return icm_send_driver_message (icm_object,
                                        ICM_DECOMPRESS_END, 0, 0) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}


/*  ja:展開する
    icm_object,IcmObject
         flags,フラグ
         input,圧縮されたデータ
        output,展開されたデータ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
icm_decompress (IcmObject   *icm_object,
                const guint  flags,
                gpointer     input,
                gpointer     output)
{
#if defined (G_OS_WIN32) || defined (W32CODECDIR)
  ICDECOMPRESS icdecompress;

  if (!icm_object || !input || !output
                            || !icm_object->bmih_in || !icm_object->bmih_out)
    return FALSE;
  icdecompress.dwFlags = 0;
  if (flags & ICM_DECOMPRESS_HURRYUP)
    icdecompress.dwFlags |= ICDECOMPRESS_HURRYUP;
  if (flags & ICM_DECOMPRESS_UPDATE)
    icdecompress.dwFlags |= ICDECOMPRESS_UPDATE;
  if (flags & ICM_DECOMPRESS_PREROLL)
    icdecompress.dwFlags |= ICDECOMPRESS_PREROLL;
  if (flags & ICM_DECOMPRESS_NULLFRAME)
    icdecompress.dwFlags |= ICDECOMPRESS_NULLFRAME;
  if (flags & ICM_DECOMPRESS_NOTKEYFRAME)
    icdecompress.dwFlags |= ICDECOMPRESS_NOTKEYFRAME;
  (BitmapInfoHeader *)icdecompress.lpbiInput = icm_object->bmih_in;
  icdecompress.lpInput = input;
  (BitmapInfoHeader *)icdecompress.lpbiOutput = icm_object->bmih_out;
  icdecompress.lpOutput = output;
  icdecompress.ckid = 0;
  return icm_send_driver_message (icm_object, ICM_DECOMPRESS,
                    (LONG)&icdecompress, sizeof (ICDECOMPRESS)) == ICERR_OK;
#else /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
  return FALSE;
#endif /* not defined (G_OS_WIN32) || defined (W32CODECDIR) */
}
