/*
 * Asprintf
 *
 * Copyright (c) 2014 Akira Sasaki
 *
 * This software is released under the MIT License.
 *
 * http://opensource.org/licenses/mit-license.php
 */

/**
 * @mainpage
 *
 * @section manpage Asprintf manpage
 *
 * <h2>NAME</h2>
 * @asprintf, @vasprintf --- print to allocated string
 *
 * <h2>SYNOPSIS</h2>
 * <code><pre>
 * #include "asprintf.h"
 *
 * int %asprintf(char **strp, const char *format, ...);
 * int %vasprintf(char **strp, const char *format, va_list ap);
 * </pre></code>
 *
 * <h2>MOTIVATION</h2>
 * From @asprintf(3) in Linux Manual Pages:
 *
 * <blockquote>
 * If memory allocation wasn't possible, or some other error occurs, these
 * functions will return @c -1, and the contents of @c strp is undefined.
 * </blockquote>
 *
 * Therefore, when @asprintf returns @c -1, no one knows whether @c *strp
 * should be passed to @free or not.
 *
 * <h2>DESCRIPTION</h2>
 * Same as in Linux Manual Pages.
 *
 * <h2>RETURN VALUE</h2>
 * When successful, same as in Linux Manual Pages.
 *
 * If memory allocation wasn't possible, or some other error occurs, these
 * functions will return @c -1 and set @c *strp to be @c NULL, and no memory
 * is allocated.
 *
 * If an error occurs after memory is allocated successfully, these function
 * calls @free for the allocated memory.
 *
 * <h2>SEE ALSO</h2>
 * @malloc(3), @printf(3), @stdarg(3)
 *
 * <h2>BUGS</h2>
 * The current implementation calls @vsnprintf twice; first call is for
 * getting size of memory allocation, and second call is for getting output
 * string.
 * When a return value of the second @vsnprintf call is different from the
 * first, the current implementation considers that some error occurred.
 *
 * @section license License
 * Asprintf
 *
 * Copyright (c) 2014 Akira Sasaki
 *
 * This software is released under the MIT License.
 *
 * %http://opensource.org/licenses/mit-license.php
 */

/**
 * @file
 * @brief Header file for @asprintf and @vasprintf.
 *
 * This is a header file for @asprintf and @vasprintf.
 *
 * The function @asprintf and @vasprintf are similar to @sprintf and
 * @vsprintf respectively, except that they allocate memory to store
 * output string, and return a pointer to it via the first parameter.
 */

#ifndef ASPRINTF_H
#define ASPRINTF_H

#include <stdio.h>
#include <stdarg.h>

#ifdef UNIT_TEST
int (*mockable_vsnprintf1)(
    char * restrict str, size_t size,
    const char * restrict format, va_list ap);
int (*mockable_vsnprintf2)(
    char * restrict str, size_t size,
    const char * restrict format, va_list ap);
void *(*mockable_malloc)(size_t size);
void (*mockable_free)(void *ptr);
#endif

/**
 * @brief Print to allocated string.
 *
 * This function is similar to @vsprintf, except that this function allocates
 * memory to store output string, and returns a pointer to it via the first
 * parameter.
 *
 * @param[out] strp If no error occurs, this function sets @c *strp to be
 * a pointer to output string.
 * Otherwise, this function sets @c *strp to be @c NULL.
 * If an error occurs after memory is allocated successfully, this function
 * calls @free for the allocated memory.
 * @param[in] format Same as @sprintf and/or @vsprintf format string.
 * @param[in] ap Same as @vsprintf variable argument list.
 *
 * @return If no error occurs, this function returns the number of bytes of
 * output string, not including the final NUL character.
 * Otherwise, this function returns @c -1.
 */
int vasprintf(char **strp, const char *format, va_list ap)
    __attribute__((format(printf, 2, 0)));

/**
 * @brief Print to allocated string.
 *
 * This function is similar to @sprintf, except that this function allocates
 * memory to store output string, and returns a pointer to it via the first
 * parameter.
 *
 * @param[out] strp Same as @vasprintf.
 * @param[in] format Same as @vasprintf.
 * @param[in] ... Same as @sprintf variable argument list.
 *
 * @return Same as @vasprintf.
 */
int asprintf(char **strp, const char *format, ...)
    __attribute__((format(printf, 2, 3)));

#endif
