package shooting;

/*Modelクラス改
 * 
 * jarファイルにまとまるにあたり改造した。
 * 元は桃缶たべたいさんのHPからいただいた。
 * ★getResource()をつかうのでテクスチャーファイルを指定する方法を変更。
 * その結果、コンストラクタのインスタンス、ObjFileが不要になった。
 * ★なぜかエクリプスで起動するとクラスローダを使ってテクスチャーを読もうとModelクラスを呼ぶとエラーになる。
 * jarファイルにした時はクラスローダを使わなければならない。
 * そのため、エクリプスで動かすときとjarにエクスポートするときではソースを変更しなければ
 * ならない。
 * プログラム上で「今、jarから動いているのかエクリプスから動かしているのか」判断できるようにするべき。
 * 
 */
import static org.lwjgl.opengl.GL11.GL_COMPILE;
import static org.lwjgl.opengl.GL11.GL_TRIANGLES;
import static org.lwjgl.opengl.GL11.glBegin;
import static org.lwjgl.opengl.GL11.glCallList;
import static org.lwjgl.opengl.GL11.glDeleteLists;
import static org.lwjgl.opengl.GL11.glEnd;
import static org.lwjgl.opengl.GL11.glEndList;
import static org.lwjgl.opengl.GL11.glGenLists;
import static org.lwjgl.opengl.GL11.glNewList;
import static org.lwjgl.opengl.GL11.glNormal3f;
import static org.lwjgl.opengl.GL11.glTexCoord2f;
import static org.lwjgl.opengl.GL11.glVertex3f;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.lwjgl.util.vector.Vector3f;

import shooting.Texture;
import shooting.TextureLoader;
 
public class Model {
    private final File                  imagesDir;
    private final List<Vector3f>      vertices = new ArrayList<Vector3f>();
    private final List<Point>         textureVertices = new ArrayList<Point>();
    private final List<Vector3f>      normals = new ArrayList<Vector3f>();
    private final Map<String, Material>   materials = new HashMap<String, Material>();
    private final List<DisplayList>       displayLists = new ArrayList<DisplayList>();
     
    public Model(String imagesDir) {
        this.imagesDir = new File(imagesDir);
    }
 
    /**
     *  この Model を最適化し、必要なリソースを確保する
     */
    public void compile(TextureLoader textureLoader) throws IOException {
        for (DisplayList displayList: displayLists) {
            displayList.compile(textureLoader);
        }
    }
 
    /**
     *  この Model が持つすべてのリソースを破棄する
     */
    public void dispose() {
        for (DisplayList displayList: displayLists) {
            displayList.dispose();
        }
    }
 
    /**
     *  この Model を描画する
     */
    public void render() {
        for (DisplayList displayList: displayLists) {
            Material    material = materials.get(displayList.materialName);
 
            if (material != null) {
                Texture texture = material.texture;
                texture.bind();
            }
 
            glCallList(displayList.objectDisplayList);
        }
    }
 
    public DisplayList createDisplayList() {
        DisplayList displayList = new DisplayList();
         
        displayLists.add(displayList);
         
        return displayList;
    }
 
    public void addVertex(float x, float y, float z) {
        vertices.add(new Vector3f(x, y, z));
    }
     
    public Vector3f getVertex(int index) {
        return vertices.get(index - 1);
    }
     
    public void addNormal(float x, float y, float z) {
        normals.add(new Vector3f(x, y, z));
    }
     
    public Vector3f getNormal(int index) {
        return normals.get(index - 1);
    }
     
    public Point getTextureVertex(int index) {
        return textureVertices.get(index - 1);
    }
     
    public void addTextureVertex(float x, float y) {
        textureVertices.add(new Point(x, y));
    }
     
    public Material createMaterial(String name) {
        Material    material = new Material(name);
 
        this.materials.put(name, material);
         
        return material;
    }
 
    /**
     *  モデルを構成するオブジェクトを表すクラス
     */
    class DisplayList {
        private final List<Face>  faces = new ArrayList<Face>();
         
        private String          materialName;
        private int             objectDisplayList;
 
        public void setMaterialName(String materialName) {
            this.materialName = materialName;
        }
 
        public void compile(TextureLoader textureLoader) throws IOException {
            //  ディスプレイリストを確保し、ID を生成する
            objectDisplayList = glGenLists(1);
             
            //  ディスプレイリストへ描画処理を保存する
            glNewList(objectDisplayList, GL_COMPILE);
            {
                glBegin(GL_TRIANGLES);
 
                for (Face face: faces.toArray(new Face[] {})) {
                    Point       texCord1 = (0 < face.textureVertex1Index)? getTextureVertex(face.textureVertex1Index): null;
                    Point       texCord2 = (0 < face.textureVertex2Index)? getTextureVertex(face.textureVertex2Index): null;
                    Point       texCord3 = (0 < face.textureVertex3Index)? getTextureVertex(face.textureVertex3Index): null;
 
                    Vector3f    normal1 = getNormal(face.normal1Index);
                    Vector3f    normal2 = getNormal(face.normal2Index);
                    Vector3f    normal3 = getNormal(face.normal3Index);
                    Vector3f    vector1 = getVertex(face.vertex1Index);
                    Vector3f    vector2 = getVertex(face.vertex2Index);
                    Vector3f    vector3 = getVertex(face.vertex3Index);
 
                    glNormal3f(normal1.x, normal1.y, normal1.z);
                    if (texCord1 != null) glTexCoord2f(texCord1.getX(), texCord1.getY() * -1);
                    glVertex3f(vector1.x, vector1.y, vector1.z);
 
                    glNormal3f(normal2.x, normal2.y, normal2.z);
                    if (texCord2 != null) glTexCoord2f(texCord2.getX(), texCord2.getY() * -1);
                    glVertex3f(vector2.x, vector2.y, vector2.z);
 
                    glNormal3f(normal3.x, normal3.y, normal3.z);
                    if (texCord3 != null) glTexCoord2f(texCord3.getX(), texCord3.getY() * -1);
                    glVertex3f(vector3.x, vector3.y, vector3.z);
                }
                glEnd();
            }
            //  ディスプレイリストへの保存を終了する
            glEndList();
             
            //  テクスチャーをロードする
            Material    material = materials.get(materialName);
 
            if (material != null) {
                String  texturePath;
                //WindowsでもLinuxでもjarファイルの中でのディレクトリのセパレータは「/」でよい。
                texturePath = imagesDir + "/" + material.textureName;
                
                //以下のコードは不要になった。
                //File    textureFile = new File(material.textureName);
               /* if (textureFile.isAbsolute()) {
                    texturePath = textureFile.getPath();
                } else if (imagesDir.isAbsolute()) {
                    texturePath = imagesDir.getCanonicalFile().getParent() + File.separator + textureFile.getPath();
                } else {
                    texturePath = objFile.getCanonicalFile().getParent() + File.separator + imagesDir.getPath() + File.separator + textureFile.getPath();
                }
                System.out.println("texturePath=" + texturePath);*/
                
                //エクリプスで起動するときはクラスローダーを引数にしないようにすること
                //material.texture = textureLoader.loadTexture(texturePath);
                material.texture = textureLoader.loadTexture(texturePath, Model.class.getClassLoader());
            }
        }
 
        public void addFace(
                int vertex1Index, int vertex2Index, int vertex3Index,
                int textureVertex1Index, int textureVertex2Index, int textureVertex3Index,
                int normal1Index, int normal2Index, int normal3Index) {
            faces.add(new Face(
                    vertex1Index, vertex2Index, vertex3Index,
                    textureVertex1Index, textureVertex2Index, textureVertex3Index,
                    normal1Index, normal2Index, normal3Index));
        }
 
        public void dispose() {
            //  テクスチャーを破棄する
            for (Material material: materials.values()) {
                material.dispose();
            }
 
            //  モデル描画の記録を破棄する
            glDeleteLists(objectDisplayList, 1);
        }
    }
     
    /**
     *  オブジェクトを構成する面を表すクラス
     *  面は三角形のみとし、各頂点がそれぞれ頂点座標、テクスチャー座標、法線のインデックスを持つ
     */
    class Face {
        private final int       vertex1Index;
        private final int       vertex2Index;
        private final int       vertex3Index;
 
        private final int       textureVertex1Index;
        private final int       textureVertex2Index;
        private final int       textureVertex3Index;
 
        private final int       normal1Index;
        private final int       normal2Index;
        private final int       normal3Index;
         
        public Face(
                int vertex1Index, int vertex2Index, int vertex3Index,
                int textureVertex1Index, int textureVertex2Index, int textureVertex3Index,
                int normal1Index, int normal2Index, int normal3Index) {
            this.vertex1Index = vertex1Index;
            this.vertex2Index = vertex2Index;
            this.vertex3Index = vertex3Index;
            this.textureVertex1Index = textureVertex1Index;
            this.textureVertex2Index = textureVertex2Index;
            this.textureVertex3Index = textureVertex3Index;
            this.normal1Index = normal1Index;
            this.normal2Index = normal2Index;
            this.normal3Index = normal3Index;
        }
    }
     
    /**
     *  二次元座標を表すクラス
     */
    class Point {
        private final float x;
        private final float y;
 
        public Point(float x, float y) {
            this.x = x;
            this.y = y;
        }
 
        public float getX() {
            return x;
        }
 
        public float getY() {
            return y;
        }
    }
 
    /**
     *  マテリアルを表すクラス
     */
    class Material {
        private String      textureName;
        private Texture     texture;
 
        public Material(String name) {
        }
 
        public void setTextureName(String textureName) {
            this.textureName = textureName;
        }
 
        public void dispose() {
            if (texture != null) {
                texture.dispose();
            }
        }
    }
}