/**@file
 *
 *@brief	QRL Maker
 *@date		Wed,28 Oct,2015 - Sat,31 Oct,2015
 *@date		Sun,22 Nov,2015
 *@date		Sun,10 Jan,2016
 *@date		Thu,11 Feb,2016 - Sun,14 Feb,2016
 *@date		Tue,15 Aug,2017
 *@date		Fri,18 Aug,2017
 *@date		Sat,27 Oct,2018
 *@date		Thu,29 Nov,2018
 *@date		Sat,16 Feb,2019
 *@date		Sat,25 Jan,2020
 *@date		Sun,13 Sep,2020
 *@date		Wed,16 Sep,2020
 *@date		Wed,28 Oct,2020
 *@date		Sat,31 Oct,2020
 *@date		Sun,08 Nov,2020
 *@date		Wed,17 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,10 Oct,2021
 *@date		Sat,30 Oct,2021
 *@date		Sat,22 Apr,2023
 *@date		Mon,24 Apr,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "settings.h"
#include "qrlmap.h"
#include "qrl.h"
#include <QString>
#include <QStringList>
#include <QFile>
#include <QTextStream>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
# include <QStringConverter>
#else
# include <QTextCodec>
#endif

bool QRL::QRLMap::save(const QString& filename) const noexcept
{
	QFile file(filename);
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
		return false;
	}
  #else
	if (!file.open(QIODevice::WriteOnly)) {
		return false;
	}
  #endif

	bool ret = true;

	QTextStream out(&file);
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	out.setEncoding(QStringConverter::Utf8);
  #else
	QTextCodec* const codec = QTextCodec::codecForName("UTF-8");
	if (codec) {
		out.setCodec(codec);
	}
  #endif

	ret &= qrlmaker->QRLLibrary()->out_header(out);
	ret &= qrlmaker->QRLLibrary()->out_stamp(out);
	out << QChar(0x0D) << QChar(0x0A) << Qt::flush;

	for (QRL::QRLChipNumber i = 0; i < this->maxlist(); ++i) {
		const QRL::QRLChip* const ptr = this->ptr(i);
		assert (NULL != ptr);
		if (ptr->chip_number <= 0) {
			continue;
		}

		const QRL::QRLChipLibrary* const chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
		assert (NULL != chip);

		const QString chip_number_label = QString("%1").arg(ptr->chip_number, Settings::QRLChipLibrary.Limit.Number.Digit, 10, QChar('0'));

		{ // Header
			const QString address = QString("%1 , %2").arg(ptr->addr.x + 1).arg(ptr->addr.y + 1);
			const QString arrow_address_condition_true_label = QString("%1").arg(ptr->next_address_condition_true, Settings::QRLChipLibrary.Limit.Arrow.Digit, 10, QChar('0'));
			const QString arrow_address_condition_false_label = QString("%1").arg(ptr->next_address_condition_false, Settings::QRLChipLibrary.Limit.Arrow.Digit, 10, QChar('0'));

			QStringList list = chip->header->split("\n");
			for (int n = 0; n < list.size(); ++n) {
				bool hit = false;
				QString line = list[n];

				if (!hit) {
					if (line.length() < 1) {
						hit = true;
					}
				}

				if (!hit) {
					const int begin_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Begin);
					if (0 < begin_ptr) {
						out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int name_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.name);
					if (0 <= name_ptr) {
						out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int size_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.size);
					if (0 <= size_ptr) {
						hit = true;
					}
				}

				if (!hit) {
					const int footprint_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.footprint);
					if (0 <= footprint_ptr) {
						hit = true;
					}
				}

				if (!hit) {
					const int chip_coordinates_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_coordinates_tag);
					if (0 <= chip_coordinates_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int chip_number_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_number_tag);
					if (0 <= chip_number_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int chip_footprint_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_footprint_tag);
					if (0 <= chip_footprint_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int chip_number_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_number);
					if (0 <= chip_number_ptr) {
						out << QString(line.left(chip_number_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_number.length())) << QChar(' ') << chip_number_label << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int address_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.address);
					if (0 <= address_ptr) {
						out << QString(line.left(address_ptr + Settings::QRLChipLibrary.Syntax.Header.address.length())) << QChar(' ') << address << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_coordinates_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_coordinates_tag);
					if (0 <= next_address_condition_true_coordinates_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true_tag);
					if (0 <= next_address_condition_true_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_true_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true);
					if (0 <= next_address_condition_true_ptr) {
						out << QString(line.left(next_address_condition_true_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true.length())) << QChar(' ') << arrow_address_condition_true_label << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_coordinates_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_coordinates_tag);
					if (0 <= next_address_condition_false_coordinates_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_tag_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false_tag);
					if (0 <= next_address_condition_false_tag_ptr) {
						// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int next_address_condition_false_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false);
					if (0 <= next_address_condition_false_ptr) {
						out << QString(line.left(next_address_condition_false_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false.length())) << QChar(' ') << arrow_address_condition_false_label << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						hit = true;
					}
				}

				if (!hit) {
					const int argument_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_num);
					if (0 <= argument_ptr) {
						for (int i = 0; i < ptr->arg.size(); ++i) {
							const QString arg = QString("%1").arg((ptr->arg)[i]);
							out << QString(line.left(argument_ptr)) << Settings::QRLChipLibrary.Syntax.Header.argument_arg << (i + 1) << Settings::QRLChipLibrary.Syntax.Header.argument_closebracket << QChar(' ') << arg << QChar(0x0D) << QChar(0x0A) << Qt::flush;
						}
						hit = true;
					}
				}

				if (!hit) {
					// out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
				}
			}
		}
		{ // Body
			const QString chip_coordinates_label = QString("%1-%2").arg((ptr->addr.x + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg((ptr->addr.y + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
			QString chip_footprint_label = "";
			{
				const struct QRL::ChipAddrXY max = chip->size;
				QStringList::const_iterator it = chip->footprint->begin();
				struct QRL::ChipAddrXY cur;
				for (cur.y = 0; cur.y < max.y; ++cur.y, ++it) {
					for (cur.x = 0; cur.x < max.x; ++cur.x) {
						if (QChar('#') == (*it)[static_cast<int>(cur.x)]) {
							chip_footprint_label += QString("%1-%2,").arg((ptr->addr.x + cur.x + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg((ptr->addr.y + cur.y + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
						}
					}
				}
			}

			QString next_address_condition_true_coordinates_label = QString("%1-%2").arg(0, Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg(0, Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
			QString next_address_condition_true_label = Settings::QRLChipLibrary.Syntax.Label_OutOfRange;
			QString next_address_condition_false_coordinates_label = QString("%1-%2").arg(0, Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg(0, Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
			QString next_address_condition_false_label = Settings::QRLChipLibrary.Syntax.Label_OutOfRange;

			QRL::ChipAddrXY xy_true;
			if (qrlmaker->QRLLibrary()->convert_arrowaddress(chip, ptr->addr, ptr->next_address_condition_true, &xy_true)) {
				next_address_condition_true_coordinates_label = QString("%1-%2").arg((xy_true.x + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg((xy_true.y + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
				QRL::QRLChipNumber num = map(xy_true);
				if (0 <= num) {
					next_address_condition_true_label = QString("%1").arg(num, Settings::QRLChipLibrary.Limit.Number.Digit, 10, QChar('0'));
				}
			}

			QRL::ChipAddrXY xy_false;
			if (qrlmaker->QRLLibrary()->convert_arrowaddress(chip, ptr->addr, ptr->next_address_condition_false, &xy_false)) {
				next_address_condition_false_coordinates_label = QString("%1-%2").arg((xy_false.x + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0')).arg((xy_false.y + 1), Settings::QRLChipLibrary.Limit.Address.Digit, 10, QChar('0'));
				QRL::QRLChipNumber num = map(xy_false);
				if (0 <= num) {
					next_address_condition_false_label = QString("%1").arg(num, Settings::QRLChipLibrary.Limit.Number.Digit, 10, QChar('0'));
				}
			}

			QStringList list = chip->body->split("\n");
			for (int n = 0; n < list.size(); ++n) {
				QString line = list[n];

				QString str;

				if (NULL != chip->chip_coordinates_tag) {
					do {
						str = line;
						line.replace(*(chip->chip_coordinates_tag), chip_coordinates_label);
					} while (str != line);
				}

				if (NULL != chip->chip_number_tag) {
					do {
						str = line;
						line.replace(*(chip->chip_number_tag), chip_number_label);
					} while (str != line);
				}

				if (NULL != chip->chip_footprint_tag) {
					do {
						str = line;
						line.replace(*(chip->chip_footprint_tag), chip_footprint_label);
					} while (str != line);
				}

				if (NULL != chip->next_address_condition_true_coordinates_tag) {
					do {
						str = line;
						line.replace(*(chip->next_address_condition_true_coordinates_tag), next_address_condition_true_coordinates_label);
					} while (str != line);
				}

				if (NULL != chip->next_address_condition_true_tag) {
					do {
						str = line;
						line.replace(*(chip->next_address_condition_true_tag), next_address_condition_true_label);
					} while (str != line);
				}

				if (NULL != chip->next_address_condition_false_coordinates_tag) {
					do {
						str = line;
						line.replace(*(chip->next_address_condition_false_coordinates_tag), next_address_condition_false_coordinates_label);
					} while (str != line);
				}

				if (NULL != chip->next_address_condition_false_tag) {
					do {
						str = line;
						line.replace(*(chip->next_address_condition_false_tag), next_address_condition_false_label);
					} while (str != line);
				}

				for (uint_fast8_t i = 0; i < chip->argument_num; ++i) {
					if (NULL != chip->argument[i].tag) {
						if (NULL != chip->argument[i].tag_raw) {
							do {
								str = line;
								line.replace(*(chip->argument[i].tag_raw), (ptr->arg)[i]);
							} while (str != line);
						}

						Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind type = chip->argument[i].type;
						switch (chip->argument[i].type) {
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM:
							do {
								str = line;
								line.replace(*(chip->argument[i].tag), qrlmaker->QRLLibrary()->convert_argument_num_to_string((ptr->arg)[i]));
							} while (str != line);
							break;
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR:
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING:
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL:
							do {
								str = line;
								line.replace(*(chip->argument[i].tag), (ptr->arg)[i]);
							} while (str != line);
							break;
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR_NUM:
							if (Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM == qrlmaker->QRLLibrary()->get_argumenttype((ptr->arg)[i])) {
								type = Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM;
								do {
									str = line;
									line.replace(*(chip->argument[i].tag), qrlmaker->QRLLibrary()->convert_argument_num_to_string((ptr->arg)[i]));
								} while (str != line);
							} else {
								type = Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR;
								do {
									str = line;
									QString tmp = Settings::QRLChipLibrary.Syntax.Argument.var_syntax;
									tmp.replace(Settings::QRLChipLibrary.Syntax.Argument.var_argument, (ptr->arg)[i]);
									line.replace(*(chip->argument[i].tag), tmp);
								} while (str != line);
							}
							break;
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LIST:
							do {
								str = line;
								line.replace(*(chip->argument[i].tag), (ptr->arg)[i]);
							} while (str != line);
							break;
						case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL:
							break;
						}
						if ((NULL != chip->argument[i].type_selected) && (type < Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL)) {
							do {
								str = line;
								line.replace(*(chip->argument[i].type_selected), Settings::QRLChipLibrary.Syntax.Header.argument_type_kind[type].label );
							} while (str != line);
						}
					}
				}

				out << line << QChar(0x0D) << QChar(0x0A) << Qt::flush;
			}
		}
	}

	ret &= qrlmaker->QRLLibrary()->out_footer(out);

	return ret;
}
// [ End of File ]
