/**@file
 *
 *@brief	QRL Maker
 *@date		Sat,31 Oct,2015
 *@date		Sun,10 Jan,2016
 *@date		Thu,11 Feb,2016
 *@date		Sun,09 Jul,2017
 *@date		Fri,18 Aug,2017
 *@date		Sat,12 May,2018
 *@date		Sat,17 Nov,2018
 *@date		Sat,24 Nov,2018
 *@date		Thu,29 Nov,2018
 *@date		Wed,16 Sep,2020
 *@date		Sun,27 Dec,2020
 *@date		Wed,17 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,30 May,2021
 *@date		Sun,10 Oct,2021
 *@date		Sat,30 Oct,2021
 *@date		Sat,22 Apr,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "settings.h"
#include "qrlmap.h"
#include "qrl.h"
#include <QString>
#include <QStringList>
#include <QTranslator>
#include <QFile>
#include <QTextStream>
#include <QDateTime>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
# include <QStringConverter>
#else
# include <QTextCodec>
#endif

bool QRL::QRLChipList::load(const QString& filename) noexcept(false)
{
	QStringList errmsg;

	QFile file(filename);
	if (!file.open(QIODevice::ReadOnly)) {
		return false;
	}

	bool ret = true;

	clear();
	const QRL::QRLChipLibrary* chip = NULL;
	QRL::QRLChipNumber i = -1;
	QRL::QRLChip* ptr = NULL;

	enum LineType { NONE, HEADER, BODY } type = NONE;

	QTextStream in(&file);
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	in.setEncoding(QStringConverter::Utf8);
  #else
	QTextCodec* const codec = QTextCodec::codecForName("UTF-8");
	if (codec) {
		in.setCodec(codec);
	}
  #endif

	while (!in.atEnd()) {
		QString line = in.readLine(0);

		if (0 < line.indexOf(Settings::QRLChipLibrary.Syntax.Begin)) {
			type = HEADER;
			++i;
			if (this->maxlist() <= i) {
				return false;
			}
			ptr = this->ptr_(i);
		} else if (0 < line.indexOf(Settings::QRLChipLibrary.Syntax.Body)) {
			type = BODY;
		} else if (0 < line.indexOf(Settings::QRLChipLibrary.Syntax.End)) {
			type = NONE;
		} else if (HEADER == type) {
			if (NULL == ptr) {
				return false;
			}

			bool hit = false;

			line = line.trimmed();
			line = line.simplified();

			if (!hit) {
				if (line.length() < 1) {
					hit = true;
				}
			}

			if (!hit) {
				const int name_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.name);
				if (0 <= name_ptr) {
					(ptr->name) = QString(line.mid(name_ptr + Settings::QRLChipLibrary.Syntax.Header.name.length()).trimmed());
					chip = qrlmaker->QRLLibrary()->chipSearch(ptr->name);
					// assert (NULL != chip);
					if (NULL == chip) {
						errmsg.append(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
									  + tr("A Chip Was Not Found in The Library : '%1'").arg(ptr->name));
						ptr->chip_number = 0;
						type = NONE;
						hit = true;
						ret = false;
					} else {
						while (0 < ptr->arg.size()) {
							ptr->arg.removeFirst();
						}
						for (uint_fast8_t i = 0; i < chip->argument_num; ++i) {
							ptr->arg.append("");
						}
						hit = true;
					}
				}
			}

			if (!hit) {
				const int chip_number_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.chip_number);
				if (0 <= chip_number_ptr) {
					ptr->chip_number = line.mid(chip_number_ptr + Settings::QRLChipLibrary.Syntax.Header.chip_number.length()).trimmed().toInt();
					hit = true;
				}
			}

			if (!hit) {
				const int address_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.address);
				if (0 <= address_ptr) {
					QString str = line.mid(address_ptr + Settings::QRLChipLibrary.Syntax.Header.address.length()).trimmed();
					QStringList list2 = str.split(",");
					assert (2 == list2.size());
					ptr->addr.x = (list2[0].toInt() - 1);
					ptr->addr.y = (list2[1].toInt() - 1);
					hit = true;
				}
			}

			if (!hit) {
				const int next_address_condition_true_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true);
				if (0 <= next_address_condition_true_ptr) {
					ptr->next_address_condition_true = line.mid(next_address_condition_true_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_true.length()).trimmed().toLong();
					hit = true;
				}
			}

			if (!hit) {
				const int next_address_condition_false_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false);
				if (0 <= next_address_condition_false_ptr) {
					ptr->next_address_condition_false = line.mid(next_address_condition_false_ptr + Settings::QRLChipLibrary.Syntax.Header.next_address_condition_false.length()).trimmed().toLong();
					hit = true;
				}
			}

			if (!hit) {
				const int argument_arg_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_arg);
				const int argument_closebracket_ptr = line.indexOf(Settings::QRLChipLibrary.Syntax.Header.argument_closebracket);
				if ((0 <= argument_arg_ptr) && (0 <= argument_closebracket_ptr)) {
					QString str = line.mid(argument_arg_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_arg.length(), (argument_closebracket_ptr - (argument_arg_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_arg.length()))).trimmed();
					hit = true;
					const uint_fast8_t i = str.toInt() - 1;
					if ((NULL != chip) && ((0 <= i) && (i < chip->argument_num))) {
						(ptr->arg)[i] = QString(line.mid(argument_closebracket_ptr + Settings::QRLChipLibrary.Syntax.Header.argument_closebracket.length()).trimmed());
					}
				}
			}

			if (!hit) {
				errmsg.append(QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
							  + tr("Unknown Header : '%1'").arg(line));
				ret = false;
			}
		}
	}


	if (!errmsg.isEmpty()) {
		throw errmsg;
	}
	return ret;
}

bool QRL::QRLMap::load(const QString& filename) noexcept(false)
{
	bool ret = true;

	try {
		ret = qrlchiplist->load(filename);
		rebuild_map();
	}
	catch (QStringList errmsg) {
		rebuild_map();
		throw std::move(errmsg);
	}

	return ret;
}
// [ End of File ]
