/**@file
 *
 *@brief	QRL Maker
 *@date		Thu,17 Sep,2015 - Sat,19 Sep,2015
 *@date		Sun,20 Sep,2015 - Tue,29 Sep,2015
 *@date		Sat,03 Oct,2015 - Mon,12 Oct,2015
 *@date		Tue,13 Oct,2015 - Thu,15 Oct,2015
 *@date		Sat,17 Oct,2015 - Sun,18 Oct,2015
 *@date		Thu,22 Oct,2015 - Sun,25 Oct,2015
 *@date		Wed,28 Oct,2015 - Sat,31 Oct,2015
 *@date		Sun,10 Jan,2016
 *@date		Sun,16 Apr,2017
 *@date		Sun,09 Jul,2017
 *@date		Sat,15 Jul,2017
 *@date		Tue,27 Nov,2018
 *@date		Fri,28 Dec,2018
 *@date		Sun,30 Dec,2018
 *@date		Mon,21 Jan,2019
 *@date		Sat,19 Sep,2020
 *@date		Sun,27 Dec,2020
 *@date		Wed,17 Feb,2021
 *@date		Fri,19 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,30 May,2021
 *@date		Fri,04 Jun,2021
 *@date		Sat,05 Jun,2021
 *@date		Thu,02 Nov,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef QRLMAP_H
#define QRLMAP_H

#include <assert.h>
#include <cstdint>
#include <vector>
#include "qrlmaker.h"
#include "qrl.h"
#include <QTranslator>
#include <QString>
#include <QStringList>

namespace QRL {

class QRLChip
{
public:
	explicit QRLChip(void) noexcept;
	~QRLChip(void) noexcept;

	void clear(void) noexcept;

	QString name;
	QRLChipNumber chip_number;
	struct ChipAddrXY addr;
	QRLChipArrowAddress next_address_condition_true;
	QRLChipArrowAddress next_address_condition_false;
	QStringList arg;

	QRL::QRLChip& operator =(const QRLChip& arg) noexcept;
	bool operator ==(const QRLChip& arg) const noexcept;
	bool operator !=(const QRLChip& arg) const noexcept;

protected:

private:
};

class QRLChipList
{
	Q_DECLARE_TR_FUNCTIONS(QRLLibrary)

public:
	explicit QRLChipList(const QRLMaker* const q, QRLChipNumber size_num) noexcept;
	explicit QRLChipList(const QRLMaker* const q) noexcept;
	~QRLChipList(void) noexcept;

	void clear(void) noexcept;
	QRL::QRLChip* ptr_(QRLChipNumber num) noexcept;
	const QRL::QRLChip* ptr(QRLChipNumber num) const noexcept;
	QRL::QRLChip* search_(QRL::QRLChipNumber num) noexcept;
	const QRL::QRLChip* search(QRL::QRLChipNumber num) const noexcept;
	QRLChipNumber maxlist(void) const noexcept;
	QRLChipNumber numlist(void) const noexcept;

	bool put(const QRLChip* const chip) noexcept;
	QRL::QRLChipNumber put(const ChipAddrXY& addr, const QRLChipLibrary* const chip, const QRL::QRLChip* const ptr) noexcept;
	bool remove(QRL::QRLChipNumber num) noexcept;

	bool reallocate(struct ChipAddrXY* const ret_base_chipaddr) noexcept;

	bool load(const QString& filename) noexcept(false);

	QRL::QRLChipList& operator =(const QRLChipList& arg) noexcept;
	bool operator ==(const QRLChipList& arg) const noexcept;

protected:

private:
	const QRLMaker* qrlmaker;

	class QRLChip* qrlchips;
	QRLChipNumber listsize;
};

class QRLMap
{
public:
	explicit QRLMap(const QRLMaker* const q, const ChipAddrXY& size, QRLChipNumber size_num) noexcept;
	~QRLMap(void) noexcept;

	void clear(void) noexcept;

	QRL::QRLChip* ptr_(QRLChipNumber num) noexcept;
	const QRL::QRLChip* ptr(QRLChipNumber num) const noexcept;
	QRL::QRLChip* search_(QRL::QRLChipNumber num) noexcept;
	const QRL::QRLChip* search(QRL::QRLChipNumber num) const noexcept;
	QRLChipNumber maxlist(void) const noexcept;
	QRLChipNumber numlist(void) const noexcept;
	QRL::QRLChipNumber map(const ChipAddrXY& addr) const noexcept;
	bool is_put(const ChipAddrXY& addr, const QRLChipLibrary* const chip) const noexcept;
	bool put_to_map(const ChipAddrXY& addr, const QRL::QRLChipLibrary* const chip, const QRLChipNumber chip_number) noexcept;
	bool put(const ChipAddrXY& addr, const QRLChipLibrary* const chip) noexcept;
	bool put(const ChipAddrXY& addr, const QRLChip* const ptr, bool* const chip_number_changed = NULL) noexcept;
	bool remove_from_map(const ChipAddrXY& st, const QRL::QRLChipLibrary* const chip) noexcept;
	bool remove(QRL::QRLChipNumber num) noexcept;
	bool remove(const ChipAddrXY& addr) noexcept;
	int_fast8_t get_undo_count(void) const noexcept;
	int_fast8_t get_redo_count(void) const noexcept;
	const QString get_undo_msg(const int_fast8_t buf_ptr) const noexcept;
	const QString get_redo_msg(const int_fast8_t buf_ptr) const noexcept;
	int_fast8_t get_undo_ptr(void) const noexcept;
	int_fast8_t reset_undo(const QString& msg) noexcept;
	int_fast8_t push_undo(const QString& msg) noexcept;
	int_fast8_t undo(QString* const msg) noexcept;
	int_fast8_t redo(QString* const msg) noexcept;

	bool save(const QString& filename) const noexcept;
	bool load(const QString& filename) noexcept(false);

protected:

private:
	QRL::QRLChipNumber set_map(const ChipAddrXY& addr, QRL::QRLChipNumber num) noexcept;
	bool rebuild_map(void) noexcept;

	const QRLMaker* qrlmaker;

	QRL::QRLChipList* qrlchiplist;

	std::vector<QRL::QRLChipList> qrlchiplist_undobuf;
	QString* qrlchiplist_undobuf_memo;
	int_fast8_t undo_size;
	int_fast8_t undo_ptr;
	int_fast8_t undo_limit;
	int_fast8_t redo_limit;

	QRL::QRLChipNumber* qrlchipmap;
	ChipAddrXY mapsize;
};
};



inline QRL::QRLChip& QRL::QRLChip::operator =(const QRL::QRLChip& arg) noexcept
{
	this->chip_number = arg.chip_number;
	this->addr = arg.addr;
	this->name = arg.name;
	this->next_address_condition_true = arg.next_address_condition_true;
	this->next_address_condition_false = arg.next_address_condition_false;
	this->arg = arg.arg;

	return *this;
}

inline bool QRL::QRLChip::operator ==(const QRL::QRLChip& arg) const noexcept
{
	if (this->chip_number != arg.chip_number) {
		return false;
	}
	if (this->addr != arg.addr) {
		return false;
	}
	if (this->name != arg.name) {
		return false;
	}
	if (this->next_address_condition_true != arg.next_address_condition_true) {
		return false;
	}
	if (this->next_address_condition_false != arg.next_address_condition_false) {
		return false;
	}
	if (this->arg != arg.arg) {
		return false;
	}

	return true;
}

inline bool QRL::QRLChip::operator !=(const QRL::QRLChip& arg) const noexcept
{
	return !((*this) == arg);
}

inline QRL::QRLChipList& QRL::QRLChipList::operator =(const QRL::QRLChipList& arg) noexcept
{
	assert (this->listsize == arg.listsize);
	for (QRL::QRLChipNumber i = 0; i < this->listsize; ++i) {
		this->qrlchips[i] = arg.qrlchips[i];
	}
	return *this;
}

inline bool QRL::QRLChipList::operator ==(const QRL::QRLChipList& arg) const noexcept
{
	if (this->listsize != arg.listsize) {
		return false;
	}

	for (QRL::QRLChipNumber i = 0; i < this->listsize; ++i) {
		if (this->qrlchips[i] != arg.qrlchips[i]) {
			return false;
		}
	}
	return true;
}

inline QRL::QRLChip* QRL::QRLChipList::ptr_(QRLChipNumber num) noexcept
{
	if ((num < 0) || (this->listsize < num)) {
		return NULL;
	}
	return &(qrlchips[num]);
}

inline const QRL::QRLChip* QRL::QRLChipList::ptr(QRLChipNumber num) const noexcept
{
	if ((num < 0) || (this->listsize < num)) {
		return NULL;
	}
	return &(qrlchips[num]);
}

inline QRL::QRLChip* QRL::QRLChipList::search_(QRL::QRLChipNumber num) noexcept
{
	for (QRL::QRLChipNumber i = 0; i < this->listsize; ++i) {
		if (num == qrlchips[i].chip_number) {
			return &(qrlchips[i]);
		}
	}
	return NULL;
}

inline const QRL::QRLChip* QRL::QRLChipList::search(QRL::QRLChipNumber num) const noexcept
{
	for (QRL::QRLChipNumber i = 0; i < this->listsize; ++i) {
		if (num == qrlchips[i].chip_number) {
			return &(qrlchips[i]);
		}
	}
	return NULL;
}

inline QRL::QRLChipNumber QRL::QRLChipList::maxlist(void) const noexcept
{
	return listsize;
}

inline QRL::QRLChipNumber QRL::QRLChipList::numlist(void) const noexcept
{
	QRL::QRLChipNumber num = 0;
	for (QRL::QRLChipNumber i = 0; i < this->listsize; ++i) {
		if (0 < qrlchips[i].chip_number) {
			++num;
		}
	}
	return num;
}

inline QRL::QRLChip* QRL::QRLMap::ptr_(QRL::QRLChipNumber num) noexcept
{
	return qrlchiplist->ptr_(num);
}

inline const QRL::QRLChip* QRL::QRLMap::ptr(QRL::QRLChipNumber num) const noexcept
{
	return qrlchiplist->ptr(num);
}

inline QRL::QRLChip* QRL::QRLMap::search_(QRL::QRLChipNumber num) noexcept
{
	return qrlchiplist->search_(num);
}

inline const QRL::QRLChip* QRL::QRLMap::search(QRL::QRLChipNumber num) const noexcept
{
	return qrlchiplist->search(num);
}

inline QRL::QRLChipNumber QRL::QRLMap::maxlist(void) const noexcept
{
	return qrlchiplist->maxlist();
}

inline QRL::QRLChipNumber QRL::QRLMap::numlist(void) const noexcept
{
	return qrlchiplist->numlist();
}

inline QRL::QRLChipNumber QRL::QRLMap::map(const ChipAddrXY& addr) const noexcept
{
	if ((addr.x < 0) || (addr.y < 0)) {
		return QRLChipOutOfMapFlag;
	}
	if ((this->mapsize.x <= addr.x) || (this->mapsize.y <= addr.y)) {
		return QRLChipOutOfMapFlag;
	}
	return qrlchipmap[addr.x + (addr.y * this->mapsize.x) ];
}

inline QRL::QRLChipNumber QRL::QRLMap::set_map(const ChipAddrXY& addr, QRL::QRLChipNumber num) noexcept
{
	if ((addr.x < 0) || (addr.y < 0)) {
		return QRLChipOutOfMapFlag;
	}
	if ((this->mapsize.x <= addr.x) || (this->mapsize.y <= addr.y)) {
		return QRLChipOutOfMapFlag;
	}
	qrlchipmap[addr.x + (addr.y * this->mapsize.x) ] = num;
	return num;
}

inline int_fast8_t QRL::QRLMap::get_undo_count(void) const noexcept
{
	int_fast8_t count = (this->undo_ptr - this->undo_limit);
	if (count < 0) {
		count += this->undo_size;
	}
	return count;
}

inline int_fast8_t QRL::QRLMap::get_redo_count(void) const noexcept
{
	int_fast8_t count = (this->redo_limit - this->undo_ptr);
	if (count < 0) {
		count += this->undo_size;
	}
	return count;
}

inline const QString QRL::QRLMap::get_undo_msg(const int_fast8_t buf_ptr) const noexcept
{
	if (get_undo_count() < buf_ptr) {
		return "";
	}
	int_fast8_t target_ptr = (this->undo_ptr - buf_ptr + 1);
	if (target_ptr < 0) {
		target_ptr += this->undo_size;
	}
	return ((0 <= target_ptr) && (target_ptr < this->undo_size)) ? qrlchiplist_undobuf_memo[target_ptr] : "";
}

inline const QString QRL::QRLMap::get_redo_msg(const int_fast8_t buf_ptr) const noexcept
{
	if (get_redo_count() < buf_ptr) {
		return "";
	}
	int_fast8_t target_ptr = (this->undo_ptr + buf_ptr);
	if (this->undo_size <= target_ptr) {
		target_ptr -= this->undo_size;
	}
	return ((0 <= target_ptr) && (target_ptr < this->undo_size)) ? qrlchiplist_undobuf_memo[target_ptr] : "";
}

inline int_fast8_t QRL::QRLMap::get_undo_ptr(void) const noexcept
{
	return this->undo_ptr;
}

#endif // QRLMAP_H
// [ End of File ]
