/**@file
 *
 *@brief	QRL Maker
 *@date		Sat,27 Oct,2018
 *@date		Sun,20 Jan,2019
 *@date		Sun,15 Nov,2020
 *@date		Sat,21 Nov,2020
 *@date		Sun,27 Dec,2020
 *@date		Fri,19 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Mon,22 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Sun,23 May,2021
 *@date		Mon,24 May,2021
 *@date		Fri,28 May,2021
 *@date		Sat,29 May,2021
 *@date		Sun,30 May,2021
 *@date		Wed,02 Jun,2021
 *@date		Sat,18 Feb,2023
 *
 *@author	Copyright(C)2018-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "qrl.h"
#include "settings.h"
#include "qrlmaker.h"
#include "dialog_listdialog.h"
#include "ui_dialog_listdialog.h"
#include <QString>
#include <QStringList>
#include <QList>
#include <QTranslator>
#include <QDateTime>
#include <QPoint>
#include <QSize>
#include <QResizeEvent>
#include <QMoveEvent>
#include <QTreeWidget>
#include <QTreeWidgetItem>

Dialog_ListDialog::Dialog_ListDialog(const QRLMaker* const q, const QStringList* const arg_list, QWidget *parent) noexcept :
	QDialog(parent),
	ui(new(std::nothrow) Ui::Dialog_ListDialog),
	qrlmaker(NULL),
	list(NULL)
{
	assert (NULL != q);
	assert (NULL != ui);

	this->qrlmaker = q;

	this->list = arg_list;
	this->str = "";
	this->err = false;

	ui->setupUi(this);

	ui->treeWidget->setContextMenuPolicy(Qt::CustomContextMenu);

	for (QStringList::const_iterator itr = this->list->constBegin(); itr != this->list->constEnd(); ++itr) {
		QTreeWidgetItem* const item = new(std::nothrow) QTreeWidgetItem;
		assert (NULL != item);
		const QString tmp_tr = QCoreApplication::translate(Settings::QRLChipLibrary.tr_context.ChipArgumentList, qPrintable(*itr), Settings::QRLChipLibrary.tr_disambiguation.ArgumentList);
		item->setText(0, tmp_tr + " ( " + *itr + " )");
		item->setText(1, *itr);
		ui->treeWidget->addTopLevelItem(item);
		ui->treeWidget->expandItem(item);
	}

	{
		Settings::ini->beginGroup(Settings::AppConfig.get_AppName());

		const QPoint pos  = Settings::SettingsCandidates.dialog_listdialog_pos ;
		const QSize  size = Settings::SettingsCandidates.dialog_listdialog_size;

		Settings::ini->endGroup();

		if (!size.isEmpty() && !pos.isNull()) {
			this->resize(size);
			this->move(pos);
		}
	}

	ui->treeWidget->hideColumn(1);
	return;
}

Dialog_ListDialog::~Dialog_ListDialog() noexcept
{
	assert (NULL != ui);
	assert (NULL != qrlmaker);
	assert (NULL != list);

	delete ui;
	return;
}

void Dialog_ListDialog::setParam(const QString& label, const QString* const text) noexcept
{
	ui->label->setText(label);
	if (NULL == text) {
		ui->treeWidget->selectionModel()->clear();
		this->str = "";
	} else {
		const QList<QTreeWidgetItem*> items = ui->treeWidget->findItems(*text, Qt::MatchExactly, 1);
		if (!items.isEmpty()) {
			ui->treeWidget->setCurrentItem(items[0]);
			this->str = *text;
		} else {
			ui->treeWidget->selectionModel()->clear();
			this->str = "";
		}
	}
	ui->treeWidget->setFocus();
	this->err = false;
	return;
}

const QString Dialog_ListDialog::getText(void) const noexcept
{
	return this->str;
}

bool Dialog_ListDialog::getErr(void) const noexcept
{
	return this->err;
}

void Dialog_ListDialog::moveEvent(QMoveEvent* const event) noexcept
{
	if (!event->oldPos().isNull() && !event->pos().isNull()) {
		Settings::SettingsCandidates.dialog_listdialog_pos = event->pos();
	}
	return;
}

void Dialog_ListDialog::resizeEvent(QResizeEvent* const event) noexcept
{
	if (!event->oldSize().isEmpty() && !event->size().isEmpty()) {
		Settings::SettingsCandidates.dialog_listdialog_size = event->size();
	}
	return;
}

void Dialog_ListDialog::on_buttonBox_accepted(void) noexcept
{
	const QString tmp = ui->treeWidget->currentItem()->text(1);
	this->str = tmp;
	const bool ret = qrlmaker->QRLLibrary()->check_argumenttype_list((*list), tmp);
	if (ret) {
		this->err = false;
		accept();
	} else {
		this->err = true;
		errmsg = QDateTime::currentDateTime().toString(QObject::tr("ddd,dd MMM,yyyy hh:mm:ss")) + " "
				+ tr("Argument Mismatch : '%1'").arg(tmp);
		reject();
	}
	return;
}

void Dialog_ListDialog::on_buttonBox_rejected(void) noexcept
{
	this->err = false;
	reject();
	return;
}
// [ End of File ]
