/**@file
 *
 *@brief	QRL Maker
 *@date		Sun,25 Oct,2015
 *@date		Wed,28 Oct,2015
 *@date		Sun,14 Feb,2016
 *@date		Sun,16 Apr,2017
 *@date		Sat,27 Oct,2018
 *@date		Tue,27 Nov,2018 - Thu,29 Nov,2018
 *@date		Wed,28 Oct,2020
 *@date		Sat,31 Oct,2020
 *@date		Thu,18 Feb,2021
 *@date		Sat,20 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Sun,10 Oct,2021
 *@date		Sat,22 Apr,2023
 *
 *@author	Copyright(C)2015-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <assert.h>
#include "settings.h"
#include "qrl.h"
#include <QChar>
#include <QString>
#include <QStringList>
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
# include <QRegularExpression>
#else
# include <QRegExp>
#endif

bool QRL::QRLLibrary::convert_arrowaddress(const QRL::QRLChipLibrary* const chip, const ChipAddrXY& ofs_arg, QRL::QRLChipArrowAddress arrow_addr, QRL::ChipAddrXY* const xy) const noexcept
{
	assert (NULL != chip);
	assert (NULL != xy);

	if (arrow_addr <= 0) {
		return false;
	}

	const struct QRL::ChipAddrXY ofs = {
		.x = static_cast<QRL::ChipAddr>((arrow_addr / Settings::QRLChipLibrary.Limit.Address.Base / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
		.y = static_cast<QRL::ChipAddr>((arrow_addr / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
	};
	const QRL::ChipArrow arrow = static_cast<QRL::ChipArrow>(arrow_addr % 10);

	if (QChar('#') != (*chip->footprint)[static_cast<int>(ofs.y - 1)][static_cast<int>(ofs.x - 1)]) {
		return false;
	}

	xy->x = ofs_arg.x + ofs.x - 1 + AngDir[arrow].x;
	xy->y = ofs_arg.y + ofs.y - 1 + AngDir[arrow].y;

	if (xy->x < 0) {
		return false;
	}
	if (xy->y < 0) {
		return false;
	}
	if (Settings::QRLChipLibrary.Limit.Address.Max < xy->x) {
		return false;
	}
	if (Settings::QRLChipLibrary.Limit.Address.Max < xy->x) {
		return false;
	}

	return true;
}

bool QRL::QRLLibrary::check_arrowaddress(const QRL::QRLChipLibrary* const chip, QRL::QRLChipArrowAddress arrow_addr) const noexcept
{
	assert (NULL != chip);

	if (arrow_addr <= 0) {
		return false;
	}

	const struct QRL::ChipAddrXY ofs = {
		.x = static_cast<QRL::ChipAddr>((arrow_addr / Settings::QRLChipLibrary.Limit.Address.Base / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
		.y = static_cast<QRL::ChipAddr>((arrow_addr / 10) % Settings::QRLChipLibrary.Limit.Address.Base),
	};
	const QRL::ChipArrow arrow = static_cast<QRL::ChipArrow>(arrow_addr % 10);

	if (QChar('#') != (*chip->footprint)[static_cast<int>(ofs.y - 1)][static_cast<int>(ofs.x - 1)]) {
		return false;
	}

	const struct QRL::ChipAddrXY xy = {
		.x = static_cast<QRL::ChipAddr>(ofs.x - 1 + AngDir[arrow].x),
		.y = static_cast<QRL::ChipAddr>(ofs.y - 1 + AngDir[arrow].y),
	};

	if (xy.x < 0) {
		return true;
	}
	if (xy.y < 0) {
		return true;
	}
	if (chip->size.x <= xy.x) {
		return true;
	}
	if (chip->size.y <= xy.y) {
		return true;
	}

	if (QChar('#') != (*chip->footprint)[static_cast<int>(xy.y)][static_cast<int>(xy.x)]) {
		return true;
	}

	return false;
}

bool QRL::QRLLibrary::check_argumenttype_freestring(Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind type, const QString& str) const noexcept
{
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	QRegularExpression num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);
	QRegularExpression var(Settings::QRLChipLibrary.Syntax.Argument.regex_var);
	QRegularExpression string(Settings::QRLChipLibrary.Syntax.Argument.regex_str);
	QRegularExpression label(Settings::QRLChipLibrary.Syntax.Argument.regex_label);

	switch (type) {
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR:
		if (var.match(str).hasMatch()) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM:
		if (num.match(str).hasMatch()) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR_NUM:
		if (var.match(str).hasMatch() || num.match(str).hasMatch()) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING:
		if (string.match(str).hasMatch()) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL:
		if (label.match(str).hasMatch()) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LIST:
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL:
		return false;
	}
  #else
	QRegExp num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);
	QRegExp var(Settings::QRLChipLibrary.Syntax.Argument.regex_var);
	QRegExp string(Settings::QRLChipLibrary.Syntax.Argument.regex_str);
	QRegExp label(Settings::QRLChipLibrary.Syntax.Argument.regex_label);

	switch (type) {
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR:
		if (var.exactMatch(str)) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM:
		if (num.exactMatch(str)) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR_NUM:
		if (var.exactMatch(str) || num.exactMatch(str)) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING:
		if (string.exactMatch(str)) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL:
		if (label.exactMatch(str)) {
			return true;
		}
		break;
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LIST:
	case Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL:
		return false;
	}
  #endif
	return false;
}

bool QRL::QRLLibrary::check_argumenttype_list(const QStringList& arg_list, const QString& str) const noexcept
{
	for (int i = 0; i < arg_list.size(); ++i) {
		if (arg_list[i] == str) {
			return true;
		}
	}
	return false;
}

Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::ArgumentType_Kind QRL::QRLLibrary::get_argumenttype(const QString& str) const noexcept
{
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	QRegularExpression num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);
	QRegularExpression var(Settings::QRLChipLibrary.Syntax.Argument.regex_var);
	QRegularExpression string(Settings::QRLChipLibrary.Syntax.Argument.regex_str);
	QRegularExpression label(Settings::QRLChipLibrary.Syntax.Argument.regex_label);

	if (num.match(str).hasMatch()) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM;
	} else if (var.match(str).hasMatch()) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR;
	} else if (string.match(str).hasMatch()) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING;
	} else if (label.match(str).hasMatch()) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL;
	}
  #else
	QRegExp num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);
	QRegExp var(Settings::QRLChipLibrary.Syntax.Argument.regex_var);
	QRegExp string(Settings::QRLChipLibrary.Syntax.Argument.regex_str);
	QRegExp label(Settings::QRLChipLibrary.Syntax.Argument.regex_label);

	if (num.exactMatch(str)) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::NUM;
	} else if (var.exactMatch(str)) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::VAR;
	} else if (string.exactMatch(str)) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::STRING;
	} else if (label.exactMatch(str)) {
		return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::LABEL;
	}
  #endif
	return Settings::QRLChip_Library::QRL_Syntax::QRL_Header::Argument_Type::SENTINEL;
}

QString QRL::QRLLibrary::convert_argument_num_to_string(const QString& str) const noexcept
{
  #if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
	QRegularExpression num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);

	if (num.match(str).hasMatch()) {
		QString in = str;

		for (const struct Settings::QRLChip_Library::QRL_Syntax::QRL_Argument::QRL_Argument_Type_Num_List* list = Settings::QRLChipLibrary.Syntax.Argument.argument_type_num; (0 < list->regex.length()); ++list) {
			QRegularExpression num_regex(list->regex);

			QString tmp;
			if (num_regex.match(in).hasMatch()) {
				in.replace(num_regex, "\\1");
				tmp = list->syntax;
				tmp.replace(list->argument, in);
				return tmp;
			}
		}

		return str;
	}
  #else
	QRegExp num(Settings::QRLChipLibrary.Syntax.Argument.regex_num);

	if (num.exactMatch(str)) {
		QString in = str;

		for (const struct Settings::QRLChip_Library::QRL_Syntax::QRL_Argument::QRL_Argument_Type_Num_List* list = Settings::QRLChipLibrary.Syntax.Argument.argument_type_num; (0 < list->regex.length()); ++list) {
			QRegExp num_regex(list->regex);

			QString tmp;
			if (num_regex.exactMatch(in)) {
				in.replace(num_regex, "\\1");
				tmp = list->syntax;
				tmp.replace(list->argument, in);
				return tmp;
			}
		}

		return str;
	}
  #endif
	return "";
}
// [ End of File ]
