/**@file
 *
 *@brief	QRL Maker
 *@date		Sun,10 Aug,2014
 *@date		Thu,28 May,2015 - Sun,31 May,2015
 *@date		Tue,08 Sep,2015
 *@date		Fri,11 Sep,2015 - Tue,15 Sep,2015
 *@date		Sun,20 Sep,2015 - Mon,21 Sep,2015
 *@date		Sat,03 Oct,2015 - Wed,07 Oct,2015
 *@date		Tue,13 Oct,2015
 *@date		Mon,19 Oct,2015
 *@date		Mon,26 Oct,2015
 *@date		Sat,31 Oct,2015
 *@date		Sun,16 Jul,2017
 *@date		Wed,26 Jul,2017
 *@date		Sat,17 Mar,2018
 *@date		Fri,23 Nov,2018
 *@date		Wed,28 Nov,2018
 *@date		Fri,28 Dec,2018
 *@date		Thu,19 Nov,2020
 *@date		Sun,29 Nov,2020
 *@date		Sun,27 Dec,2020
 *@date		Fri,19 Feb,2021
 *@date		Sun,21 Feb,2021
 *@date		Tue,23 Feb,2021
 *@date		Sat,29 May,2021
 *@date		Fri,08 Jul,2022
 *@date		Wed,13 Jul,2022
 *@date		Sat,16 Jul,2022
 *@date		Thu,16 Feb,2023
 *
 *@author	Copyright(C)2014-2023 G-HAL. All rights reserved.
 *
 */
/**\mainpage
	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions
	are met:

 * Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.

 * Neither the name of the copyright holders nor the names of its contributors
 may be used to endorse or promote products derived from this software
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <cstdint>
#include "settings.h"
#include "mainwindow.h"
#include "mytypes.h"
#include "ui_mainwindow.h"
#include <QString>

void MainWindow::showEvent(_unused QShowEvent* const e) noexcept
{
	on_actionReload_Settings_triggered();
	return;
}

float MainWindow::get_dB2scale(void) const noexcept
{
	const float value = ui->horizontalSlider->value();
	return powf(10.0, (value / 20.0));
}

float MainWindow::get_scale2dB(float arg) const noexcept
{
	const float dbMin = - 40.0;
	if (0.0 < arg) {
		const float decibel = 20.0 * log10f(arg);
		return qMax(decibel, dbMin);
	}
	return dbMin;
}

void MainWindow::draw_ScaleLabel() noexcept
{
	const float value = ui->horizontalSlider->value();
	const float scaling = get_dB2scale();
	if (0.0 <= value) {
		ui->Scale->setText(tr("Scale : %1 / 1").arg(scaling, 4, 'f', 2));
	} else {
		ui->Scale->setText(tr("Scale : 1 / %1").arg((1.0 / scaling), 4, 'f', 2));
	}
	return;
}

bool MainWindow::select_scaling_dB_reset(void) noexcept
{
	ui->horizontalSlider->setValue(0.0);
	draw_ScaleLabel();
	return true;
}

bool MainWindow::select_scaling_dB_expand(void) noexcept
{
	const int value = ui->horizontalSlider->value() + 1.0;
	ui->horizontalSlider->setValue(value);
	draw_ScaleLabel();
	return true;
}

bool MainWindow::select_scaling_dB_shrink(void) noexcept
{
	const int value = ui->horizontalSlider->value() - 1.0;
	ui->horizontalSlider->setValue(value);
	draw_ScaleLabel();
	return true;
}

void MainWindow::paintEvent(_unused QPaintEvent* const e) noexcept
{
	if (this->needRefresh) {
		this->needRefresh = false;
		scaling_dB_changed();
	}
	return;
}

void MainWindow::scaling_dB_changed(void) noexcept
{
	draw_ScaleLabel();
	const float scaling = get_dB2scale();

	const uint_fast32_t area_w = static_cast<uint_fast32_t>(ui->form_mainform->width()  - Settings::BuildImage.GaugeSize - Settings::BuildImage.EdgeSize);
	const uint_fast32_t area_h = static_cast<uint_fast32_t>(ui->form_mainform->height() - Settings::BuildImage.GaugeSize - Settings::BuildImage.EdgeSize);
	const uint_fast16_t scaled_gs = static_cast<uint_fast16_t>(floor(Settings::BuildImage.Chip.GapSize * scaling) / scaling);
	const uint_fast16_t scaled_csgs = static_cast<uint_fast16_t>(floor((Settings::BuildImage.Chip.GapSize + Settings::BuildImage.Chip.CoreSize) * scaling) / scaling);
	const int_fast16_t num_x = static_cast<int_fast16_t>((area_w / scaling - scaled_gs) / scaled_csgs);
	const int_fast16_t num_y = static_cast<int_fast16_t>((area_h / scaling - scaled_gs) / scaled_csgs);

	if (Settings::BuildImage.Chip.Count.x < num_x) {
		ui->horizontalScrollBar->setRange(0, 0);
	} else {
		ui->horizontalScrollBar->setRange(0, (Settings::BuildImage.Chip.Count.x - num_x));
	}
	if (Settings::BuildImage.Chip.Count.y < num_y) {
		ui->verticalScrollBar->setRange(0, 0);
	} else {
		ui->verticalScrollBar->setRange(0, (Settings::BuildImage.Chip.Count.y - num_y));
	}

	ui->form_mainform->scale_changed(scaling);

	return;
}

void MainWindow::select_form_treeitem(void) noexcept
{
	QList<QTreeWidgetItem *> selected_list = ui->form_treewidget->selectedItems();
	if (selected_list.length() < 1) {
		ui->form_mainform->select_form_treeitem_name("", "");
		ui->form_mainform->select_form_treeitem(Settings::EditorCommand::NORMAL, NULL);
		return;
	}

	QTreeWidgetItem* const item = selected_list[0];
	const QString str = item->text(1);

	if (str.length() < 2) {
		ui->form_mainform->select_form_treeitem_name("", "");
		ui->form_mainform->select_form_treeitem(Settings::EditorCommand::NORMAL, NULL);
		return;
	}

	// Editor Command
	if (' ' == str[0]) {
		if (' ' == str[1]) {
			ui->form_mainform->select_form_treeitem_name(str.mid(2), item->text(0));
			ui->form_mainform->select_form_treeitem(Settings::EditorCommand::NORMAL, NULL);
			return;
		}

		const QTreeWidgetItem* const parent = item->parent();
		if (NULL != parent) {
			const int idx = parent->indexOfChild(item);
			ui->form_mainform->select_form_treeitem_name(Settings::EditorCommands[idx].label, item->text(0));
			ui->form_mainform->select_form_treeitem(static_cast<enum Settings::EditorCommand::EditorCommand_TYPE>(idx + 1), NULL);
			return;
		}

		ui->form_mainform->select_form_treeitem_name(str.mid(1), item->text(0));
		ui->form_mainform->select_form_treeitem(Settings::EditorCommand::NORMAL, NULL);
		return;
	}

	// Chip Library
	ui->form_mainform->select_form_treeitem_name(str, item->text(0));
	for (QVector<QRL::QRLChipLibrary_List>::const_iterator itr = qrlmaker->QRLLibrary()->libraryBegin(); itr != qrlmaker->QRLLibrary()->libraryEnd(); ++itr) {
		switch (itr->type) {
		case QRL::QRLChipLibrary_List::NONE:
			break;

		case QRL::QRLChipLibrary_List::LABEL:
			break;

		case QRL::QRLChipLibrary_List::CHIP:
			if (*(itr->data.chip.name) == str) {
				ui->form_mainform->select_form_treeitem(Settings::EditorCommand::PUT_CHIP, &(itr->data.chip));
				return;
			}
		}
	}

	ui->form_mainform->select_form_treeitem(Settings::EditorCommand::NORMAL, NULL);
	return;
}
// [ End of File ]
