// 
// Copyright (c) 2003-2010, MIST Project, Nagoya University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// 
// 3. Neither the name of the Nagoya University nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 

/// @file mist/io/dicom.h
//!
//! @brief DICOM摜ǂݏ邽߂̃Cu
//!
#ifndef __INCLUDE_MIST_DICOM__
#define __INCLUDE_MIST_DICOM__


#ifndef __INCLUDE_MIST_CONF_H__
#include "../config/mist_conf.h"
#endif

#ifndef __INCLUDE_MIST_H__
#include "../mist.h"
#endif

#ifndef __INCLUDE_MIST_ENDIAN__
#include "../config/endian.h"
#endif

#ifndef __INCLUDE_MIST_LIMITS__
#include "../limits.h"
#endif

// J[摜̐ݒǂݍ
#ifndef __INCLUDE_MIST_COLOR_H__
#include "../config/color.h"
#endif

#ifndef __INCLUDE_MIST_DICOM_TAG__
#include "./dicom_tag.h"
#endif

#ifndef __INCLUDE_MIST_DICOM_INFO__
#include "./dicom_info.h"
#endif

#ifndef __INCLUDE_MIST_SINGLETON__
#include "../singleton.h"
#endif

// mistOԂ̎n܂
_MIST_BEGIN


//! @addtogroup image_group 摜o
//!  @{

//! @addtogroup dicom_group DICOM摜o
//!
//! @code ̃wb_CN[h
//! #include <mist/io/dicom.h>
//! @endcode
//!
//!  @{


// ̃}N`ƁCR\[ɓǂݍ񂾃^O\
// #define __SHOW_DICOM_TAG__
// #define __SHOW_DICOM_UNKNOWN_TAG__
// #define __SHOW_DICOM_ZEROBYTE_TAG__
// #define __CHECK_TAG_FORMAT__

// ̃}N`ƁCFs\ȃ^O͔r
// #define __EXCLUDE_DICOM_UNKNOWN_TAG__
// #define __EXCLUDE_DICOM_ZEROBYTE_TAG__


/// DICOMt@C𑀍삷֐ENX܂ޖO
namespace dicom
{
	/// @brief w肵̈悪DICOM摜Ă邩ǂ`FbN
	//! 
	//! @attention ̈ 132 oCgȏȂĂ͂ȂȂ
	//! 
	//! @param[in] p c 擪ʒu
	//! @param[in] e c ʒu
	//! 
	//! @return DICOM̃vAu̎w|C^
	//! 
	inline unsigned char *check_dicom_file( unsigned char *p, unsigned char *e )
	{
		if( p == NULL || p + 4 >= e )
		{
			return( NULL );
		}

		char *dicm = reinterpret_cast< char * >( p ) + 128;
		if( dicm[ 0 ] == 'D' && dicm[ 1 ] == 'I' && dicm[ 2 ] == 'C' && dicm[ 3 ] == 'M' )
		{
			if( p + 4 + 128 >= e )
			{
				return( NULL );
			}
			else
			{
				return( p + 128 + 4 );
			}
		}

		// DICOMp̃vAu݂Ȃ߁C擪̃|C^Ԃ
		return( p );
	}


	/// @brief V[PX^Oǂ𔻒肷
	//! 
	//! @param[in] p c 擪|C^
	//! @param[in] e c |C^
	//! 
	//! @retval true  c V[PX^Ȍꍇ
	//! @retval false c ȊO
	//! 
	inline bool is_sequence_separate_tag( const unsigned char *p, const unsigned char *e )
	{
		if( p + 4 > e )
		{
			return( false );
		}
		return( p[ 0 ] == 0xfe && p[ 1 ] == 0xff && p[ 2 ] == 0x00 && p[ 3 ] == 0xe0 );
	}

	/// @brief V[PXvfI^Oǂ
	//! 
	//! @param[in] p c 擪|C^
	//! @param[in] e c |C^
	//! 
	//! @retval true  c V[PXvfI^Ȍꍇ
	//! @retval false c ȊO
	//! 
	inline bool is_sequence_element_end( const unsigned char *p, const unsigned char *e )
	{
		if( p + 8 > e )
		{
			return( false );
		}
		return( p[ 0 ] == 0xfe && p[ 1 ] == 0xff && p[ 2 ] == 0x0d && p[ 3 ] == 0xe0 && p[ 4 ] == 0x00 && p[ 5 ] == 0x00 && p[ 6 ] == 0x00 && p[ 7 ] == 0x00 );
	}

	/// @brief V[PXI^Oǂ
	//! 
	//! @param[in] p c 擪|C^
	//! @param[in] e c |C^
	//! 
	//! @retval true  c V[PXI^Ȍꍇ
	//! @retval false c ȊO
	//! 
	inline bool is_sequence_tag_end( const unsigned char *p, const unsigned char *e )
	{
		if( p + 8 > e )
		{
			return( false );
		}
		return( p[ 0 ] == 0xfe && p[ 1 ] == 0xff && p[ 2 ] == 0xdd && p[ 3 ] == 0xe0 && p[ 4 ] == 0x00 && p[ 5 ] == 0x00 && p[ 6 ] == 0x00 && p[ 7 ] == 0x00 );
	}

	template < bool _IS_SIGNED_ >
	struct __check_num_bytes_function__
	{
		template < class T >
		static ptrdiff_t check( const T &nbytes ) 
		{
			if( nbytes == type_limits< T >::maximum( ) )
			{
				return( -1 );
			}
			else
			{
				return( nbytes );
			}
		}
	};

	template < >
	struct __check_num_bytes_function__< true >
	{
		template < class T >
		static ptrdiff_t check( const T &nbytes ) 
		{
			return( nbytes );
		}
	};

	template < class T >
	inline ptrdiff_t __check_num_bytes__( const T &nbytes )
	{
		return( __check_num_bytes_function__< std::numeric_limits< T >::is_signed >::check( nbytes ) );
	}

	/// @brief DICOM̃^Oǂݍ݁Ce[uɓo^Ă̂Əƍ
	//! 
	//! e[uɓo^ĂȂꍇ́Cǂݔ΂D
	//! CK؂DICOMt@CłȂꍇ-2ԂCf[^̏I[̓t@C̓ǂݍ݂Ɏsꍇ-1ԂD
	//! āCǂݔ΂ꍇ0ԂCe[uɓo^Ăꍇ͎ɑ݂f[^̃oCgԂD
	//! f[^擪̃|C^ 'p' ƁCf[^̍Ō{1|C^ 'e' ^
	//! 
	//! @param[in]  p        c 擪ʒu
	//! @param[in]  e        c ʒu
	//! @param[out] tag      c 擾ꂽDICOM̃^O
	//! @param[out] numBytes c DICOM^OgpĂ郁̃oCg
	//! @param[in]  from_little_endian c ̓f[^gGfBAǂ
	//! 
	//! @return ^Of[^ew擪|C^
	//! 
	inline unsigned char *read_dicom_tag( unsigned char *p, unsigned char *e, dicom_tag &tag, difference_type &numBytes, bool from_little_endian = true )
	{
		if( p == NULL || p + 8 > e )
		{
			numBytes = -1;
			return( e );
		}

		unsigned char *data = p;

		unsigned short group   = to_current_endian( byte_array< unsigned short >( data ), from_little_endian ).get_value( );
		unsigned short element = to_current_endian( byte_array< unsigned short >( data + 2 ), from_little_endian ).get_value( );

		data += 4;

		char VR[ 3 ] = { static_cast< char >( data[ 0 ] ), static_cast< char >( data[ 1 ] ), '\0' };
		dicom_vr vr = get_dicom_vr( VR );

		difference_type num_bytes = 0;
		const dicom_tag_table &dicom_table = singleton< dicom_tag_table >::get_instance( );
		tag = dicom_table.get_tag( group, element, vr );

		if( tag.tag != 0xffffffff )
		{
			//^Oe[uɓo^Ăꍇ
			if( tag.vr == vr )
			{
				// IVR
				switch( vr )
				{
				case OB:
				case OW:
				case UN:
				case SQ:
				case OF:
				case UT:
					data += 4;
					num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( data ), from_little_endian ).get_value( ) );
					break;

				default:
					num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned short >( data + 2 ), from_little_endian ).get_value( ) );
					break;
				}
			}
			else if( vr != UNKNOWN )
			{
				// IVRł邪CƈقȂꍇ
				tag.vr = vr;
				switch( vr )
				{
				case OB:
				case OW:
				case UN:
				case SQ:
				case OF:
				case UT:
					data += 4;
					num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( data ), from_little_endian ).get_value( ) );
					break;

				default:
					num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned short >( data + 2 ), from_little_endian ).get_value( ) );
				}
			}
			else
			{
				// ÎIVR
				num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( data ), from_little_endian ).get_value( ) );
			}

			numBytes = num_bytes;
			return( reinterpret_cast< unsigned char * >( data + 4 ) );
		}
		else
		{
			//^Oe[uɓo^ĂȂꍇ̓f[^XLbv
			switch( vr )
			{
			case AE:
			case AS:
			case AT:
			case CS:
			case DA:
			case DS:
			case DT:
			case FL:
			case FD:
			case IS:
			case LO:
			case LT:
			case PN:
			case SH:
			case SL:
			case SS:
			case ST:
			case TM:
			case UI:
			case UL:
			case US:
				num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned short >( data + 2 ), from_little_endian ).get_value( ) );
				break;

			case OB:
			case OW:
			case UN:
			case SQ:
			case OF:
			case UT:
				data += 4;
				num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( data ), from_little_endian ).get_value( ) );
				break;

			default:
				num_bytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( data ), from_little_endian ).get_value( ) );
				break;
			}

#if defined( __SHOW_DICOM_ZEROBYTE_TAG__ ) && defined( __SHOW_DICOM_UNKNOWN_TAG__ )
			printf( "( %04x, %04x, %s, % 8d ) = Unknown Tags!!\n", group, element, VR, num_bytes );
#endif
			if( vr == SQ )
			{
				// sȃ^OǃV[PX^OƂƂmF
				tag = dicom_tag( construct_dicom_tag( group, element ), SQ, -1, "UNKNOWN" );
				numBytes = num_bytes;
				return( data + 4 );
			}
			else if( data + 4 + 4 <= e && data[ 0 ] == 0xff && data[ 1 ] == 0xff && data[ 2 ] == 0xff && data[ 3 ] == 0xff && is_sequence_separate_tag( data + 4, e ) )
			{
				// sȃ^OǃV[PX^Ỏ\ꍇƂƂmF
				tag = dicom_tag( construct_dicom_tag( group, element ), SQ, -1, "UNKNOWN" );
				numBytes = num_bytes;
				return( data + 4 );
			}
			else if( data + 4 + num_bytes <= e )
			{
				if( element == 0x0000 )
				{
					// Group Length ^O
					char str[ 50 ];
					sprintf( str, "Group %04d Length", group );
					tag = dicom_tag( construct_dicom_tag( group, element ), UL, 1, str );
					numBytes = num_bytes;
					return( reinterpret_cast< unsigned char * >( data + 4 ) );
				}
				else
				{
					tag = dicom_tag( construct_dicom_tag( group, element ), vr, -1, "UNKNOWN" );
					numBytes = num_bytes;
					return( reinterpret_cast< unsigned char * >( data + 4 ) );
				}
			}
			else
			{
				numBytes = -2;
				return( p );
			}
		}
	}



	/// @brief DICOM̃^Oɑ΂CeVRƂ̏s
	//! 
	//! @param[in]     tag       c DICOM^O
	//! @param[in,out] byte      c oCg
	//! @param[in]     num_bytes c oCg̒
	//! @param[in]     from_little_endian c ̓f[^gGfBAǂ
	//! 
	//! @return o^ĂȂ^Ȍꍇ false Cꂽꍇ̂ true Ԃ
	//! 
	inline bool process_dicom_tag( const dicom_tag &tag, unsigned char *byte, difference_type num_bytes, bool from_little_endian = true )
	{
		switch( tag.vr )
		{
		case AE:
			// Application Entity
			// u16oCgȉv
			// 擪ƖɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// Xy[X͓ɈӖȂ
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case AS:
			// N\
			// u4oCgŒv
			// ̏ nnnD, nnnW, nnnM, nnnY ̂ꂩ
			// gp\ȒĺC0-9, D, W, M, Y ̂ꂩ
			// @D: nnn  
			// @W: nnn  T
			// @M: nnn  
			// @Y: nnn  N
			//
			// F "018M" ͐18\
			if( tag.vm != -1 && ( num_bytes % 4 != 0 || (int)( num_bytes / 4 ) > tag.vm ) )
			{
				return( false );
			}
			break;

		case AT:
			// Attribute Tag
			// u4oCgŒv
			// 216rbgȂ̕
			//
			// F (0018,00FF) ̃^ÓC4oCg̕тŕ\CGfBÄႢɂȉ̂悤ɕ
			// gGfBA]\  18H, 00H, FFH, 00H
			// rbOGfBA]\  00H, 18H, 00H, FFH
			if( tag.vm != -1 && ( num_bytes % 4 != 0 || (int)( num_bytes / 4 ) > tag.vm ) )
			{
				return( false );
			}

			if( _is_little_endian_( ) != from_little_endian )
			{
				for( long i = 0 ; i < num_bytes ; i += 4 )
				{
					unsigned char v0 = byte[ i + 0 ];
					unsigned char v1 = byte[ i + 1 ];
					unsigned char v2 = byte[ i + 2 ];
					unsigned char v3 = byte[ i + 3 ];
					byte[ i + 0 ] = v1;
					byte[ i + 1 ] = v0;
					byte[ i + 2 ] = v3;
					byte[ i + 3 ] = v2;
				}
			}
			break;

		case CS:
			// Code String
			// u16oCgȉv
			// 擪ƖɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// Xy[X͓ɈӖȂ
			// 啶 A-Z, 0-9, A_[o[i_ĵ݂gp\D
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case DA:
			// t\
			// u8oCgŒvCDICOM2.0ł́u10oCgŒv̂߂ǂ悤ɂKv
			// yyyymmdd  yyyy.mm.dd ̏œt𕄍
			// ́Cyyyy N mm  dd \
			// 0-9, sIhi.ĵ݂gp\D
			//
			// F19930822  1993 N 8  22 \
			if( tag.vm != -1 && ( num_bytes % 8 != 0 || (int)( num_bytes / 8 ) > tag.vm ) && ( num_bytes % 10 != 0 || (int)( num_bytes / 10 ) > tag.vm ) )
			{
				return( false );
			}
			break;

		case DS:
			// 10i\
			// u16oCgȉv
			// Œ菬_͕_\镶łC擪ƖɃXy[Xi20Hj\̂镶
			// Œ菬_̏ꍇ 0-9 ɉC擪 +  -Cɏ_Cӂ . ܂
			// _́CANSI X3.9 ɏ]Cw E  e ܂
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case DT:
			// \
			// u26oCgȉv
			// YYYYMMDDHHMMSS.FFFFFF&ZZZZ ̏ɏ]ē\C擪ƖɃXy[Xi20Hj\̂镶
			// ́CYYYY N MM  DD  HH  MM  SS.FFFFFF biFFFFFF͕ȁj\
			// ܂C&  +  - ̂ǂ炩łCZZZZ ͎Ԃƕ̃ItZbg\
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			if( tag.vm != -1 && num_bytes > 26 * tag.vm )
			{
				return( false );
			}
			break;

		case FL:
			// Px
			// u4oCgŒv
			if( tag.vm != -1 && ( num_bytes % 4 != 0 || (int)( num_bytes / 4 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< float > tmp = to_current_endian( byte_array< float >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
				byte[ 2 ] = tmp[ 2 ];
				byte[ 3 ] = tmp[ 3 ];
			}
			break;

		case FD:
			// {x
			// u8oCgŒv
			if( tag.vm != -1 && ( num_bytes % 8 != 0 || (int)( num_bytes / 8 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< double > tmp = to_current_endian( byte_array< double >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
				byte[ 2 ] = tmp[ 2 ];
				byte[ 3 ] = tmp[ 3 ];
				byte[ 4 ] = tmp[ 4 ];
				byte[ 5 ] = tmp[ 5 ];
				byte[ 6 ] = tmp[ 6 ];
				byte[ 7 ] = tmp[ 7 ];
			}
			break;

		case IS:
			// \
			// u16oCgȉv
			// 10 Ƃ鐮i10ij\킷ŁC擪 +  - ܂łǂC擪ƖɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// -2^31  n  (2^31-1) ͈̔͂\邱Ƃ\D
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case LO:
			// 
			// u64oCgȉv
			// 擪ƖɃXy[Xi20Hj
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// 䕶͊܂܂Ȃ
			if( tag.vm != -1 && num_bytes > 64 * tag.vm )
			{
				return( false );
			}
			break;

		case LT:
			// eLXg
			// u10240oCgȉv
			// ɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// 䕶R[h܂
			if( tag.vm != -1 && num_bytes > 10240 * tag.vm )
			{
				return( false );
			}
			break;

		case OB:
			// oCg
			// uv
			// ]\̃GfBA`ɈˑȂf[^
			// oCg̒ɕۂ߂ɁCNULLli00HjŖ߂\
			break;

		case OF:
			// 4oCg̃oCg̕у[h
			// uv
			// ]\̃GfBA`ɈˑāCoCg̕тωf[^
			if( _is_little_endian_( ) != from_little_endian )
			{
				for( long i = 0 ; i < num_bytes ; i += 4 )
				{
					unsigned char v1 = byte[ i + 0 ];
					unsigned char v2 = byte[ i + 1 ];
					unsigned char v3 = byte[ i + 2 ];
					unsigned char v4 = byte[ i + 3 ];
					byte[ i + 0 ] = v4;
					byte[ i + 1 ] = v3;
					byte[ i + 2 ] = v2;
					byte[ i + 3 ] = v1;
				}
			}
			break;

		case OW:
			// [h
			// uv
			// ]\̃GfBA`ɈˑāCoCg̕тωf[^
			if( _is_little_endian_( ) != from_little_endian )
			{
				for( long i = 0 ; i < num_bytes ; i += 2 )
				{
					unsigned char tmp = byte[ i ];
					byte[ i ] = byte[ i + 1 ];
					byte[ i + 1 ] = tmp;
				}
			}
			break;

		case PN:
			// Җ\
			// u64oCgȉv
			// 䕶؊܂܂CɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// Ɩ̋؂ɂ́C^ (5EH) p
			if( tag.vm != -1 && num_bytes > 64 * tag.vm )
			{
				return( false );
			}
			break;

		case SH:
			// Z
			// u16oCgȉv
			// 擪ƖɃXy[Xi20Hj
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// 䕶͊܂܂Ȃ
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case SL:
			// t4oCg
			// u4oCgŒv
			if( tag.vm != -1 && ( num_bytes % 4 != 0 || (int)( num_bytes / 4 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< signed int > tmp = to_current_endian( byte_array< signed int >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
				byte[ 2 ] = tmp[ 2 ];
				byte[ 3 ] = tmp[ 3 ];
			}
			break;

		case SQ:
			// V[PX\^O
			// uv
			// DICOM^ÕV[PXi[^O
			break;

		case SS:
			// t2oCg
			// u2oCgŒv
			if( tag.vm != -1 && ( num_bytes % 2 != 0 || (int)( num_bytes / 2 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< signed short > tmp = to_current_endian( byte_array< signed short >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
			}
			break;

		case ST:
			// ZeLXg
			// ɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// 䕶R[h܂
			if( tag.vm != -1 && num_bytes > 1024 * tag.vm )
			{
				return( false );
			}
			break;

		case TM:
			// \
			// u16oCgȉv
			// hhmmss.frac ̏Ŏ𕄍
			// ́Chh  mm  ss b (frac 11bP) \
			// 24Ԍ`̎\LpC[240ƂĈ
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			if( tag.vm != -1 && num_bytes > 16 * tag.vm )
			{
				return( false );
			}
			break;

		case UI:
			// UID\
			// u64oCgȉv
			// ̒ɕۂ߂ɁCNULL (00H) }Kv
			if( tag.vm != -1 && num_bytes > 64 * tag.vm )
			{
				return( false );
			}
			break;

		case UL:
			// 4oCg
			// u4oCgŒv
			if( tag.vm != -1 && ( num_bytes % 4 != 0 || (int)( num_bytes / 4 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< unsigned int > tmp = to_current_endian( byte_array< unsigned int >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
				byte[ 2 ] = tmp[ 2 ];
				byte[ 3 ] = tmp[ 3 ];
			}
			break;

		case UN:
			// sȃ^O
			break;

		case US:
			// 2oCg
			// u2oCgŒv
			if( tag.vm != -1 && ( num_bytes % 2 != 0 || (int)( num_bytes / 2 ) > tag.vm ) )
			{
				return( false );
			}

			{
				byte_array< unsigned short > tmp = to_current_endian( byte_array< unsigned short >( byte ), from_little_endian );
				byte[ 0 ] = tmp[ 0 ];
				byte[ 1 ] = tmp[ 1 ];
			}
			break;

		case UT:
			// ̃eLXg
			// ɃXy[Xi20Hj\̂镶
			// ̒ɕۂ߂ɁCɃXy[X}Kv
			// 䕶R[h܂
			break;

		default:
			// VR킩ȂƂ肠ǂƂ
			break;
		}

		return( true );
	}


	/// @brief DICOM̃^O
	//! 
	//! V[PX\DICOM^Oꍇ́CċAIɖ{֐Kp
	//! 
	//! @param[out] dicm        c DICOMf[^̏o͐
	//! @param[in]  pointer     c 擪|C^
	//! @param[in]  end_pointer c 擪|C^
	//! @param[in]  from_little_endian c ̓f[^gGfBAǂ
	//! @param[in]  is_in_sequence_tag c VRSQ̃^Oł邩ǂiSQ^O͈ɂ܂Ƃ߂j
	//! 
	//! @return ̃^Ow|C^
	//! 
	inline unsigned char *process_dicom_tag( dicom_tag_container &dicm, unsigned char *pointer, unsigned char *end_pointer, bool from_little_endian = true, bool is_in_sequence_tag = false )
	{
		difference_type numBytes = 0;
		dicom_tag tag;

		pointer = read_dicom_tag( pointer, end_pointer, tag, numBytes, from_little_endian );

		if( tag.vr == SQ )
		{
			unsigned char *sp = pointer;
			if( numBytes != -1 && pointer + numBytes <= end_pointer )
			{
				pointer += numBytes;
			}
			else
			{
				unsigned char *ep = numBytes == -1 ? end_pointer : pointer + numBytes;
				if( ep > end_pointer )
				{
					// Fs\ȃV[PX^O
					return( NULL );
				}
				while( pointer + 8 <= ep )
				{
					if( is_sequence_tag_end( pointer, end_pointer ) )
					{
						pointer += 8;
						break;
					}
					else
					{
						if( !is_sequence_separate_tag( pointer, ep ) )
						{
							return( NULL );
						}

						pointer += 4;
						numBytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( pointer ), from_little_endian ).get_value( ) );
						pointer += 4;

						unsigned char *epp = numBytes == -1 ? ep : pointer + numBytes;
						if( epp > ep )
						{
							return( NULL );
						}

						while( pointer + 8 <= epp )
						{
							if( is_sequence_element_end( pointer, ep ) )
							{
								pointer += 8;
								break;
							}
							else
							{
								pointer = process_dicom_tag( dicm, pointer, ep, from_little_endian, true );
								if( pointer == NULL )
								{
									return( NULL );
								}
							}
						}
					}
				}
			}

			if( !is_in_sequence_tag )
			{
				dicom_tag_container::iterator ite;
				ite = dicm.append( dicom_element( tag, sp, pointer - sp ) );

#ifdef __SHOW_DICOM_TAG__
				if( ite != dicm.end( ) )
				{
					ite->second.show_tag( );
				}
#endif
			}
		}
		else if( numBytes < -1 )
		{
			return( NULL );
		}
		else if( numBytes == -1 )
		{
			// kĂ\̂Ń`FbN
			dicom_tag_container::iterator ite;
			switch( tag.vr )
			{
			case OB:
			case OW:
				{
					unsigned char *p = pointer;
					while( p + 8 <= end_pointer )
					{
						if( is_sequence_tag_end( p, end_pointer ) )
						{
							p += 8;
							break;
						}
						else
						{
							if( !is_sequence_separate_tag( p, end_pointer ) )
							{
								return( NULL );
							}

							p += 4;
							numBytes = __check_num_bytes__( to_current_endian( byte_array< unsigned int >( p ), from_little_endian ).get_value( ) );
							p += 4 + numBytes;

							if( numBytes < 0 || p > end_pointer )
							{
								return( NULL );
							}
						}
					}
					if( p <= end_pointer )
					{
						numBytes = p - pointer;
						if( !is_in_sequence_tag )
						{
							ite = dicm.append( dicom_element( tag, pointer, numBytes ) );
						}
						pointer += 8;
					}
					else
					{
						// kV[PX̏I܂ołȂ
						return( NULL );
					}
				}
				break;

			default:
				return( NULL );
			}

			pointer += numBytes;

#ifdef __SHOW_DICOM_TAG__
			if( !is_in_sequence_tag && ite != dicm.end( ) )
			{
				ite->second.show_tag( );
			}
#endif
		}
		else if( numBytes > 0 )
		{
			// sK؂ȃoCgǂ`FbN
			if( pointer + numBytes > end_pointer )
			{
				return( NULL );
			}

#ifndef __CHECK_TAG_FORMAT__
			if( !process_dicom_tag( tag, pointer, numBytes, from_little_endian ) )
			{
				// Kiɏ]ĂȂ^O𔭌CƂ肠Ă
				std::cout << "Illegal DICOM tag is found!" << std::endl;
			}
#else
			if( !process_dicom_tag( tag, pointer, numBytes, from_little_endian ) )
			{
				// 邱ƂłȂDICOM^O𔭌̂ŏI
				return( NULL );
			}
#endif
			if( !is_in_sequence_tag )
			{
				dicom_tag_container::iterator ite;
				ite = dicm.append( dicom_element( tag, pointer, numBytes ) );

#ifdef __SHOW_DICOM_TAG__
				if( ite != dicm.end( ) )
				{
					ite->second.show_tag( );
				}
#endif
			}

			pointer += numBytes;
		}
		else if( tag.vr != UNKNOWN && !is_in_sequence_tag )
		{
			// sȃ^OXgɒǉ
			dicom_tag_container::iterator ite;
			ite = dicm.append( dicom_element( tag, NULL, 0 ) );

#if defined( __SHOW_DICOM_ZEROBYTE_TAG__ ) && defined( __SHOW_DICOM_TAG__ )
			if( ite != dicm.end( ) )
			{
				ite->second.show_tag( );
			}
#endif
		}

		return( pointer );
	}


	/// @brief DICOMt@CǂݍŁCSĂDICOM^Oe[uɓo^
	//! 
	//! @param[out] dicm             c DICOM^OɃf[^o^e[u
	//! @param[in]  buff             c DICOM̃f[^obt@̐擪|C^
	//! @param[in]  numBytes         c obt@̃f[^̃oCg
	//! @param[in]  is_little_endian c f[^̒ggGfBAƂď邩ǂ
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool read_dicom_tags( dicom_tag_container &dicm, unsigned char *buff, size_type numBytes, bool is_little_endian = true )
	{
		unsigned char *pointer = buff;
		unsigned char *end_pointer = buff + numBytes;

		// DICOMf[^̐擪܂Ń|C^ړ
		pointer = check_dicom_file( pointer, end_pointer );

		dicm.clear( );

		unsigned char *group_end_pointer = NULL;
		bool ret = true, from_little_endian = is_little_endian, once = true;
		while( pointer < end_pointer )
		{
			pointer = process_dicom_tag( dicm, pointer, end_pointer, from_little_endian );
			if( pointer == NULL )
			{
				ret = false;
				break;
			}

			if( group_end_pointer == NULL && dicm.contain( 0x0002, 0x0000 ) )
			{
				// GfBA̔ɗp͈͂肷
				group_end_pointer = pointer + find_tag( dicm, 0x0002, 0x0000, static_cast< unsigned int >( 0 ) );
			}

			if( group_end_pointer != NULL && once && dicm.contain( 0x0002, 0x0010 ) )
			{
				// xsȂ悤ɂ
				once = false;

				// GfBǍ`ݒ肷
				from_little_endian = find_tag( dicm, 0x0002, 0x0010, "" ) != "1.2.840.10008.1.2.2";

				// rbOGfBȀꍇ́C̃ubNgGfBAŏ
				if( !from_little_endian )
				{
					while( pointer < group_end_pointer )
					{
						pointer = process_dicom_tag( dicm, pointer, group_end_pointer, true );
						if( pointer == NULL )
						{
							pointer = end_pointer;
							ret = false;
							break;
						}
					}
				}
			}
		}

#ifdef __SHOW_DICOM_TAG__
		printf( "\n\n\n" );
#endif

		return( ret );
	}


	/// @brief DICOMt@CǂݍŁCSĂDICOM^Oe[uɓo^
	//! 
	//! @param[out] dicm  c DICOM^OɃf[^o^e[u
	//! @param[in]  filename c DICOM]t@C
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool read_dicom_tags( dicom_tag_container &dicm, const std::string &filename )
	{
		size_type filesize;
		FILE *fp;
		if( ( fp = fopen( filename.c_str( ), "rb" ) ) == NULL )
		{
			return( false );
		}

		// t@CTCY擾
		fseek( fp, 0, SEEK_END );
		filesize = ftell( fp );
		fseek( fp, 0, SEEK_SET );

		unsigned char *buff = new unsigned char[ filesize + 1 ];
		unsigned char *pointer = buff;
		size_type read_size = 0;
		while( feof( fp ) == 0 )
		{
			read_size = fread( pointer, sizeof( unsigned char ), 1024, fp );
			if( read_size < 1024 )
			{
				break;
			}
			pointer += read_size;
		}

		fclose( fp );

		bool ret = read_dicom_tags( dicm, buff, filesize, true );

		delete [] buff;

#ifdef __SHOW_DICOM_TAG__
		printf( "\n\n\n" );
#endif

		// K؂DICOMt@Cǂ`FbN
		if( ret )
		{
			dicom_tag_container::iterator ite = dicm.begin( );
			for( ; ite != dicm.end( ) ; ++ite )
			{
				if( ite->second.comment != "UNKNOWN" )
				{
					return( true );
				}
			}

			return( false );
		}
		else
		{
			return( false );
		}
	}


	/// @brief DICOM̃^OW Explicit VR `Ńt@Cɏo
	//! 
	//! @param[in]  group     c O[vID
	//! @param[in]  element   c GgID
	//! @param[in]  vr        c VR^O
	//! @param[in]  data      c f[^
	//! @param[in]  num_bytes c f[^̃oCg
	//! @param[out] fp    c o̓t@CXg[
	//! @param[in]  to_little_endian c o̓f[^̌`gGfBAɂ邩ǂ
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool write_dicom_tag_explicit_vr( unsigned short group, unsigned short element, dicom_vr vr, const unsigned char *data, size_t num_bytes, FILE *fp, bool to_little_endian = true )
	{
		unsigned char ZERO[] = { 0, 0, 0, 0 };
		unsigned char FFFF[] = { 0xff, 0xff, 0xff, 0xff };

		switch( vr )
		{
		case OW:
		case OF:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( get_dicom_vr( vr ).c_str( ), 1, 2, fp );
			fwrite( ZERO, 1, 2, fp );

			// f[^V[PXɂȂĂ邩𒲂ׁCo͕@؂ւ
			if( num_bytes > 4 + 8 && is_sequence_separate_tag( data, data + num_bytes ) && is_sequence_tag_end( data + num_bytes - 8, data + num_bytes ) )
			{
				fwrite( FFFF, 1, 4, fp );
			}
			else
			{
				fwrite( from_current_endian( byte_array< unsigned int >( static_cast< unsigned int >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 4, fp );
			}

			if( _is_little_endian_( ) == to_little_endian )
			{
				fwrite( data, 1, num_bytes, fp );
			}
			else
			{
				switch( vr )
				{
				case OW:
					if( num_bytes % 2 == 0 )
					{
						for( size_t i = 0 ; i < num_bytes ; i += 2 )
						{
							fwrite( from_current_endian( byte_array< unsigned short >( data + i ), to_little_endian ).get_bytes( ), 1, 2, fp );
						}
					}
					else
					{
						// f[^̃oCg
						return( false );
					}
					break;

				case OF:
				default:
					if( num_bytes % 4 == 0 )
					{
						for( size_t i = 0 ; i < num_bytes ; i += 4 )
						{
							fwrite( from_current_endian( byte_array< unsigned int >( data + i ), to_little_endian ).get_bytes( ), 1, 4, fp );
						}
					}
					else
					{
						// f[^̃oCg
						return( false );
					}
					break;
				}
			}
			break;

		case US:
		case SS:
		case AT:
		case UL:
		case SL:
		case FL:
		case FD:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( get_dicom_vr( vr ).c_str( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( static_cast< unsigned short >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 2, fp );
			if( _is_little_endian_( ) == to_little_endian )
			{
				fwrite( data, 1, num_bytes, fp );
			}
			else
			{
				switch( vr )
				{
				case US:
				case SS:
				case AT:
					if( num_bytes % 2 == 0 )
					{
						for( size_t i = 0 ; i < num_bytes ; i += 2 )
						{
							fwrite( from_current_endian( byte_array< unsigned short >( data + i ), to_little_endian ).get_bytes( ), 1, 2, fp );
						}
					}
					else
					{
						// f[^̃oCg
						return( false );
					}
					break;

				case UL:
				case SL:
				case FL:
					if( num_bytes % 4 == 0 )
					{
						for( size_t i = 0 ; i < num_bytes ; i += 4 )
						{
							fwrite( from_current_endian( byte_array< unsigned int >( data + i ), to_little_endian ).get_bytes( ), 1, 4, fp );
						}
					}
					else
					{
						// f[^̃oCg
						return( false );
					}
					break;

				case FD:
				default:
					if( num_bytes % 8 == 0 )
					{
						for( size_t i = 0 ; i < num_bytes ; i += 8 )
						{
							fwrite( from_current_endian( byte_array< double >( data + i ), to_little_endian ).get_bytes( ), 1, 8, fp );
						}
					}
					else
					{
						// f[^̃oCg
						return( false );
					}
					break;
				}
			}
			break;

		case OB:
		case UN:
		case UT:
		case SQ:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( get_dicom_vr( vr ).c_str( ), 1, 2, fp );
			fwrite( ZERO, 1, 2, fp );
			// f[^V[PXɂȂĂ邩𒲂ׁCo͕@؂ւ
			if( num_bytes > 4 + 8 && is_sequence_separate_tag( data, data + num_bytes ) && is_sequence_tag_end( data + num_bytes - 8, data + num_bytes ) )
			{
				fwrite( FFFF, 1, 4, fp );
				fwrite( data, 1, num_bytes, fp );
			}
			else
			{
				fwrite( from_current_endian( byte_array< unsigned int >( static_cast< unsigned int >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 4, fp );
				fwrite( data, 1, num_bytes, fp );
			}
			break;

		case UNKNOWN:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( "UN", 1, 2, fp );
			fwrite( get_dicom_vr( vr ).c_str( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned int >( static_cast< unsigned int >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 4, fp );
			fwrite( data, 1, num_bytes, fp );
			break;

		default:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( get_dicom_vr( vr ).c_str( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( static_cast< unsigned short >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( data, 1, num_bytes, fp );
			break;
		}

		return( true );
	}



	/// @brief DICOM̃^OW Implicit VR `Ńt@Cɏo
	//! 
	//! @param[in]  group     c O[vID
	//! @param[in]  element   c GgID
	//! @param[in]  vr        c VR^O
	//! @param[in]  data      c f[^
	//! @param[in]  num_bytes c f[^̃oCg
	//! @param[out] fp    c o̓t@CXg[
	//! @param[in]  to_little_endian c o̓f[^̌`gGfBAɂ邩ǂ
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool write_dicom_tag_implicit_vr( unsigned short group, unsigned short element, dicom_vr vr, const unsigned char *data, size_t num_bytes, FILE *fp, bool to_little_endian = true )
	{
		unsigned char FFFF[] = { 0xff, 0xff, 0xff, 0xff };

		switch( vr )
		{
		case OB:
		case UN:
		case UT:
		case SQ:
			// f[^V[PXɂȂĂ邩𒲂ׁCo͕@؂ւ
			if( num_bytes > 4 + 8 && is_sequence_separate_tag( data, data + num_bytes ) && is_sequence_tag_end( data + num_bytes - 8, data + num_bytes ) )
			{
				fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
				fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
				fwrite( FFFF, 1, 4, fp );
				fwrite( data, 1, num_bytes, fp );
			}
			else
			{
				fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
				fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
				fwrite( from_current_endian( byte_array< unsigned int >( static_cast< unsigned int >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 4, fp );
				fwrite( data, 1, num_bytes, fp );
			}
			break;

		default:
			fwrite( from_current_endian( byte_array< unsigned short >( group ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned short >( element ), to_little_endian ).get_bytes( ), 1, 2, fp );
			fwrite( from_current_endian( byte_array< unsigned int >( static_cast< unsigned int >( num_bytes ) ), to_little_endian ).get_bytes( ), 1, 4, fp );
			fwrite( data, 1, num_bytes, fp );
			break;
		}

		return( true );
	}

	/// @brief DICOM̃^OW Explicit VR `Ńt@Cɏo
	//! 
	//! @param[in]  e     c t@Cɏo͂DICOM^O
	//! @param[out] fp    c o̓t@CXg[
	//! @param[in]  to_little_endian c o̓f[^̌`gGfBAɂ邩ǂ
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool write_dicom_tag_explicit_vr( const dicom_element &e, FILE *fp, bool to_little_endian = true )
	{
		return( write_dicom_tag_explicit_vr( e.get_group( ), e.get_element( ), e.vr, e.data, e.num_bytes, fp, to_little_endian ) );
	}

	/// @brief DICOM̃^OW Implicit VR `Ńt@Cɏo
	//! 
	//! @param[in]  e     c t@Cɏo͂DICOM^O
	//! @param[out] fp    c o̓t@CXg[
	//! @param[in]  to_little_endian c o̓f[^̌`gGfBAɂ邩ǂ
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool write_dicom_tag_implicit_vr( const dicom_element &e, FILE *fp, bool to_little_endian = true )
	{
		return( write_dicom_tag_implicit_vr( e.get_group( ), e.get_element( ), e.vr, e.data, e.num_bytes, fp, to_little_endian ) );
	}


	/// @brief DICOM̃^Oo͂ۂɊe^OKvƂoCgvZ
	//! 
	//! @param[in] e          c DICOM^O
	//! @param[in] implicitVR c Implicit VR `ǂ
	//! 
	//! @return ^ÕoCg
	//! 
	inline size_t get_write_dicom_tag_size( const dicom_element &e, bool implicitVR )
	{
		if( implicitVR )
		{
			return( 8 + e.num_bytes );
		}
		else
		{
			size_t num_bytes = 0;

			switch( e.vr )
			{
			case OW:
			case OF:
			case OB:
			case UN:
			case SQ:
			case UT:
			case UNKNOWN:
				num_bytes = 12 + e.num_bytes;
				break;

			case US:
			case SS:
			case AT:
			case UL:
			case SL:
			case FL:
			case FD:
			default:
				num_bytes = 8 + e.num_bytes;
				break;
			}

			return( num_bytes );
		}
	}


	/// @brief DICOM Group Length vZC^Oɐݒ肷
	//! 
	//! @param[in,out] dicm       c DICOM^OɃf[^o^e[u
	//! @param[in]     implicitVR c Implicit VR `ǂ
	//! 
	inline void compute_group_length( dicom_tag_container &dicm, bool implicitVR )
	{
		std::map< unsigned short, size_t > group_length;
		dicom_tag_container::iterator ite;
		for( ite = dicm.begin( ) ; ite != dicm.end( ) ; ++ite )
		{
			const dicom_element &e = ite->second;
			if( e.enable )
			{
				unsigned short group     = e.get_group( );
				unsigned short element   = e.get_element( );
				// Group 0002  Explicit VR Little Endian Transfer Syntax ŃGR[hKv
				size_t num_bytes = get_write_dicom_tag_size( e, group == 0x0002 ? false : implicitVR );

				if( element != 0x0000 )
				{
					// Group Length ̃^O̓XLbv
					if( group_length.find( group ) != group_length.end( ) )
					{
						group_length[ group ] += num_bytes;
					}
					else
					{
						group_length[ group ] = num_bytes;
					}
				}
			}
		}

		for( ite = dicm.begin( ) ; ite != dicm.end( ) ; ++ite )
		{
			dicom_element &e = ite->second;
			unsigned short group   = e.get_group( );
			unsigned short element = e.get_element( );

			if( element == 0x0000 )
			{
				// Group Length ̃^OɃf[^ݒ肷
				size_t num_bytes = group_length[ group ];

				switch( e.num_bytes )
				{
				case 2:
					{
						byte_array< unsigned short > b( static_cast< unsigned short >( num_bytes ) );
						memcpy( e.data, b.get_bytes( ), e.num_bytes );
					}
					break;

				case 4:
				default:
					{
						byte_array< unsigned int > b( static_cast< unsigned int >( num_bytes ) );
						memcpy( e.data, b.get_bytes( ), e.num_bytes );
					}
					break;
				}
			}
		}
	}


	/// @brief DICOM̃^OWt@Cɏo
	//! 
	//! @param[out] dicom    c DICOM^OɃf[^o^e[u
	//! @param[in]  filename c oDICOMt@C
	//! 
	//! @retval true  c DICOMt@C̏ɐ
	//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
	//! 
	inline bool write_dicom_tags( const dicom_tag_container &dicom, const std::string &filename )
	{
		FILE *fp;
		if( ( fp = fopen( filename.c_str( ), "wb" ) ) == NULL )
		{
			return( false );
		}

		// DICOM̃vAũ^O݂邩ǂ𒲂ׂ
		bool hasPreamble = false;
		{
			unsigned short elements[] = { 0x0001, 0x0002, 0x0003, 0x0010, 0x0012, 0x0013, 0x0016, 0x0100, 0x0102 };
			size_t num = sizeof( elements ) / sizeof( unsigned short );
			for( size_t i = 0 ; i < num ; i++ )
			{
				if( dicom.contain( 0x0002, elements[ i ] ) )
				{
					// f[^݂̂ŃvAu𖄂ߍ
					hasPreamble = true;
					break;
				}
			}
		}

		dicom_tag_container dicm( dicom );
		bool implicitVR = true;

		if( hasPreamble )
		{
			// DICOM̃wb_
			unsigned char ZERO[ 128 ];
			unsigned char DICM[ 4 ];
			memset( ZERO, 0, sizeof( unsigned char ) * 128 );
			DICM[ 0 ] = 'D';
			DICM[ 1 ] = 'I';
			DICM[ 2 ] = 'C';
			DICM[ 3 ] = 'M';
			fwrite( ZERO, sizeof( unsigned char ), 128, fp );
			fwrite( DICM, sizeof( unsigned char ), 4, fp );


			// IVR̎w肪Ȃꍇ͒ǉ
			if( !dicm.contain( 0x0002, 0x0010 ) )
			{
				// Transfer Syntax UID ݂Ȃꍇ́CuImplicit VR Little Endianvw肷
				std::string syntax = "1.2.840.10008.1.2";
				dicm.append( dicom_element( 0x0002, 0x0010, reinterpret_cast< const unsigned char * >( syntax.c_str( ) ), syntax.length( ) ) );
			}
			else if( find_tag( dicm, 0x0002, 0x0010, "" ) == "1.2.840.10008.1.2.1" )
			{
				// uExplicit VR Little Endianvw肳Ăꍇ́CuExplicit VR Little EndianvɕύX
				implicitVR = false;
			}
			else if( find_tag( dicm, 0x0002, 0x0010, "" ) == "1.2.840.10008.1.2.2" )
			{
				// uImplicit VR Big Endianvw肳Ăꍇ́CuExplicit VR Little EndianvɕύX
				implicitVR = false;
				std::string syntax = "1.2.840.10008.1.2.1";
				dicm( 0x0002, 0x0010 ).copy( reinterpret_cast< const unsigned char * >( syntax.c_str( ) ), syntax.length( ) );
			}
			else if( get_compress_type( find_tag( dicm, 0x0002, 0x0010, "" ) ) != RAW )
			{
				// kĂꍇ́uExplicit VR Little EndianvɕύX
				implicitVR = false;
			}

			// Group 0002 ̒}ĂȂꍇ͑}
			if( !dicm.contain( 0x0002, 0x0000 ) )
			{
				byte_array< unsigned int > b( static_cast< unsigned int >( 0 ) );
				dicm.append( dicom_element( 0x0002, 0x0000, b.get_bytes( ), b.length( ) ) );
			}
		}

		// DICOM ̃^Ogpɏ悤ɂ
		dicm.update( );

		// Group Length 𐳂ݒ肷
		compute_group_length( dicm, implicitVR );

		// VRɉďo͂
		dicom_tag_container::const_iterator ite = dicm.begin( );
		for( ; ite != dicm.end( ) ; ++ite )
		{
			if( ite->second.enable )
			{
				// Group 0002  Explicit VR Little Endian Transfer Syntax ŃGR[hKv
				if( implicitVR && ite->second.get_group( ) != 0x0002 )
				{
					write_dicom_tag_implicit_vr( ite->second, fp, true );
				}
				else
				{
					write_dicom_tag_explicit_vr( ite->second, fp, true );
				}
			}
		}

		fclose( fp );

		return( true );
	}
}




/// @brief DICOMf[^摜Reiɓǂݍ
//! 
//! @note {ł́Csvȃ^O͑SĎ菜Ă܂߁CKvȃf[^ꍇ́C{֐QlɂĎ삵Ă
//! 
//! @param[out] image    c o͉摜
//! @param[in]  filename c ̓t@C
//! 
//! @retval true  c DICOMt@C̏ɐ
//! @retval false c DICOMt@Cł͂ȂCłȂ^OEf[^݂ꍇ
//! 
template < class T, class Allocator >
bool read_dicom( array2< T, Allocator > &image, const std::string &filename )
{
	typedef typename array2< T, Allocator >::value_type value_type;
	typedef typename array2< T, Allocator >::size_type size_type;
	typedef _pixel_converter_< T > pixel_converter;
	typedef typename pixel_converter::color_type color_type;

	dicom::dicom_tag_container dicm;
	dicom::dicom_info info;

	if( !dicom::read_dicom_tags( dicm, filename ) )
	{
		return( false );
	}

	dicom::get_dicom_info( dicm, info );

	double window_level = info.window_center;
	double window_width = info.window_width;

	if( dicm.contain( 0x7fe0, 0x0010 ) )
	{
		switch( info.compression_type )
		{
		case dicom::JPEG:
		case dicom::JPEGLS:
		case dicom::JPEG2000:
			// ̂Ƃ떢T|[g
			return( false );

		default:
			break;
		}

		dicom::dicom_element element = dicm( 0x7fe0, 0x0010 );
		if( !dicom::decode( element, info ) )
		{
			// k̉𓀂̓f[^
			return( false );
		}

		image.resize( info.cols, info.rows );
		image.reso1( info.pixel_spacing_x );
		image.reso2( info.pixel_spacing_y );

		switch( info.photometric_type )
		{
		case dicom::RGB:
			if( info.compression_type == dicom::RLE )
			{
				// RLEňkۂ́CFƂɈk݂
				for( size_type j = 0 ; j < image.height( ) ; j++ )
				{
					unsigned char *r = element.data + j * info.cols;
					unsigned char *g = element.data + j * info.cols + info.cols * info.rows;
					unsigned char *b = element.data + j * info.cols + info.cols * info.rows * 2;
					for( size_type i = 0 ; i < image.width( ) ; i++ )
					{
						image( i, j ) = pixel_converter::convert_to( r[ i ], g[ i ], b[ i ] );
					}
				}
			}
			else
			{
				for( size_type j = 0 ; j < image.height( ) ; j++ )
				{
					unsigned char *data = element.data + j * image.width( ) * 3;
					for( size_type i = 0 ; i < image.width( ) ; i++ )
					{
						size_type ii = i * 3;
						image( i, j ) = pixel_converter::convert_to( data[ ii + 2 ], data[ ii + 1 ], data[ ii + 0 ] );
					}
				}
			}
			break;

		case dicom::PALETTE_COLOR:
			if( dicm.contain( 0x0028, 0x1201 ) && dicm.contain( 0x0028, 0x1202 ) && dicm.contain( 0x0028, 0x1203 ) )
			{
				const dicom::dicom_element &red   = dicm( 0x0028, 0x1201 );
				const dicom::dicom_element &green = dicm( 0x0028, 0x1202 );
				const dicom::dicom_element &blue  = dicm( 0x0028, 0x1203 );

				const short *r = reinterpret_cast< const short * >( red.data );
				const short *g = reinterpret_cast< const short * >( green.data );
				const short *b = reinterpret_cast< const short * >( blue.data );

				bool is_big_endian = false;

				int max_value = ( 2 << info.bits_stored ) - 1;
				for( size_t i = 0 ; i < red.num_bytes / 2 ; i++ )
				{
					if( r[ i ] > max_value || g[ i ] > max_value || b[ i ] > max_value )
					{
						is_big_endian = true;
						break;
					}
				}

				if( is_big_endian )
				{
					// pbg̃f[^́CKrbOGfBAŃCvgĂ
					for( size_type j = 0 ; j < image.height( ) ; j++ )
					{
						unsigned char *data = element.data + j * image.width( );
						for( size_type i = 0 ; i < image.width( ) ; i++ )
						{
							unsigned char R = static_cast< unsigned char >( from_current_endian( byte_array< short >( r[ data[ i ] ] ), false ).get_value( ) );
							unsigned char G = static_cast< unsigned char >( from_current_endian( byte_array< short >( g[ data[ i ] ] ), false ).get_value( ) );
							unsigned char B = static_cast< unsigned char >( from_current_endian( byte_array< short >( b[ data[ i ] ] ), false ).get_value( ) );
							image( i, j ) = pixel_converter::convert_to( R, G, B );
						}
					}
				}
				else
				{
					// pbg̃f[^́CKrbOGfBAŃCvgĂ
					for( size_type j = 0 ; j < image.height( ) ; j++ )
					{
						unsigned char *data = element.data + j * image.width( );
						for( size_type i = 0 ; i < image.width( ) ; i++ )
						{
							unsigned char R = static_cast< unsigned char >( r[ data[ i ] ] );
							unsigned char G = static_cast< unsigned char >( g[ data[ i ] ] );
							unsigned char B = static_cast< unsigned char >( b[ data[ i ] ] );
							image( i, j ) = pixel_converter::convert_to( R, G, B );
						}
					}
				}
			}
			break;

		case dicom::MONOCHROME1:
		case dicom::MONOCHROME2:
		default:
			if( info.bits_allocated == 8 )
			{
				for( size_type j = 0 ; j < image.height( ) ; j++ )
				{
					unsigned char *data = element.data + j * image.width( );
					for( size_type i = 0 ; i < image.width( ) ; i++ )
					{
						image( i, j ) = data[ i ];
					}
				}
			}
			else if( info.bits_allocated == 16 )
			{
				// MONOCHROME1 ł͔wiftHg
				// MONOCHROME2 ł͔wiftHg
				if( info.photometric_type == dicom::MONOCHROME1 )
				{
					window_width = -window_width;
				}

				double offset = info.rescale_intercept;
				if( dicm.contain( 0x0028, 0x1050 ) && dicm.contain( 0x0028, 0x1051 ) && dicm( 0x0028, 0x1050 ).num_bytes > 0 && dicm( 0x0028, 0x1051 ).num_bytes > 0 )
				{
					const short *bytes = reinterpret_cast< const short * >( element.data );
					for( size_type j = 0 ; j < image.height( ) ; j++ )
					{
						const short *data = bytes + j * image.width( );
						for( size_type i = 0 ; i < image.width( ) ; i++ )
						{
							double pix = ( ( data[ i ] + offset - window_level ) / window_width + 0.5 ) * 255.0;
							pix = pix > 255.0 ? 255.0 : pix;
							pix = pix <   0.0 ?   0.0 : pix;
							image( i, j ) = static_cast< unsigned char >( pix );
						}
					}
				}
				else
				{
					// ܂C`ɗp͈͂肷
					const unsigned short *bytes = reinterpret_cast< const unsigned short * >( element.data );
					double min = bytes[ 0 ], max = bytes[ 0 ];
					for( size_t l = 1 ; l < element.num_bytes / 2 ; l++ )
					{
						if( bytes[ l ] < min )
						{
							min = bytes[ l ];
						}
						else if( max < bytes[ l ] )
						{
							max = bytes[ l ];
						}
					}

					double ww = max - min + 1;
					double wl = ( max + min ) / 2.0;

					// MONOCHROME1 ł͔wiftHg
					// MONOCHROME2 ł͔wiftHg
					if( info.photometric_type == dicom::MONOCHROME1 )
					{
						ww = -ww;
					}

					for( size_type j = 0 ; j < image.height( ) ; j++ )
					{
						const unsigned short *data = bytes + j * image.width( );
						for( size_type i = 0 ; i < image.width( ) ; i++ )
						{
							double pix = ( ( data[ i ] - wl ) / ww + 0.5 ) * 255.0;
							pix = pix > 255.0 ? 255.0 : pix;
							pix = pix <   0.0 ?   0.0 : pix;
							image( i, j ) = static_cast< unsigned char >( pix );
						}
					}
				}
			}
			else
			{
				return( false );
			}
			break;
		}
	}
	else
	{
		return( false );
	}

	return( true );
}


/// @}
//  DICOM摜o̓O[v̏I

/// @}
//  摜o̓O[v̏I


// mistOԂ̏I
_MIST_END


#endif // __INCLUDE_MIST_DICOM__
