# Copyright (c) 2021 Jone Bontus.
# All rights reserved.
# This component and the accompanying materials are made available
# under the terms of "Eclipse Public License v1.0"
# which accompanies this distribution, and is available
# at the URL "http://www.eclipse.org/legal/epl-v10.html".
#
# Initial Contributors:
# Jone Bontus.
#
# Contributors:
#
# Description:
# epoc32\tools\tranasm2.pl
# Transform assembler generated from CIA files
# Syntax:
# perl tranasm2.pl [--output=<file>] <file>
#
#

use strict;
use Getopt::Long;

sub croak ($);
sub frameGetLength ($);
sub frameProcessRegRange ($$);

my %Options;

unless (GetOptions(\%Options, 'output:s')) {
	exit 1;
}

unless (@ARGV==1) {
	die "perl tranasm2.pl [--output=<file>] <file>\n";
}
my $outfile = $Options{"output"} if $Options{"output"};
my $infile = $ARGV[0];
my $savedOut;

open IN, $infile or die "Can't open $infile for read\n";

if ($outfile) {
	open OUT, ">$outfile" or die "Can't open $outfile for write\n";
	$savedOut = select OUT;
}

my $line;
my @buffer;
my $sym;
my $funcname;
my $inside_function;
my $frame_seen;
my $frame_cfa;
my $frame_used;
my $frame_reg;

while ($line = shift @buffer or $line = <IN>) {
	if ($line =~ /^\s*\.type\s+(\S+),\s*(\S+)/) {
		$sym = $1;
		if ($2 eq "%function") {
			$funcname = $1;
		}
	} elsif ($line =~ /^\s*(\S+):/) {
		# Beginning of function
		if ($1 eq $funcname) {
			my $line2;
			while ($line2 = <IN>) {
				push @buffer, $line2;
				if ($line2 =~ /^\s*@\s+Naked\s+Function:/) {
					$inside_function = 1;
					last;
				} elsif ($line2 =~ /^\s*@\s+args\s+/) {
					last;
				}
			}
		}
	} elsif ($line =~ /^\s*\.size\s+(\S+),\s*\S+/) {
		# End of function
		if ($1 eq $funcname) {
			undef $funcname;
			undef $inside_function;
		}
	}

	if ($inside_function) {
		if ($line =~ /^\s*\.fnstart/) {
			$frame_cfa = 0;
			$frame_used = 0;
			$frame_reg = "sp";
		} elsif ($line =~ /^\s*\.fnend/) {
			if (!$frame_seen) {
				print "\t.cantunwind\n";
			}
			undef $frame_seen;
			undef $frame_reg;
		} elsif ($line =~ /^\s*\.cantunwind/) {
			$line = "";
		} elsif ($line =~ /^\s*@\s+[0-9]+\s+"\S+"\s+[0-9]+/) {
			# @ 78 "M:/sf/os/kernelhwsrv/kernel/eka/common/arm/atomics.cia" 1
			# Emitted inline assembly is in next line
			print $line;
			$line = <IN>;
			if ($line =~ /^\s*\.setfp\s+(\S+),\s*sp,\s*#(\S+)/) {
				$frame_seen = 1;
				$frame_cfa = -$2;
				$frame_reg = $1;
			} elsif ($line =~ /^\s*FRAME\s+ADDRESS\s+(\S+),\s*(\S+)/) {
				$frame_seen = 1;
				$frame_cfa = $2;
				$frame_reg = $1;
				$line = "\t.setfp $1, sp, #-$2\n";
			} elsif ($line =~ /^\s*FRAME\s+PUSH\s+{\s*(\S+),\s*(\S+)\s*}/) {
				$frame_seen = 1;
				my $frame_length = frameGetLength(lc $1) + frameGetLength(lc $2);
				if ($frame_cfa > 0) {
					print "\t.pad #$frame_cfa\n";
					$frame_used += $frame_cfa;
					$frame_cfa = 0;
				}
				$frame_used += $frame_length;
				print "\t.save {$1,$2}\n";
				$line = "\t.setfp $frame_reg, sp, #0\n";
			} elsif ($line =~ /^\s*FRAME\s+SAVE\s+{\s*(\S+)\s*},\s*(\S+)/) {
				$frame_seen = 1;
				my $frame_length = frameGetLength(lc $1);
				my $frame_reserve = -$2 - $frame_length - $frame_used;
				if ($frame_reserve < 0) {
					if ($frame_used > 0) {
						croak("Frame save used or reserved\n");
					}
					croak("Frame save goes beyond CFA\n");
				}
				if ($frame_reserve > 0) {
					print "\t.pad #$frame_reserve\n";
					$frame_used += $frame_reserve;
					$frame_cfa -= $frame_reserve;
				}
				$frame_used += $frame_length;
				$frame_cfa -= $frame_length;
				$line = "\t.save {$1}\n";
			}
		} elsif ($line =~ /^\s*\.word/) {
			# Impossible to go here for now
			$line = "";
		} elsif ($line =~ /^\s*nop($|\s+)/) {
			# No operation
			# Useless but does not harm
		} elsif ($line !~ /^\s*@/) {
			if ($line !~ /^\s*(\S+):/ and $line !~ /^\s*\.(\S+)/) {
				# Not a label nor pseudo op
				# Useless stuff generated by GCC
				$line = "";
			}
		}
	}

	print $line;
}

if ($outfile) {
	select $savedOut;
	close OUT;
}

close IN;

sub croak ($)
{
	my $msg = shift;
	print STDERR "Error: line $.: $msg";
	if ($outfile) {
		select $savedOut;
		close OUT;
		unlink $outfile;
	}
	close IN;
	exit 1;
}

sub frameGetLength ($)
{
	my $reg = shift;

	if ($reg =~ /(\S+)-(\S+)/) {
		return frameProcessRegRange($1, $2) * 4;
	}

	return 4;
}

sub frameProcessRegRange ($$)
{
	my ($reg1, $reg2) = @_;
	my %regIndex = ("sb" => 9,
		"sl" => 10,
		"fp" => 11,
		"ip" => 12,
		"sp" => 13,
		"lr" => 14,
		"pc" => 15
	);

	$reg1 = $regIndex{$reg1} ? $regIndex{$reg1} : substr($reg1, 1);
	$reg2 = $regIndex{$reg2} ? $regIndex{$reg2} : substr($reg2, 1);
	($reg1, $reg2) = $reg1 > $reg2 ? ($reg2, $reg1) : ($reg1, $reg2);

	return $reg2 - $reg1 + 1;
}
