package jp.nanah.bastub.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.ui.ModelMap;

import jp.nanah.bastub.BastubConsts;
import jp.nanah.bastub.data.FilterParam;
import jp.nanah.bastub.data.JsonInfo;
import jp.nanah.bastub.data.KvData;
import jp.nanah.bastub.data.UsingInfo;
import jp.nanah.bastub.util.BastubUtils;

@Service
public class JsonService {

	protected static final Logger logger = LoggerFactory.getLogger(JsonService.class);

	/**
	 * エクセルシートによるフィルタリングがないことを示すリスト。
	 * この値はオブジェクトの参照比較で使用する。
	 */
	public static final List<Row> NO_FILTERING_DATA_NONE = new ArrayList<Row>();

	public static final String JSON_DATA_NONE_TEXT = "{\"key\":\"auto created.\"}";

	@Value("${auto_create:false}")
	protected boolean isAutoCreate;

	@Autowired
	private PathService pathService;

	/**
	 * 【共通処理】ここが主処理。
	 *
	 * @param pathList
	 * @param body nullの場合あり
	 * @param model 未使用。将来削除するかも。
	 * @param req HTTPリクエスト
	 * @param res HTTPレスポンス
	 * @return
	 */
	public String getAny(List<String> pathList, Map body, ModelMap model, HttpServletRequest req, HttpServletResponse res){
		try {
			File pageDir = pathService.getPageDir();

			if (pageDir.exists() == false){
				String errormsg = "Bastubの設定フォルダ[" + pageDir.getAbsolutePath() + "]が見つかりません。application.propertiesのpagedata.rootを見直してください。";
				logger.info(errormsg);
				return errormsg;
			}

			UsingInfo ui = UsingInfo.getInitInstance(pathList, req, body);

			//パスを生成
			String path = StringUtils.join(pathList, "/");
			File dataFile = BastubUtils.getPagedataPath(pageDir, path, req.getMethod(), ".xlsx,.xls");
			File jsonFile = BastubUtils.getPagedataPath(pageDir, path, req.getMethod(), ".json");

			//データファイルを読み込む
			//-----------------------------------------------------------------------------------------------------
			//データファイルは、パス名＋"_"＋HTTPメソッド名のファイルがあればそれを優先する。ない場合はパス名のみ。
			//データファイルはファイルなし、あるいはファイル内のdataシートやfilterシートが無くてもOK。
			//==>データファイルがなければデータ埋め込みされないのでJSONは固定値で返る。
			//-----------------------------------------------------------------------------------------------------
			Sheet paramSheet = null;
			Sheet dataSheet = null;
			Workbook wb = readWorkbook(dataFile);
			if (wb == null) {
				logger.warn("Excelファイル無しのためJSONを固定で応答 (ExcelFile=[{}])", dataFile.getAbsolutePath());
			} else {
				paramSheet = wb.getSheet("filter");
				dataSheet = wb.getSheet("data");
				wb.close();

				if (paramSheet == null) {
					logger.debug("定義ファイル[{}]に[filter]シートが見つかりません", dataFile.getAbsolutePath());
				}
				if (dataSheet == null) {
					logger.debug("定義ファイル[{}]に[data]シートが見つかりません", dataFile.getAbsolutePath());
				}
			}
			//データ範囲をトリミング(エクセルはデータがなくても編集範囲だったりするので)
			trimDataSheet(dataSheet);

			//=================================
			// [1] リクエスト値をKeyValueリスト化
			//=================================
			List<KvData> requestData = getRequestData(req.getParameterMap(), body);

			//=================================
			// [2] 絞り込み条件を取得
			//=================================
			List<FilterParam> allFilter = getFilterParamList(paramSheet);
			List<FilterParam> validFilter = null;
			if (allFilter != null){
				validFilter = getValidFilter(allFilter, requestData, pathList);
			}

			//=================================
			// [3] エクセルデータから応答に返すレコードを絞り込む
			//=================================
			List<Row> resultTarget = pickupSheetData(dataSheet, validFilter);

			//=================================
			// [4] 応答用の入れ物を読み込む
			//=================================
			JsonInfo jsonInfo = readJsonFile(jsonFile);
			if (jsonInfo.getJsonObject() == null) {
				//ファイルがない場合は自動で作られるのでここを通過することは殆どない
				res.setStatus(HttpServletResponse.SC_NOT_FOUND);
				String jsonPath = jsonFile.getAbsolutePath();
				logger.warn("## ◆ 応答JSON異常: 理由=[{}] パス=[{}]", jsonInfo.getErrorMessage(), jsonPath);
				//String errorJson = "{ \"no_file\": \"" + jsonPath.replaceAll("\\\\", "/") + "\"}";
				String errorJson = "no_file - " + jsonPath.replaceAll("\\\\", "/") + "]";
				return errorJson;
			}

			//=================================
			// [5] 応答データを生成
			//=================================
			setDataToJsonObject(null, jsonInfo.getJsonObject(), resultTarget, ui);

			//応答
			String jsonText = jsonInfo.getJsonObject().toString(4);
			if (jsonInfo.isTopArray()){
				//{"dummy":xxxx  … } のカッコを除去する
				int n1 = jsonText.indexOf(":");
				int n2 = jsonText.lastIndexOf("}");
				jsonText = jsonText.substring(n1+1, n2);
			}

			//Edgeから直接リクエストするとエラーになることへの対応
			res.addHeader("Access-Control-Allow-Origin", "*");

			return jsonText;

		} catch (Throwable th) {
			logger.info("処理異常: {}", th);
			return th.toString();
		}

	}

	//-------------------
	// Excelデータの処理
	//-------------------

	/**
	 * エクセルファイルごと保存。
	 */
	private Map<String, Object[]> workbookCache = new TreeMap<String, Object[]>();

	/**
	 * Workbookを読み込む。
	 * キャッシュがあればそれを返す。
	 *
	 * @param file
	 * @return
	 * @throws IOException
	 */
	protected synchronized Workbook readWorkbook(File file) throws IOException {
		if (file.exists() == false) {
			return null;
		}

		String key = file.getAbsolutePath();
		Object[] cache = workbookCache.get(key);
		if (cache != null){
			Long filetime = (Long)cache[0];
			if (filetime.longValue() == file.lastModified()){
				return (Workbook)cache[1];
			}
		}

		Workbook wb =  WorkbookFactory.create(file, null, true);//new HSSFWorkbook(poiFileSystem);
		workbookCache.put(key, new Object[]{file.lastModified(), wb});
		return wb;
	}

	protected void trimDataSheet(Sheet sheet){
		if (sheet == null || sheet.getLastRowNum() == 0){
			return;
		}

		//１行目(ヘッダ行)から、列数を確認
		Row topRow = sheet.getRow(0);
		int lastColNum = topRow.getLastCellNum() - 1;
		while (lastColNum >= 0){
			Cell cell = topRow.getCell(lastColNum);
			String s = BastubUtils.getCellText(cell);
			if (StringUtils.isNotBlank(s)){
				break;
			}
			lastColNum--;
		}

		int rowNum = sheet.getLastRowNum();
		while (rowNum >= 0) {
			Row row = sheet.getRow(rowNum);
			if (row != null){
				if (BastubUtils.isBlankRow(row)){
					sheet.removeRow(row);
				} else {
					for (int i=row.getLastCellNum()-1; i>lastColNum; i--){
						Cell cell = row.getCell(i);
						if (cell != null){
							row.removeCell(cell);
						}
					}
				}
			}
			rowNum--;
		}
	}

	/**
	 * [1]
	 * @param paramMap
	 * @param body
	 * @return
	 */
	public List<KvData> getRequestData(Map<String, String[]> paramMap, Map body){
		List<KvData> paramList = new ArrayList<KvData>();

		//URLの後ろに渡されるリクエストパラメータをKey/Value形式にする
		if (paramMap != null) {
			int cnt = 0;
			for (Map.Entry<String, String[]> ent : paramMap.entrySet()) {
				//BODYで渡していてもParamMapに入っているので除外する。
				if (BastubUtils.isJsonEntry(ent)){
					continue;
				}
				List<String> key = Arrays.asList(new String[] {ent.getKey()});
				for (String val : ent.getValue()) {
					KvData kv = appendKvParam(new KvData(key, val), paramList);
					cnt++;
				}
			}
		}

		//BodyリクエストをKey/Value形式にする
		if (body != null) {
			int startSize = paramList.size();
			appendEntryToParamList(body.entrySet(), null, paramList);
			for (int i=startSize; i<paramList.size(); i++){
				KvData kv = paramList.get(i);
			}
		}

		return paramList;
	}

	//■TODO 見直し候補
	//ここの考え方は見直したほうがよい。
	//リクエスト(QueryStringやBODY)の取得時点ではキー値は重複していてよいし、
	//key1/key2で組み合わせのときに一致する考え方でないとうまくいかなくなる。
	//クラス化してあるべきデータ保持形式にすべき。

	/**
	 * キーが一意になるようにKvDataを追加する。
	 * リクエストやJSONで同一キーがある場合、どれかが一致するような判断となるようにする。
	 *
	 * @param src
	 * @param dstList
	 */
	private KvData appendKvParam(KvData src, List<KvData> dstList){
		KvData nowval = null;
		for (int i=0; i<dstList.size(); i++){
			if (BastubUtils.equals(dstList.get(i).getKey(), src.getKey())){
				nowval = dstList.get(i);
				break;
			}
		}

		if (nowval == null){
			dstList.add(src);
			nowval = src;

		} else {
			nowval.addValue(src.getValueAsOne());
		}
		return nowval;
	}

	private void appendEntryToParamList(Set<Map.Entry<String, Object>> set, List<String> parentKeys, List<KvData> paramList){
		int count = 0;
		for (Map.Entry<String, Object> ent : set) {
			String key = ent.getKey();
			Object val = (ent.getValue() == null) ? "" : ent.getValue();

			List<String> thisKeys = new ArrayList<String>();
			if (parentKeys != null){
				thisKeys.addAll(parentKeys);
			}
			thisKeys.add(key);

			if (val instanceof Map){

				Map<String, Object> vmap = (Map<String, Object>)val;
				appendEntryToParamList(vmap.entrySet(), thisKeys, paramList);

			} else if (val instanceof List){

				List<Object> vlist = (List<Object>)val;
				appendListToParamList(vlist, thisKeys, paramList);

			} else {
				appendToParamList(val, thisKeys, paramList);
			}
			count++;
		}
	}

	private void appendListToParamList(List<Object> list, List<String> parentKeys, List<KvData> paramList){
		for (Object o : list){
			if (o instanceof Map){
				Map<String, Object> map = (Map<String, Object>)o;
				appendEntryToParamList(map.entrySet(), parentKeys, paramList);
			} else if (o instanceof List){
				List<Object> vlist = (List<Object>)o;
				appendListToParamList(vlist, parentKeys, paramList);
			} else {
				appendToParamList(o, parentKeys, paramList);
			}
		}
	}

	private void appendToParamList(Object obj, List<String> parentKeys, List<KvData> paramList){
		String valstr = obj.toString();
		KvData kv = new KvData(parentKeys, valstr);
		appendKvParam(kv, paramList);
	}

	/**
	 * [2]
	 * エクセルデータをフィルタリングするときの定義パラメータを取得する。
	 * ここでは置き換え文字や接尾辞を加工できないので設定値をそのまま取り出す(置き換えない)
	 *
	 * @param sheet
	 * @return Map<Cellの列名, フィルタリングデータのリスト>
	 */
	private List<FilterParam> getFilterParamList(Sheet sheet){
		List<FilterParam> filterList = null;
		if (sheet == null) {
			return filterList;
		}

		for (int i=0; i<=sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			List<String> cellList = BastubUtils.getRowValueList(row, false);

			//String prevName = "";
			if (cellList.size() >= 3) {
				// １行でもフィルタリングデータがあるときだけフィルタリングを適用する
				if (filterList == null){
					filterList =new ArrayList<FilterParam>();
				}
				String cellName = cellList.get(0);
				String compType = cellList.get(1);
				List<String> paramKey = cellList.subList(2, cellList.size());
				FilterParam fp = new FilterParam(cellName, compType, paramKey);

				filterList.add(fp);
			}
		}
		return filterList;
	}

	/**
	 * [2]-2
	 * フィルタリング時に判定する、データシートの列値と比較する判定値を設定する。
	 * フィルタリングで使用するキー値がrequestDataにあるときだけ、requestDataの値をフィルタリングに採用する。
	 * 判定値がないときはnullのまま。
	 *
	 * @param requestData
	 * @param filterParamList
	 */
	public List<FilterParam> getValidFilter(List<FilterParam> filterParamList, List<KvData> requestData, List<String> pathList) {
		List<FilterParam> validList = new ArrayList<FilterParam>();
		for (FilterParam fp : filterParamList) {
			//パスインデックスのときは、URLのパス値に置き換える
			if (fp.getRequestKeys().size() > 0) {
				String key0 = fp.getRequestKeys().get(0);
				if (key0.startsWith(BastubConsts.PATH_REPLACE_HEAD)) {
					int pathNumber = NumberUtils.toInt(key0.substring(1));
					if (pathNumber > 0 && pathNumber <= pathList.size()) {
						fp.setOneValue(pathList.get(pathNumber - 1));
						validList.add(fp);
						continue;
					}
				}
			}

			//要求キーのときは要求パラメータから値を探す
			List<String> reqKey = fp.getRequestKeys(); //こっちは定義

			//実際のリクエストのキー値と比較
			boolean isFound = false;
			String delimText = getDelimText(fp.getCompareType());
			for (KvData kv : requestData) {
				if (BastubUtils.equalsTail(kv.getKey(), reqKey)) {
					List<String> fpValues = toFilterValue(delimText, kv.getValues());
					fp.setValues(fpValues);
					validList.add(fp);
					isFound = true;
					//break;
				}
			}
			if (isFound == false){
				logger.warn("HTTPリクエスト内に、項目値[{}]が見つかりません。", reqKey);
			}
			if (delimText != null){
				fp.setCompareType("=");
			}
		}
		return validList;
	}

	private String getDelimText(String compareType){
		int n1 = compareType.indexOf("[");
		int n2 = compareType.indexOf("]");
		if (n1 <0 || n2 < 0){
			return null;
		}
		String delim = compareType.substring(n1+1, n2);
		return delim;
	}

	// 記号が=[x]のとき、値を分割する
	private List<String> toFilterValue(String delimText, List<String> org){
		if (delimText == null){
			return org;
		}

		List<String> dst = new ArrayList<String>();
		for (String s: org){
			dst.addAll(Arrays.asList((String[]) s.split(delimText, 0)));
		}
		return dst;
	}

	/**
	 * [3]
	 */
	public List<Row> pickupSheetData(Sheet sheet, List<FilterParam> validFilter) {
		List<Row> noDatasheet = new ArrayList<Row>();
		if (sheet == null) {
			return NO_FILTERING_DATA_NONE;
		}

		//まずは全データを対象にする
		List<Row> sheetData = new ArrayList<Row>();
		for (int i=sheet.getFirstRowNum(); i<=sheet.getLastRowNum(); i++) {
			Row row = sheet.getRow(i);
			if (row != null){
				sheetData.add(sheet.getRow(i));
			}
		}

		// フィルターがないときは全データ対象
		if (validFilter == null || validFilter.isEmpty()){
			return sheetData;
		}

		//列名
		String[] columnNames = getColumnNames(sheet);

		//フィルターがあるときは適合しないものを除去する
		for (FilterParam fp : validFilter) {
			int columnIndex = ArrayUtils.indexOf(columnNames, fp.getColumnName());
			if (columnIndex < 0) {
				logger.warn("***** dataシートに、列名[{}]がありません。", fp.getColumnName());
				continue;
			}

			//↓先頭行は列名なので除外する
			for (int i=1; i<sheetData.size(); i++) {
				Row row = sheetData.get(i);
				if (row == null) {
					continue;
				}

				Cell cell = row.getCell(columnIndex);
				String v = BastubUtils.getCellText(cell);
				if (v == null){
					continue;
				}

				boolean isMatch = false;
				String ct = fp.getCompareType();
				for (String fv : fp.getValues()){

					int n;
					if (StringUtils.isEmpty(fv) && ct.equals("==") == false){
						//条件が未指定のとき、厳密な等価判定以外は条件なしとして扱う
						n = 0;
					} else {
						if (StringUtils.isNumeric(v) && StringUtils.isNumeric(fv)){
							logger.info("数値変換[{}]: [{}]<-->[{}]", columnIndex, v, fv);
							Integer v1 = Integer.parseInt(v);
							Integer v2 = Integer.parseInt(fv);
							n = v1.compareTo(v2);
						} else {
							n = v.compareTo(fv);
						}
					}

					if (ct.equals("=") || ct.equals("==")) {
						isMatch = (n == 0);
					} else if (ct.equals("<")) {
						isMatch = n < 0;
					} else if (ct.equals("<=")) {
						isMatch = n <= 0;
					} else if (ct.equals(">")) {
						isMatch = n > 0;
					} else if (ct.equals(">=")) {
						isMatch = n >= 0;
					} else if (ct.equals("!=") || ct.equals("<>")) {
						isMatch = n != 0;
					}
					if (isMatch){
						break;
					}
				}

				if (isMatch == false) {
					sheetData.set(i, null);
				}
			}
		}

		sheetData.removeAll(Collections.singleton(null));

		//debug
		for (Row row : sheetData) {
			StringBuilder sb = new StringBuilder();
			for (int i=row.getFirstCellNum(); i<row.getLastCellNum(); i++) {
				if (sb.length() > 0) {
					sb.append(",");
				}
				sb.append(BastubUtils.getCellText(row.getCell(i)));
			}
		}

		return sheetData;
	}

	private String[] getColumnNames(Sheet sheet) {
		Row row = sheet.getRow(sheet.getFirstRowNum());
		String[] columns = new String[row.getLastCellNum()];
		for (int i=row.getFirstCellNum(); i<row.getLastCellNum(); i++) {
			columns[i] = BastubUtils.getCellText(row.getCell(i));
		}
		return columns;
	}





















































	/**
	 * [4]
	 * @param file
	 * @return
	 * @throws IOException
	 */
	public JsonInfo readJsonFile(File file) {
		JsonInfo jsonInfo = new JsonInfo();

		//ファイルがなかったら新規に作る
		if (file.exists() == false) {
			file.getParentFile().mkdirs();
			jsonInfo.setErrorMessage("JSONファイルが未設定です");
			try {
				if (isAutoCreate){
					FileUtils.writeByteArrayToFile(file, JSON_DATA_NONE_TEXT.getBytes());
					//logger.info("応答JSON - 自動生成[正常]: path=[{}]", file.getAbsolutePath());
					file.setLastModified(0); //自動生成であることがわかるようにするため
				}
			} catch (Exception e){
				logger.info("応答JSON - 自動生成 [失敗]: path=[{}]", file.getAbsolutePath());
			}
			return jsonInfo;
		}

		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			byte[] buff = new byte[(int)file.length()];
			IOUtils.readFully(fis, buff);
			String text = new String(buff, "UTF-8");

			//自動生成したファイルと同じなら未設定と判断する
			if (text.equals(JSON_DATA_NONE_TEXT)){
				jsonInfo.setErrorMessage("JSONファイルが未設定です");
				return jsonInfo;
			}

			//text = BastubUtils.replaceEmbedText(text);

			//${xxxx}の前後がダブルクォーテーションでくくっていないときは
			//数値化指定(#N)付きの文字列にする。
			String reptext = adjustReplaceNumberValue(text);

			String trimText = reptext.trim();
			if (trimText.substring(0,1).equals("[")){
				reptext = "{\"dummy\":" + reptext + "}";
				jsonInfo.setTopArray(true);
			}
			try {
				JSONObject jsonObject = new JSONObject( reptext );
				jsonInfo.setJsonObject(jsonObject);
				logger.debug("reptext ==[{}]", reptext);
				logger.debug("逆変換  ==[{}]", jsonInfo.getJsonObject().toString(2));
			} catch (Throwable th){
				jsonInfo.setErrorMessage("JSON解析に失敗しました。書式が正しくない可能性があります。");
			}

		} catch (Throwable th) {
			logger.warn("応答ファイル読み込み失敗: path=[{}]", file.getAbsolutePath(), th);

		} finally {
			IOUtils.closeQuietly(fis);
		}

		return jsonInfo;
	}

	/**
	 * JSONの中に数値の置き換え文字があったら、その前後にダブルクォーテーションをつけて
	 * 文字列データにする。そうしないとJSONObjectがJSONと認識しない。
	 *
	 * @param org
	 * @param numKeys
	 * @return
	 */
	private String adjustReplaceNumberValue(String org) {
		int pos = 0;
		String dst = org; //org.replaceAll("\r|\n", "");

		while (pos < dst.length()) {
			int top = dst.indexOf("${", pos);
			if (top < 0) {
				break;
			}

			int last = BastubUtils.getNumberLast(dst, top);
			if (last >= 0) {
				String v = dst.substring(top, last) + "#N";
				dst = dst.substring(0, top) + "\"" + v + "\"" + dst.substring(last);
				pos = last + 1;
			} else {
				pos = pos + 1;
			}
		}
		return dst;
	}

	/**
	 * [5]
	 *
	 *
	 * @param keyList
	 * @param jsonObj
	 * @param dataList
	 */
	public void setDataToJsonObject(List<String> keyList, JSONObject jsonObj, List<Row> dataList, UsingInfo ui) {
		if (keyList == null){
			keyList = new ArrayList<String>();
		}
		if (dataList == null){
			dataList = new ArrayList<Row>();
		}

		// 【走査１】
		// １．直値で置き換え文字なら、
		// 　　　・1行目から順に、未使用データを適用する。
		// 　　　　つまりすでに適用していれば2行目、3行目のデータを使う。
		// 　　　・1行目データならKey/ValueをHashMapに記録。
		//       ※直値でない場合は特に何もしない。走査２で処理する。

		UsingInfo ud = new UsingInfo(dataList, ui);

		//置換文字が２か所で使われたとき同じ結果を返すようにするため
		List<String> allkeys = BastubUtils.sortIterator(jsonObj.keys());

		for (String key : allkeys){

			//JSONのキーを取得
			if (key == null){
				continue;
			}

			if (jsonObj.optJSONArray(key) != null) {
				//配列はあとで処理
				continue;
			} else if (jsonObj.optJSONObject(key) != null) {
				//JSONオブジェクトのときもあとで処理
				continue;
			} else {
				//直値ならここで置換
				String orgval = jsonObj.optString(key);
				if (orgval != null){
					try {
						String dstval = ud.getDirectValue(keyList, key, orgval);
						//nullを設定するとキーがremoveされてしまうのでここはnullじゃない前提。
						jsonObj.put(key, dstval);
					} catch (Exception e){
						logger.warn("置換異常", e);
					}
				}
			}
		}

		// 【走査２】
		// 処理に入る前に、操作１で置き換えたHashMapに合致したデータに絞り込む。
		// 1.配列なら、配列長が１かそれ以上かを見る。	//実値のとき、置き換え文字だったら置き換える。
		// 　　配列長＝１なら、絞り込みデータN件１つ１つで展開する。	//置き換えは先頭行から。一度使った行は使わず2行目、3行目…を適用する。
		// 　　展開結果を結果を文字列で受け取り、その文字列がユニークなら配列に登録、既出なら無視して、次のデータを処理する。	//
        // 2.JSONオブジェクトなら無条件で再帰呼び出し。データは絞り込み済のデータを渡す。
        // 3. 配列長＞１なら、1つ目の要素は1つ目の要素、2つ目の要素は2つ目とし、配列長だけ実施する。

		List<Row> nextLayerTarget = ud.getLoweredData();
		if (nextLayerTarget == null){
			return;
		}

		for (String key : allkeys){

			//JSONのキーを取得
			if (key == null){
				continue;
			}
			List<String> nextKeyList = new ArrayList<String>(keyList);
			nextKeyList.add(key);

			if (jsonObj.optJSONArray(key) != null) {

				JSONArray jsonAry = jsonObj.optJSONArray(key);
				JSONArray extendAry = getExtendJsonArray(nextKeyList, jsonAry, nextLayerTarget, ui);
				try {
					//展開したJSONを元の配列データと置き換える
					jsonObj.put(key, extendAry);
				} catch (Throwable th){
					logger.warn("配列データ設定異常", th);
				}

			} else if (jsonObj.optJSONObject(key) != null) {

				JSONObject nextJson = jsonObj.optJSONObject(key);
				setDataToJsonObject(nextKeyList, nextJson, nextLayerTarget, ui);

			}
		}

	}

	/**
	 *
	 *
	 * @param templateJson
	 * @param resultTarget
	 * @return
	 */
	public JSONArray getExtendJsonArray(List<String> keyList, JSONArray jsonAry, List<Row> dataList, UsingInfo ui) {
		JSONArray extendAry = new JSONArray();
		UsingInfo ud = new UsingInfo(dataList, ui);
		for (int i=0; i<jsonAry.length(); i++){

			try {
				if (jsonAry.optJSONObject(i) != null){
					//JSONObjectのとき
					String baseText = jsonAry.getJSONObject(i).toString();
					List<String> extendedList = new ArrayList<String>();
					if (dataList != NO_FILTERING_DATA_NONE){
						for (int r=1; r<dataList.size(); r++){
							List<Row> lowerDataList = new ArrayList<Row>();
							lowerDataList.add(dataList.get(0));
							lowerDataList.add(dataList.get(r));
							JSONObject extendJson = new JSONObject(baseText);
							setDataToJsonObject(keyList, extendJson, lowerDataList, ui);
							String extendText = extendJson.toString();
							if (extendedList.contains(extendText) == false){
								extendedList.add(extendText);
								extendAry.put(extendAry.length(), extendJson);
							}
						}
					} else {
						// フィルターデータがそもそも未指定のときは通過させる
						JSONObject extendJson = new JSONObject(baseText);
						setDataToJsonObject(keyList, extendJson, dataList, ui);
						extendAry.put(extendAry.length(), extendJson);
					}
					//配列の中を直接書き換えるからこれでOK

				} else if (jsonAry.optJSONArray(i) != null){
					//配列のとき
					JSONArray thisAry = jsonAry.getJSONArray(i);
					JSONArray childAry = getExtendJsonArray(keyList, thisAry, dataList, ui);
					extendAry.put(extendAry.length(), childAry);
				} else {
					//実値のとき
					String orgval = jsonAry.getString(i);
					if (orgval != null){
						try {
							String dstval = ud.getDirectValue(keyList, "", orgval);
							//nullを設定するとキーがremoveされてしまうのでここはnullじゃない前提。
							extendAry.put(extendAry.length(), dstval);
						} catch (Exception e){
							logger.warn("置換異常", e);
						}
					}
				}
			} catch (Throwable th){
				logger.warn("応答データ生成異常",th);
			}
		}

		return extendAry;
	}
}
