/*
 * cppds
 * kdtree.h
 * KDTree class
 * Copylight (C) 2008,2009 mocchi
 * mocchi_2003@yahoo.co.jp
 * License: Boost ver.1
 */

#include <cstdlib>
#include <stack>
#include <list>
#include <vector>
#include <cmath>
#include <limits>
#include <map>
#include <algorithm>

#ifndef NOMINMAX
#define NOMINMAX
#endif

namespace cppds{
template <typename T, int N, typename P = double> class kdtree{
public:
	class key_type{
		P pos[N];
	public:
		key_type(){
			for (int i = 0; i < N; ++i) pos[i] = P();
		}
		key_type(const P p[N]){
			for (int i = 0; i < N; ++i) pos[i] = p[i];
		}
		key_type(const key_type &k){
			for (int i = 0; i < N; ++i) pos[i] = k.pos[i];
		}
		operator const P *() const{
			return pos;
		}
		key_type &operator =(const key_type &k){
			for (int i = 0; i < N; ++i) pos[i] = k.pos[i];
			return *this;
		}
		friend class KDNode;
	};
	// ======== typedef of kdtree ========
	typedef size_t size_type;
	typedef T mapped_type;
	typedef ptrdiff_t difference_type;
	typedef std::pair<const key_type, mapped_type> value_type;
	typedef value_type* pointer;
	typedef const value_type* const_pointer;
	typedef value_type& reference;
	typedef const value_type& const_reference;
	// ======== typedef of kdtree ========

	static const int DIM = N;
private:
	static const int DIRSTART = 0;
	struct KDNode{
		value_type pos_value;
	private:
		KDNode *left, *right;
	public:
		KDNode(const value_type &pv) : pos_value(pv), left(0), right(0){
		}
		KDNode(const P pos_[N], const T &value_) : pos_value(key_type(pos_), value_), left(0), right(0){
		}
		friend class kdtree;
	};
	static inline int rotate(int prev){
		++prev;
		if (prev == N) prev = 0;
		return prev;
	}
	static inline int invrotate(int prev){
		--prev;
		if (prev < 0) prev = N - 1;
		return prev;
	}

	static inline bool same_pos(const P *p1, const P *p2){
		for (int i = 0; i < N; ++i){
			if (p1[i] != p2[i]) return false;
		}
		return true;
	}

	typedef std::stack<KDNode *> stack_t;

	// ======== member variable ========
	size_t count_nodes;
	KDNode *root;
	// ======== member variable ========
public:
	class iterator_io{
		KDNode *current;
		iterator_io(KDNode *current_) : current(current_){
		}
		bool is_end(){
			return current == NULL;
		}
		friend class kdtree;
	public:
		typedef const value_type& const_reference;
		typedef value_type& reference;
		typedef value_type* pointer;
		typedef const value_type* const_pointer;
		typedef size_t size_type;
		iterator_io() : current(0){
		}
		reference operator *(){
			return current->pos_value;
		}

		pointer operator->(){
			return &current->pos_value;
		}

		const_reference operator *() const{
			return current->pos_value;
		}

		const_pointer operator->() const{
			return &current->pos_value;
		}
		bool operator ==( const iterator_io &iter ) const {
			return (current == iter.current);
		}

		bool operator !=( const iterator_io &iter ) const {
			return (current != iter.current);
		}
	};
	template <typename Algo> class iterator_abstract{
	protected:
		stack_t buffer;
		KDNode *root;
		Algo algo;
		KDNode *current, *prev;
		int dir;
		iterator_abstract(KDNode *root_, const Algo &algo_) : root(root_), algo(algo_){
			dir = 0;
			current = root;
			prev = current;
			if (root && !algo.is_point_inrange(root)) next();
		}

		KDNode * get_current_node(){
			return current;
		}

		inline void next(){
			bool going_up = false;
			while(current){
				algo.prepare(current, dir);
				if ((!going_up) && algo.is_nearleaf_inrange()){
					buffer.push(current);
					prev = current;
					current = algo.node_near;
					dir = rotate(dir);
					going_up = false;
				}else if ((!going_up || (prev != NULL && prev == algo.node_near)) && algo.is_farleaf_inrange()){
					buffer.push(current);
					prev = current;
					current = algo.node_far;
					dir = rotate(dir);
					going_up = false;
				}else{
					if (buffer.size() == 0){
						current = prev = NULL;
						break;
					}
					prev = current;
					current = buffer.top();
					buffer.pop();
					going_up = true;
					dir = invrotate(dir);
				}
				if ((!going_up) && algo.is_point_inrange(current)){
					return;
				}
			}
		}

		bool is_end(){
			return current == NULL;
		}

	public:
		typedef ptrdiff_t difference_type;
		typedef const value_type& const_reference;
		typedef value_type& reference;
		typedef value_type* pointer;
		typedef const value_type* const_pointer;
		typedef size_t size_type;

		iterator_abstract(const Algo &algo_) : algo(algo_){
			buffer = stack_t();
			dir = 0;
			current = NULL;
			prev = NULL;
		}

		iterator_abstract(const iterator_abstract &instance) : algo(instance.algo){
			buffer = instance.buffer;
			root = instance.root;
			dir = instance.dir;
			current = instance.current;
			prev = instance.prev;
		}

		reference operator *(){
			return current->pos_value;
		}

		pointer operator->(){
			return &current->pos_value;
		}

		const_reference operator *() const{
			return current->pos_value;
		}

		const_pointer operator->() const{
			return &current->pos_value;
		}

		iterator_abstract & operator ++(){
			next();
			return *this;
		}

		const iterator_abstract operator ++(int){
			iterator_abstract cpy(*this);
			++(*this);
			return cpy;
		}

		bool operator ==( const iterator_abstract &iter ) const {
			if (current == NULL && iter.current == NULL) return true; // current == NULL  algoɊ֌WȂ end Ƃ݂Ȃ
			if (current == iter.current && root == iter.root &&
				buffer.size() == iter.buffer.size() && algo == iter.algo){
				if (buffer.size() == 0 || buffer.top() == iter.buffer.top()) return true;
			}
			return false;
		}

		bool operator !=( const iterator_abstract &iter ) const {
			return (*this == iter) ? false : true;
		}
		friend class kdtree;
	};

	struct algo_all{
		KDNode *node_near, *node_far;
		void prepare(KDNode *node, int dir){
			node_near = node->left;
			node_far = node->right;
		}

		inline bool is_nearleaf_inrange() const {
			return (node_near != NULL);
		}
		inline bool is_farleaf_inrange() const {
			return (node_far != NULL);
		}
		bool is_point_inrange(KDNode *node) const {
			return true;
		}
		bool operator == (const algo_all &instance) const{
			return true;
		}
	};

	struct algo_eukleide_range{
		double distance2;
		double distance2_at_is_point_inrange;
		P pos[N];
		P dx2;
		KDNode *node_near, *node_far;
		inline algo_eukleide_range(){
			dx2 = 0;
			node_near = node_far = NULL;
			distance2 = distance2_at_is_point_inrange = 0;
			for (int i = 0; i < N; ++i){
				this->pos[i] = 0;
			}
		}
		inline algo_eukleide_range(const P pos[N], double range){
			dx2 = 0;
			node_near = node_far = NULL;
			distance2 = range * range;
			for (int i = 0; i < N; ++i){
				this->pos[i] = pos[i];
			}
		}
		inline void prepare(KDNode *node, int dir){
			double dx = pos[dir] - node->pos_value.first[dir];
			dx2 = dx * dx;

			if (dx <= 0.0){
				node_near = node->left;
				node_far = node->right;
			}else{
				node_near = node->right;
				node_far = node->left;
			}
		}

		inline bool is_nearleaf_inrange() const {
			if (!node_near) return false;
			return true;
		}
		inline bool is_farleaf_inrange() const {
			if (!node_far) return false;
			return dx2 <= distance2;
		}

		inline bool is_point_inrange(KDNode *node) {
			double distance2_ = 0;
			for (int i = 0; i < N; ++i){
				P d = node->pos_value.first[i] - pos[i];
				distance2_ += d * d;
			}
			distance2_at_is_point_inrange = distance2_;
			return (distance2_ <= distance2);
		}
		inline bool operator == (const algo_eukleide_range &instance) const {
			if (distance2 != instance.distance2) return false;
			for (int i = 0; i < N; ++i){
				if (pos[i] != instance.pos[i]) return false;
			}
			return true;
		}
	};

	class iterator : public iterator_abstract<algo_all>{
	public:
		iterator() : iterator_abstract<algo_all>(algo_all()){};
		iterator(KDNode *root_) : iterator_abstract<algo_all>(root_, algo_all()){
		}
	};

	class iterator_eukleide_range : public iterator_abstract<algo_eukleide_range>{
		const double &distance2_at_is_point_inrange(){
			return this->algo.distance2_at_is_point_inrange;
		}
		double &distance2(){
			return this->algo.distance2;
		}
	public:
		double get_distance(){
			return std::sqrt(this->algo.distance2_at_is_point_inrange);
		}
		iterator_eukleide_range() : iterator_abstract<algo_eukleide_range>(algo_eukleide_range()){};
		iterator_eukleide_range(KDNode *root_, const P pos[N], double range) : iterator_abstract<algo_eukleide_range>(root_, algo_eukleide_range(pos, range)){
		}
		friend class kdtree;
	};
	kdtree(){
		root = NULL;
		count_nodes = 0;
	}
private:
	void copy_from(const kdtree<T,N,P> &kdt){
		if (!kdt.root){
			root = 0;
			count_nodes = 0;
			return;
		}

		//go[XđScopy
		root = new KDNode(kdt.root->pos_value);
		stack_t buffer_from, buffer_to;
		buffer_from.push(kdt.root);
		buffer_to.push(root);
		do{ // [DT - s
			KDNode *copy_from = buffer_from.top();
			KDNode *copy_to = buffer_to.top();
			buffer_from.pop();
			buffer_to.pop();

			if (copy_from->right){
				copy_to->right = new KDNode(copy_from->right->pos_value);
				buffer_from.push(copy_from->right);
				buffer_to.push(copy_to->right);
			}
			if (copy_from->left){
				copy_to->left = new KDNode(copy_from->left->pos_value);
				buffer_from.push(copy_from->left);
				buffer_to.push(copy_to->left);
			}
		}while(buffer_from.size());
	}
public:
	kdtree(const kdtree<T,N,P> &kdt){
		copy_from(kdt);
	}
	kdtree<T,N,P> &operator = (const kdtree<T,N,P> &kdt){
		copy_from(kdt);
		return *this;
	}
	void clear(){
		if (root){
			//go[XđSdelete
			stack_t buffer;
			buffer.push(root);
			do{ // [DT - s
				KDNode *to_delete = buffer.top();
				buffer.pop();
				if (to_delete->right) buffer.push(to_delete->right);
				if (to_delete->left) buffer.push(to_delete->left);
				delete to_delete;
			}while(buffer.size());
		}
		root = NULL;
		count_nodes = 0;
	}
	void swap(kdtree<T,N,P> &kdt){
		std::swap(root, kdt.root);
		std::swap(count_nodes, kdt.count_nodes);
	}
	~kdtree(){
		clear();
	}
	size_t size(){
		return count_nodes;
	}
private:
	std::pair<iterator_io, bool> insert(KDNode *node){
		if (!root){
			root = node;
			++count_nodes;
			return std::make_pair(iterator_io(root),true);
		}
		int dir = DIRSTART;
		KDNode *cur = root;
		for(;;){
			if (cur->pos_value.first[dir] > node->pos_value.first[dir]){
				if (!cur->left){
					cur->left = node;
					++count_nodes;
					break;
				}else cur = cur->left;
			}else{
				if (same_pos(cur->pos_value.first, node->pos_value.first)) return std::make_pair(iterator_io(cur), false);
				if (!cur->right){
					cur->right = node;
					++count_nodes;
					break;
				}else cur = cur->right;
			}
			dir = rotate(dir);
		}
		return std::make_pair(iterator_io(node), true);
	}

public:
	std::pair<iterator_io, bool> insert(const value_type &value){
		return insert(new KDNode(value));
	}
	std::pair<iterator_io, bool> insert(const P pos[N], const T &value){
		return insert(new KDNode(pos, value));
	}

	iterator_io find(const key_type &k) const{
		return find((const double *)k);
	}

	iterator_io find(const P pos[N]) const{
		iterator_eukleide_range iter = begin_range(pos, 0);
		return iterator_io(iter.current);
	}

	bool empty() const{
		return (count_nodes == 0);
	}

	size_type count(const key_type &k) const{
		return find(k).current ? 1 : 0;
	}

	size_type count(const P pos[N]) const{
		return find(pos).current ? 1 : 0;
	}

	iterator begin() const{
		return iterator(root);
	}
	iterator end() const{
		iterator iter;
		return iter;
	}

	iterator_io end_io() const{
		return iterator_io();
	}

	iterator_eukleide_range begin_range(const P pos[N], double range) const{
		return iterator_eukleide_range(root, pos, range);
	}

	iterator_eukleide_range end_range() const{
		iterator_eukleide_range iter;
		return iter;
	}

	int query_eukleide_knn(const P pos[N], int k, iterator_io *values, double *distances2 = 0){
		if (!root) return 0;
		double *dist2 = distances2 ? distances2 : new double[k];
		iterator_io *node = values ? values : new iterator_io[k];
		int csize = 0, backsize = 0;
		for (iterator_eukleide_range iter = begin_range(pos, std::numeric_limits<double>::max()); !iter.is_end(); iter.next()){
			double distance2 = iter.distance2_at_is_point_inrange();
			int i;
			for (i = 0; i < k;++i){
				if (i == csize || dist2[i] >= distance2){
					for (int h = backsize; h > i; --h){
						dist2[h] = dist2[h-1];
						node[h] = node[h-1];
					}
					dist2[i] = distance2;
					node[i].current = iter.get_current_node();
					if (csize < k){
						++csize;
						if (backsize < k - 1){
							++backsize;
						}
					}else{
						iter.distance2() = dist2[k-1];
					}
					break;
				}
			}
		}
		if (dist2 != distances2) delete[] dist2;
		if (node != values) delete[] node;
		return csize;
	}

private:
	class inner {
		const int dir;
	public:
		inner(const int dir_) : dir(dir_){}
		bool operator ()(const KDNode * left, const KDNode * right) const{
			return left->pos_value.first[dir] < right->pos_value.first[dir];
		}
	};

public:
	void balance(){
		using namespace std;
		size_t count_temp = size();

		KDNode **nodes = new KDNode *[count_temp];
		size_t i = 0;
		for (iterator iter = begin(); iter != end(); ++iter, ++i){
			nodes[i] = iter.get_current_node();
		}

		count_nodes = 0;
		for (i = 0; i < count_temp; ++i){
			nodes[i]->left = nodes[i]->right = NULL;
		}
		root = NULL;

		deque<pair<int, pair<KDNode **, KDNode **> > > q;
		q.push_front(make_pair(DIRSTART, make_pair(nodes, nodes+count_temp)));
		do{
			int dir = q.front().first;
			const inner cmp(dir);
			KDNode **first = q.front().second.first, **second = q.front().second.second;
			sort(first, second, cmp);
			q.pop_front();
			KDNode **mid = first + ((second - first) / 2);
			insert(*mid);
			*mid = NULL;
			dir = rotate(dir);
			if (first < mid) q.push_front(make_pair(dir,make_pair(first, mid)));
			if (mid+1 < second) q.push_front(make_pair(dir,make_pair(mid+1,second)));
		}while(q.size());


		delete[] nodes;
	}
};
}
