/*
 * cppds
 * hash_table.h
 * KDTree class
 * Copylight (C) 2009 mocchi
 * mocchi_2003@yahoo.co.jp
 * License: Boost ver.1
 */

// hash table container Template Library
#ifndef HASH_TABLE_CONTAINER_H_
#define HASH_TABLE_CONTAINER_H_

#include <vector>
#include <list>
#include <cstdlib>

namespace cppds{
template <typename Key, typename Value> struct hash_table{
	typedef typename Key key_type;
	typedef typename Value mapped_type;
	typedef size_t size_type;
private:
	static Key key_zero;
	static Value value_zero;
	struct HashElem{
		const Key first;
		Value second;
		HashElem(const key_type &key) : first(key), second(Value()), next(0){
		}
		HashElem(const key_type &key, mapped_type &v) : first(key), second(v), next(0){
		}
	private:
		HashElem *next;
		friend struct hash_table;
		friend class hash_table::iterator;
	};
	struct TopElem{
		HashElem *top;
		HashElem *last;
		TopElem() : top(0), last(0){
		};
	};
	size_t count;
public:
	typedef typename HashElem value_type;
	class iterator{
		TopElem *top_next, *top_end;
		HashElem *elem;
		friend struct hash_table;
		iterator(TopElem *tn, TopElem *te, HashElem *e) : top_next(tn), top_end(te), elem(e){
			if (!e) Next();
		}
		iterator();
		bool Next(){
			if (top_end){
				if (elem && elem->next){
					elem = elem->next;
					return true;
				}
				if (!top_next || top_next == top_end) goto Nothing;
				do{
					++top_next;
					if (top_next == top_end) goto Nothing;
				}while(!top_next->top);
				elem = top_next->top;
				return true;
			}
		Nothing:
			top_end = 0;
			return false;
		}
	public:
		typename const key_type &key() const{
			if (!top_end) return key_zero;
			return elem->first;
		}
		typename mapped_type &value(){
			if (!top_end){
				value_zero = mapped_type();
				return value_zero;
			}
			return elem->second;
		}
		// Ou
		iterator & operator ++() {
			if (Next()) return *this;
			return iterator_end;
		}
		// u
		iterator operator ++(int) {
			iterator itr = *this;
			++itr;
			return itr;
		}
		bool operator == (const iterator &itr) const{
			return (top_end == itr.top_end);
		}
		bool operator != (const iterator &itr) const{
			return ! operator ==(itr);
		}
	};
	typedef const iterator const_iterator;
	static iterator iterator_end;
	std::vector<TopElem> bucket;
	float max_load_factor_;
	hash_table() : count(0), bucket(16), max_load_factor_(1.5f){
	}
	~hash_table(){
		clear();
	}
	size_type size(){
		return count;
	}
	void clear(){
		iterator itr = begin();
		while(itr != end()){
			HashElem *elem = itr.elem;
			++itr;
			delete elem;
		}
		count = 0;
	}
	iterator erase(const key_type &key){
		TopElem &list = bucket[hash_value(key) % bucket.size()];
		HashElem *prev = 0;
		TopElem *te_end = &bucket[0] + bucket.size();
		for (HashElem *iter = list.top; iter; prev = iter, iter = iter->next){
			HashElem &elem = *iter;
			TopElem *topelem_new = &list;
			if (key == elem.first){
				HashElem *newelem;
				if (prev) prev->next = elem.next;
				else list.top = elem.next;
				if (elem.next) newelem = elem.next;
				else{
					list.last = prev;
					while(topelem_new < te_end && !topelem_new->top) ++topelem_new;
					if (topelem_new < te_end) newelem = topelem_new->top;
				}
				delete iter;
				--count;
				return topelem_new < te_end ? iterator(topelem_new, te_end, newelem) : iterator_end;
			}
		}
		return iterator_end;
	}
	iterator erase(const iterator &itr_toerase){
		if (!itr_toerase.top_next) return iterator_end;
		TopElem &list = bucket[hash_value(itr_toerase.key()) % bucket.size()];
		HashElem *prev = 0;
		TopElem *te_end = &bucket[0] + bucket.size();
		for (HashElem *iter = list.top; iter; prev = iter, iter = iter->next){
			HashElem &elem = *iter;
			TopElem *topelem_new = &list;
			if (iter == itr_toerase.elem){
				HashElem *newelem;
				if (prev) prev->next = elem.next;
				else list.top = elem.next;
				if (elem.next) newelem = elem.next;
				else{
					list.last = prev;
					while(topelem_new < te_end && !topelem_new->top) ++topelem_new;
					if (topelem_new < te_end) newelem = topelem_new->top;
				}
				delete iter;
				--count;
				return topelem_new < te_end ? iterator(topelem_new, te_end, newelem) : iterator_end;
			}
		}
		return iterator_end;
	}
	mapped_type & operator[] (const key_type &key) {
		TopElem &list = bucket[hash_value(key) % bucket.size()];
		for (HashElem *iter = list.top; iter; iter = iter->next){
			HashElem &elem = *iter;
			if (key == elem.first){
				return elem.second;
			}
		}
		HashElem *he = new HashElem(key);
		he->next = 0;
		if (list.last) list.last->next = he;
		list.last = he;
		if (!list.top) list.top = he;
		++count;
		if ((int)((float)bucket.size() * max_load_factor_) == count) rehash((int)(bucket.size() * 4));
		return he->second;
	}
	iterator find(const key_type &key){
		TopElem &list = bucket[hash_value(key) % bucket.size()];
		for (HashElem *iter = list.top; iter; iter = iter->next){
			HashElem &elem = *iter;
			if (key == elem.first){
				return iterator (&list, &bucket[0] + bucket.size(), iter);
			}
		}
		return iterator_end;
	}
	bool empty(const key_type &key) const{
		return find(key) == iterator_end;
	}
	void rehash(int newsize_want){
		std::vector<TopElem> newbucket(newsize_want);
		for (size_t j = 0; j < bucket.size(); ++j){
			TopElem &list = bucket[j];
			for (HashElem *iter = list.top; iter;){
				HashElem *tomove = iter;
				TopElem &newlist = newbucket[hash_value(tomove->first) % newbucket.size()];
				iter = iter->next;
				tomove->next = 0;
				if (newlist.last) newlist.last->next = tomove;
				newlist.last = tomove;
				if (!newlist.top) newlist.top = tomove;
			}
		}
		bucket.swap(newbucket);
	}
	float load_factor() const{
		return (float)size() / (float)bucket_count();
	}
	float max_load_factor() const{
		return max_load_factor_;
	}
	void max_load_factor(float new_max_load_factor){
		max_load_factor_ = new_max_load_factor;
	}
	size_type bucket_count() const{
		return bucket.size();
	}
#if 0
	std::pair<iterator, bool> insert(value_type &v){
		const Key &key, const mapped_type &value){
		TopElem &list = bucket[hash_value(key) % bucket.size()];
		for (HashElem *iter = list.top; iter; iter = iter->next){
			HashElem &elem = *iter;
			if (key == elem.first){
				elem.value = value;
				return;
			}
		}
		HashElem *he = new HashElem(key);
		he->value = value;
		he->next = 0;
		if (list.last) list.last->next = he;
		list.last = he;
		if (!list.top) list.top = he;
		++count;
		if ((int)((double)bucket.size() * 1.5) == count) rehash((int)(bucket.size() * 4));
	}
#endif
	iterator begin(){
		iterator itr(&bucket[0], &bucket[0] + bucket.size(), bucket[0].top);
		return itr;
	}
	const iterator &end() const{
		return iterator_end;
	}
};
template <typename Key, typename Value> Key hash_table<Key,Value>::key_zero = Key();
template <typename Key, typename Value> Value hash_table<Key,Value>::value_zero = Value();
template <typename Key, typename Value> typename hash_table<Key,Value>::iterator hash_table<Key,Value>::iterator_end = iterator(0,0,0);
}
#endif // HASH_TABLE_CONTAINER_H_
