/* File: memory.c */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include <malloc.h>
#include <sys/mman.h>

#include "common.h"
#include "memory.h"
#include "draw.h"

#define DMA_REUSE_ARRAY		256

long gri_used_memory = 0;
long gri_used_dmemory = 0;
static unsigned long __grp_dmemory_used = 0;
static int grp_already_memory_init = FALSE;
static struct V_dma_array {
	int size;
	void *pointer;
} dma_array[DMA_REUSE_ARRAY];

void __grp_memory_init( void )
{
	if (grp_already_memory_init)
		return;
	gri_used_memory	   = 0;
	gri_used_dmemory   = 0;
	__grp_dmemory_used = 0;
	grp_already_memory_init = TRUE;
	memset( dma_array, 0, sizeof(dma_array[DMA_REUSE_ARRAY]) );
	return;
}

void *grp_memalign( int align, int n )	/* force set align 16 */
{
	unsigned char *p;

	if (!grp_already_memory_init)
		__grp_memory_init();

	p = memalign( /* align */ 16, n + 16 );
	if (!p) {
		TRACE("grp_malloc() memory allocation error. (%d/%d)\n", n, gri_used_memory);
		return NULL;
	}
	*((int *)p) = n;
	gri_used_memory += n;
	return p + 16;
}

void *grp_malloc( int n )
{
	return grp_memalign( 16, n );
}

void grp_free( void *p )
{
	void *np;

	if (!grp_already_memory_init)
		__grp_memory_init();

	np = (void *)(((unsigned char *)p) - 16);
	gri_used_memory -= *((int *)np);
	free(np);
	return;
}

void *grp_dmemalign( int align, int n )	/* force set align 16 */
{
	int i, m;
	unsigned long t;

	if (!grp_already_memory_init)
		__grp_memory_init();

	for (i = 0; i < DMA_REUSE_ARRAY; i++) {
		if (dma_array[i].size == n) {
			dma_array[i].size = 0;
			return dma_array[i].pointer;
		}
	}
	t = __grp_dmemory_used;
	if (__grl_ps2_dmamem_p == MAP_FAILED) {
		TRACE("grp_dmalloc() map failed error\n");
		return NULL;
	}
	m = n;
	n = (n + 16 + 15) & ~15; /* align */
#ifdef USE_PHSYADR_DMA
	if (__grp_dmemory_used + n >= PS2_RESERVE_MEMORY_SIZE) {
#else
	if (__grp_dmemory_used + n >= gri_dmalloc_mmapsize) {
#endif
		TRACE("grp_dmalloc() out of memory\n");
		return NULL;
	}
	gri_used_dmemory   += n;
	__grp_dmemory_used += n;
	*(int *)(__grl_ps2_dmamem_p + t) = m;
	return __grl_ps2_dmamem_p + t + 16;
}

void *grp_dmalloc( int n )
{
	return grp_dmemalign( 16, n );
}

void grp_dfree( void *p )
{
	int i;

	if (!grp_already_memory_init)
		__grp_memory_init();

	for (i = 0; i < DMA_REUSE_ARRAY; i++) {
		if (!dma_array[i].size) {
			dma_array[i].size    = *(int *)(p-16);
			dma_array[i].pointer = p;
			break;
		}
	}
	if (i >= DMA_REUSE_ARRAY)
		TRACE("grp_dfree() ignored\n");
	if (__grl_ps2_dmamem_p == MAP_FAILED) {
		TRACE("grp_dfree() map failed error\n");
		return;
	}
	return;
}
