/* File: tex.c */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "common.h"
#include "draw.h"
#include "tex.h"
#include "fileio.h"
#include "memory.h"
#include "bmp.h"

static int __grl_ps2_vraminfo_image_max;
static int __grl_ps2_vraminfo_clut_max;
static int __grl_ps2_clutsize;
static __GRL_PS2_VRAMINFO *__grl_ps2_vraminfo_image = NULL;
static __GRL_PS2_VRAMINFO *__grl_ps2_vraminfo_clut  = NULL;
static int __grl_ps2_vramusable;
static int __grl_ps2_clutusable;
static int __grl_ps2_clutsize;

static struct bmp_conv_table_v {
	int mode;
	int mul;
	int div;
} bmp_conv_table[] = {
	{PS2_GS_PSMCT16, 2,  1},
	{PS2_GS_PSMCT24, 3,  1},
	{PS2_GS_PSMCT32, 4,  1},
	{PS2_GS_PSMT4,   1,  2},
	{PS2_GS_PSMT8,   1,  1},
};

void __grl_tex_init( void )
{
	/* allocate memory */
	__grl_ps2_vraminfo_image = NULL;
	__grl_ps2_vraminfo_clut  = NULL;
	__grl_ps2_clutsize	 = 256 * VBYTE64;
	if (!grd_ps2_tex_setvram(-1)) {
		ERRMSG("Memory allocation error !\n");
		EXIT(1);
	}
	return;
}

int grd_ps2_tex_setvram( int clutsize )
{
	/* release memory */
	if (__grl_ps2_vraminfo_image && __grl_ps2_vraminfo_clut)
		grd_tex_clear();
	free( __grl_ps2_vraminfo_image );
	__grl_ps2_vraminfo_image = NULL;
	free( __grl_ps2_vraminfo_clut  );
	__grl_ps2_vraminfo_clut  = NULL;

	/* calculation */
	/* __grl_ps2_vramusable = (((SCREEN_SIZE_X*SCREEN_SIZE_IY*4 + VBYTE2048-1) & VMASK2048) * 3) >> VBYTE64SFT; */ /* Z=32bit only */
	__grl_ps2_vramusable = __grl_ps2_setdbuff_val * (VWORD2048 / VWORD64);
	if (clutsize >= 0)
		__grl_ps2_clutsize = clutsize;
	__grl_ps2_clutusable = GRL_PS2_VRAM_MAX -
		(((__grl_ps2_clutsize + VBYTE2048-1) & ~VBYTE2048) >> VBYTE64SFT);
	__grl_ps2_vraminfo_image_max = (__grl_ps2_clutusable - __grl_ps2_vramusable + (VWORD2048/VWORD64-1)) >> (VWORD2048SFT - VWORD64SFT);
	__grl_ps2_vraminfo_clut_max  = GRL_PS2_VRAM_MAX - __grl_ps2_clutusable;

	/* allocate V-RAM control info. */
	__grl_ps2_vraminfo_image = MALLOC( sizeof(__GRL_PS2_VRAMINFO[__grl_ps2_vraminfo_image_max]) );
	__grl_ps2_vraminfo_clut  = MALLOC( sizeof(__GRL_PS2_VRAMINFO[__grl_ps2_vraminfo_clut_max]) );
	if (!__grl_ps2_vraminfo_image || !__grl_ps2_vraminfo_clut) {
		free( __grl_ps2_vraminfo_image );
		__grl_ps2_vraminfo_image = NULL;
		free( __grl_ps2_vraminfo_clut  );
		__grl_ps2_vraminfo_clut  = NULL;
		return FALSE;
	}
	grd_tex_clear();

	return TRUE;
}

int grd_tex_clear( void )
{
	int i;

	/* texture information */
	/* need clear ! */

	/* clear V-RAM information */
	for (i = 0; i < __grl_ps2_vraminfo_image_max; i++) {
		if (__grl_ps2_vraminfo_image[i].attr > 0)
			__grl_ps2_vraminfo_image[i].p -> image_transferred[__grl_ps2_vraminfo_image[i].n] = FALSE;
		__grl_ps2_vraminfo_image[i].attr = 0;
		__grl_ps2_vraminfo_image[i].len  = 0;
		__grl_ps2_vraminfo_image[i].n    = 0;
		__grl_ps2_vraminfo_image[i].p    = NULL;
	}
	for (i = 0; i < __grl_ps2_vraminfo_clut_max; i++) {
		if (__grl_ps2_vraminfo_clut[i].attr > 0)
			__grl_ps2_vraminfo_clut[i].p -> clut_transferred[__grl_ps2_vraminfo_clut[i].n] = FALSE;
		__grl_ps2_vraminfo_clut[i].attr = 0;
		__grl_ps2_vraminfo_clut[i].len  = 0;
		__grl_ps2_vraminfo_clut[i].n    = 0;
		__grl_ps2_vraminfo_clut[i].p    = NULL;
	}
	return TRUE;
}

static int __grd_tex_image_allocnewspace( GRD_TEX_INFO *tx_p, int image_select, GRD_TEX_INFO *using_tx_p )
{
	int i, j;
	int len = ((tx_p -> image_length[image_select] + (VBYTE2048-1)) / VBYTE2048);

	/* search new image space */
	for (i = 0; i < __grl_ps2_vraminfo_image_max; i++) {
		for (j = 0; (j < len) && (i+j < __grl_ps2_vraminfo_image_max); j++) {
			if (!__grl_ps2_vraminfo_image[i+j].attr) {
				if (j == len - 1) {
					__grl_ps2_vraminfo_image[i].attr = 1;
					__grl_ps2_vraminfo_image[i].len  = len;
					__grl_ps2_vraminfo_image[i].n    = image_select;
					__grl_ps2_vraminfo_image[i].p    = NULL;
					for (j = 1; j < len; j++) {
						__grl_ps2_vraminfo_image[i+j].attr = -1;
						__grl_ps2_vraminfo_image[i+j].len  = 0;
						__grl_ps2_vraminfo_image[i+j].n    = 0;
						__grl_ps2_vraminfo_image[i+j].p    = NULL;
					}
					return i;
				}
			} else {
				i += j;
				break;
			}
		}
	}
	grd_tex_clear();
	return __grd_tex_image_allocnewspace( tx_p, image_select, using_tx_p );
}

static int __grd_tex_clut_allocnewspace( GRD_TEX_INFO *tx_p, int clut_select, GRD_TEX_INFO *using_tx_p )
{
	int i, j;
	int len = (tx_p -> clut_length[clut_select] + VBYTE64 - 1) / VBYTE64;

	/* search new clut space */
	for (i = 0; i < __grl_ps2_vraminfo_clut_max; i++) {
		for (j = 0; (j < len) && (i+j < __grl_ps2_vraminfo_clut_max); j++) {
			if (!__grl_ps2_vraminfo_clut[i+j].attr) {
				if (j == len - 1) {
					__grl_ps2_vraminfo_clut[i].attr = 1;
					__grl_ps2_vraminfo_clut[i].len  = len;
					__grl_ps2_vraminfo_clut[i].n    = clut_select;
					__grl_ps2_vraminfo_clut[i].p    = NULL;
					for (j = 1; j < len; j++) {
						__grl_ps2_vraminfo_clut[i+j].attr = -1;
						__grl_ps2_vraminfo_clut[i+j].len  = 0;
						__grl_ps2_vraminfo_clut[i+j].n    = 0;
						__grl_ps2_vraminfo_clut[i+j].p    = NULL;
					}
					return i;
				}
			} else {
				i += j;
				break;
			}
		}
	}
	grd_tex_clear();
	return __grd_tex_clut_allocnewspace( tx_p, clut_select, using_tx_p );
}

int grd_tex_readbmp( GRD_TEX_INFO *tx_p, unsigned char *buf )
{
	int i, j, k, t, flag;
	int gmode_c, gmode_i;
	int col_bit, col_size = 0;
	int col_size_max_x = 0, col_size_max_y = 0;
	unsigned int def_alpha = 128;
	unsigned char *im_p = NULL, *clut_p = NULL;
	unsigned char *bitdata_p, *p;
	BITMAPFILEHEADER *filehead_p;
	BITMAPINFOHEADER *fileinfo_p;
	RGBQUAD *rgbcol_p;

	filehead_p = (BITMAPFILEHEADER *)buf;
	fileinfo_p = (BITMAPINFOHEADER *)(buf + sizeof(BITMAPFILEHEADER));

	/* check bitmap type */
	if (strncmp(filehead_p -> bfType, "BM", 2)) {
		ERRMSG("not supportted bitmap type (eyecatch != BM)\n");
		return FALSE;
	}
	if (fileinfo_p -> biPlanes != 1) {
		ERRMSG("not supportted bitmap type (biPlanes != 1)\n");
		return FALSE;
	}
	if (fileinfo_p -> biCompression != BI_RGB) {
		ERRMSG("not supportted bitmap type (biCompression != BI_RGB)\n");
		return FALSE;
	}
	if (fileinfo_p -> biCompression == BI_BITFIELDS && fileinfo_p -> biBitCount != 16) {
		ERRMSG("not supportted bitmap type (biCompression != BI_RGB && BitCount != 16)\n");
		return FALSE;
	}
	col_bit = fileinfo_p -> biBitCount;
	switch (col_bit) {
		case 4  :
			if (fileinfo_p -> biClrUsed)
				col_size = fileinfo_p -> biClrUsed;
			else
				col_size = 16;
			col_size_max_x = 16;
			col_size_max_y = 1;
			break;
		case 8  :
			if (fileinfo_p -> biClrUsed)
				col_size = fileinfo_p -> biClrUsed;
			else
				col_size = 256;
			col_size_max_x = 16;
			col_size_max_y = 16;
			break;
		case 24 :
			break;
		default :
			ERRMSG("not supportted bitmap type (biBitcount != 4/8/24)\n");
			return FALSE;
	}

	/* RGB info (clut) */
	rgbcol_p = (RGBQUAD *)(((unsigned char *)fileinfo_p) + sizeof(BITMAPINFOHEADER));
	if (col_bit == 4) {
		if (!(clut_p = DMALLOC( col_size_max_x*col_size_max_y*4 ))) {
			ERRMSG("can't allocate memory for colormap.\n");
			return FALSE;
		}
		memset( clut_p, 0, col_size_max_x*col_size_max_y*4 );
		for (j = 0; j < col_size; j++) {
			clut_p[j << 2      ] = rgbcol_p[j].rgbRed;
			clut_p[(j << 2) + 1] = rgbcol_p[j].rgbGreen;
			clut_p[(j << 2) + 2] = rgbcol_p[j].rgbBlue;
			if (!rgbcol_p[j].rgbRed && !rgbcol_p[j].rgbGreen && !rgbcol_p[j].rgbBlue)
				clut_p[(j << 2) + 3] = 0;
			else
				clut_p[(j << 2) + 3] = def_alpha;
		}
		gmode_c = 3; /* RGB32 */
	} else if (col_bit == 8) {
		if (!(clut_p = DMALLOC( col_size_max_x*col_size_max_y*4 ))) {
			ERRMSG("can't allocate memory for colormap.\n");
			return FALSE;
		}
		memset( clut_p, 0, col_size_max_x*col_size_max_y*4 );
		for (j = 0; j < col_size; j++) {
			int jj;

			jj = j ^ (0x18 * (((j >> 3) ^ (j >> 4)) & 1));
			clut_p[(jj << 2)  ] = rgbcol_p[j].rgbRed;
			clut_p[(jj << 2)+1] = rgbcol_p[j].rgbGreen;
			clut_p[(jj << 2)+2] = rgbcol_p[j].rgbBlue;
			if (!rgbcol_p[j].rgbRed && !rgbcol_p[j].rgbGreen && !rgbcol_p[j].rgbBlue)
				clut_p[(jj << 2)+3] = 0;
			else
				clut_p[(jj << 2)+3] = def_alpha;
		}
		gmode_c = 3; /* RGB32 */
	} else {
		gmode_c = 0;
		clut_p = NULL;
	}

	/* image */
	bitdata_p = ((unsigned char *)rgbcol_p) + sizeof(RGBQUAD)*col_size;
	p = bitdata_p;
	switch (col_bit) {
		case 4 :
			im_p = DMALLOC( ((fileinfo_p -> biWidth)*(fileinfo_p -> biHeight)) >> 1 );
			if (!im_p) {
				ERRMSG("can't allocate memory. (image)\n");
				if (clut_p)
					DFREE(clut_p);
				return FALSE;
			}
			for (j = (fileinfo_p -> biHeight) - 1; j >= 0 ; --j) {
				int tv;

				for (k = 0; k < ((fileinfo_p -> biWidth+1) >> 1); k++) {
					tv = ((*p >> 4) & 0x0f) | ((*p << 4) & 0xf0);
					im_p[j*((fileinfo_p -> biWidth+1) >> 1) + k] = tv;
					p++;
				}
				p = p + (~((unsigned int)p - (unsigned int)bitdata_p + 3) & 3);
			}
			gmode_i = 4; /* TEX4 */
			break;
		case 8 :
			im_p = DMALLOC( (fileinfo_p -> biWidth)*(fileinfo_p -> biHeight) );
			if (!im_p) {
				ERRMSG("can't allocate memory. (image)\n");
				if (clut_p)
					DFREE(clut_p);
				return FALSE;
			}
			for (j = (fileinfo_p -> biHeight) - 1; j >= 0 ; --j) {
				for (k = 0; k < fileinfo_p -> biWidth; k++)
					im_p[j*(fileinfo_p -> biWidth)+k] = *p++;
				p = p + (~((unsigned int)p - (unsigned int)bitdata_p + 3) & 3);
			}
			gmode_i = 5; /* TEX8 */
			break;
		case 24 :
			im_p = DMALLOC( 4*(fileinfo_p -> biWidth)*(fileinfo_p -> biHeight) );
			if (!im_p) {
				ERRMSG("can't allocate memory. (image)\n");
				if (clut_p)
					DFREE(clut_p);
				return FALSE;
			}
			for (j = fileinfo_p -> biHeight - 1; j >= 0 ; --j) {
				for (k = 0; k < fileinfo_p -> biWidth; k++) {
					im_p[(j*(fileinfo_p -> biWidth)+k)*4  ] = *(p+2);
					im_p[(j*(fileinfo_p -> biWidth)+k)*4+1] = *(p+1);
					im_p[(j*(fileinfo_p -> biWidth)+k)*4+2] = *p;
					if (!*p && !*(p+1) && !*(p+2))
						im_p[(j*(fileinfo_p -> biWidth)+k)*4+3] = 0;
					else
						im_p[(j*(fileinfo_p -> biWidth)+k)*4+3] = def_alpha;
					p += 3;
				}
				p = p + (~((unsigned int)p - (unsigned int)bitdata_p + 3) & 3);
			}
			gmode_i = 3; /* RGB32 */
			break;
		default :
			ERRMSG("not supported BMP bit number.\n");
			gmode_i = 0;
			if (clut_p)
				DFREE(clut_p);
			return FALSE;
	}

	/* set data */
	tx_p -> free_p    = buf;
	tx_p -> freei_p   = im_p;
	tx_p -> freec_p   = clut_p;
	tx_p -> mipmap_max   = 1;
	tx_p -> image_format = bmp_conv_table[gmode_i-1].mode;
	tx_p -> clut_format = bmp_conv_table[gmode_c-1].mode;
	tx_p -> image_size_w = fileinfo_p -> biWidth;
	tx_p -> image_size_h = fileinfo_p -> biHeight;

	/* image */
	for (i = 0; i < GRD_MAX_MIPMAP; i++)
		tx_p -> image_p[i] = NULL;
	tx_p -> image_length[0] = (tx_p -> image_size_w)*(tx_p -> image_size_h)*bmp_conv_table[gmode_i-1].mul/bmp_conv_table[gmode_i-1].div;
	tx_p -> image_vram_adr[0] = 0;
	tx_p -> image_p[0] = (void *)im_p;
	tx_p -> image_transferred[0] = FALSE;

	/* clut */
	if ((tx_p -> image_format == PS2_GS_PSMT4) || (tx_p -> image_format == PS2_GS_PSMT8)) {
		tx_p -> clut_p = clut_p;
		tx_p -> clut_size_w = 16;
		tx_p -> clut_size_h = (col_size_max_x *  col_size_max_y + 15) >> 4;
		if (tx_p -> image_format == PS2_GS_PSMT8)
			tx_p -> clut_size_h = 16;
		else
			tx_p -> clut_size_h = 1;
		tx_p -> clut_max = 1;
		tx_p -> clut_length[0] = (tx_p -> image_format == PS2_GS_PSMT8 ? 256 : 16) * bmp_conv_table[gmode_c-1].mul;
		tx_p -> clut_vram_adr[0] = 0;
		tx_p -> clut_transferred[0] = FALSE;
	} else {
		tx_p -> clut_size_w = -1;
		tx_p -> clut_size_h = -1,
		tx_p -> clut_format = -1;
		tx_p -> clut_p = NULL;
		tx_p -> clut_max = -1;
		tx_p -> clut_length[0] = -1;
		tx_p -> clut_vram_adr[0] = -1;
		tx_p -> clut_transferred[0] = FALSE;
	}

	/* set texture bit */
	flag = FALSE;
	tx_p -> image_bit_w = 0;
	t = tx_p -> image_size_w;
	while (t > 1) {
		if (t & 1)
			flag = TRUE;
		t = t >> 1;
		tx_p -> image_bit_w++;
	}
	if (flag)
		tx_p -> image_bit_w++;
	flag = FALSE;
	tx_p -> image_bit_h = 0;
	t = tx_p -> image_size_h;
	while (t > 1) {
		if (t & 1)
			flag = TRUE;
		t = t >> 1;
		tx_p -> image_bit_h++;
	}
	if (flag)
		tx_p -> image_bit_h++;
	return TRUE;
}

int grd_tex_readtex( GRD_TEX_INFO *tx_p, void *tex_p, char *ext )
{
	if (!strcasecmp( ext, "bmp" ))
		return grd_tex_readbmp( tx_p, tex_p );
	ERRMSG("grd_tex_readtex() unknown texture file extention. (%s)\n", ext);
	return FALSE;
}

int grd_tex_read( int attr, char *buf, GRD_TEX_INFO *tx_p, char *pathname )
{
	int size;
	char fn[256], ext[256], *cp;

	cp = strrchr( pathname, '/' );
	if (!cp)
		cp = pathname;
	else
		cp++;
	strcpy( fn, cp );
	for (cp = fn; *cp != '.' && *cp != '\0'; cp++)
		;
	if (*cp != '\0') {
		*cp = '\0';
		strcpy( ext, (cp+1) );
		if (ext[0] == '\0') {
			ERRMSG("no extention texture path name. \"%s\"\n", pathname);
			if (attr & GRD_TRATTR_ERROREXIT)
				EXIT(1);
			else
				return FALSE;
		}
	}
	if (strlen(fn) >= GRD_TEX_MAX_NAMELEN) {
		TRACE("warning, longer texture name ! ignored. \"%s\"\n", fn);
		fn[GRD_TEX_MAX_NAMELEN-1] = '\0';
	}
	strcpy( tx_p->tex_name, fn );

	if (attr & GRD_TRATTR_GETSIZE) {
		if ((size = grp_fileio_getsize( pathname )) < 0) {
			ERRMSG("texture file open error. \"%s\"\n", pathname);
			if (attr & GRD_TRATTR_ERROREXIT)
				EXIT(1);
			else
				return -1;
		}
		return size;
	}
	if ((size = grp_fileio_open( pathname )) < 0) {
		ERRMSG("texture file open error. \"%s\"\n", pathname);
		if (attr & GRD_TRATTR_ERROREXIT)
			EXIT(1);
		else
			return FALSE;
	}
	if (!buf)
		buf = (char *)DMEMALIGN( 16, size );
	if (!buf) {
		ERRMSG("texture file memory allocation error.\n");
		grp_fileio_close();
		if (attr & GRD_TRATTR_ERROREXIT)
			EXIT(1);
		else
			return FALSE;
	}
	if (grp_fileio_read( buf ) < 0) {
		ERRMSG("texture_load: file read error. \"%s\"\n", pathname);
		if (attr & GRD_TRATTR_ERROREXIT)
			EXIT(1);
		else
			return FALSE;
	}
	return grd_tex_readtex( tx_p, buf, ext );
}

static void __grd_ps2_datatovram( unsigned char *p, u_long64 format, u_long64 vram_adr, u_long64 length, u_long64 size_w, u_long64 size_h, u_long64 dbw )
{
	u_long64 len, rest;

	_gps2_setdma_vif1( PS2_VIF_SET_CODE(0, 0, PS2_VIF_FLUSHA, 0) );
	_gps2_setdma_dma( 4+1, 1 );
	_gps2_setdma_gif( 4, 1, GIF_EOP | GIF_FLG_PACKED,
		GIF_REGS(GIF_REGS_A_D, 0) );
	_gps2_setdma_6464ul( PS2_GS_BITBLTBUF,
		GS_BITBLTBUF_DBP(vram_adr) | GS_BITBLTBUF_DBW(dbw) |
		GS_BITBLTBUF_DPSM(format)  | GS_BITBLTBUF_SPSM(format) );
	_gps2_setdma_6464ul( PS2_GS_TRXPOS,
		GS_TRXPOS_SSAX(0) | GS_TRXPOS_SSAY(0) |
		GS_TRXPOS_DSAX(0) | GS_TRXPOS_DSAY(0) | GS_TRXPOS_DIR(0) );
	_gps2_setdma_6464ul( PS2_GS_TRXREG, GS_TRXREG_RRW(size_w) | GS_TRXREG_RRH(size_h) );
	_gps2_setdma_6464ul( PS2_GS_TRXDIR, GS_TRXDIR_XDIR(0) );

	rest = length;
	while (rest > 0) {
		if (rest > 16*32767)
			len = 32767;
		else
			len = (rest+15) >> 4;
		if (length != rest) {
			_gps2_setdma_dma( 0, 0 );
		}
		_gps2_setdma_gif( len, 0, GIF_EOP | GIF_FLG_IMAGE, 0 );
		_gps2_setdma_dmaref_vif1direct( p, len );
		rest -= len << 4;
		p += len << 4;
	}
	grd_texture_flush();
	return;
}

static void __grd_ps2_imagetovram( GRD_TEX_INFO *tx_p )
{
	int i, index;

	for (i = 0; i < tx_p->mipmap_max; i++) {
		index = __grd_tex_image_allocnewspace( tx_p, i, NULL );
		__grl_ps2_vraminfo_image[index].p = tx_p;
		tx_p -> image_vram_adr[i] = index * (VWORD2048 / VWORD64) + __grl_ps2_vramusable;
		__grd_ps2_datatovram( tx_p->image_p[i], tx_p->image_format, tx_p->image_vram_adr[i], tx_p->image_length[i], tx_p->image_size_w/(i+1), tx_p->image_size_h/(i+1), (tx_p->image_size_w/(i+1)+63)/64 );
		tx_p -> image_transferred[i] = TRUE;
	}
	return;
}

static void __grd_ps2_cluttovram( GRD_TEX_INFO *tx_p, int clut_select )
{
	int index;

	index = __grd_tex_clut_allocnewspace( tx_p, clut_select, NULL );
	__grl_ps2_vraminfo_clut[index].p = tx_p;
	tx_p -> clut_vram_adr[clut_select] = index + __grl_ps2_clutusable;
	if (tx_p -> image_format == PS2_GS_PSMT4)
		__grd_ps2_datatovram( tx_p->clut_p+4*clut_select,  tx_p->clut_format, tx_p->clut_vram_adr[clut_select], tx_p->clut_length[clut_select], 8, 2, 8 );
	else
		__grd_ps2_datatovram( tx_p->clut_p+64*clut_select, tx_p->clut_format, tx_p->clut_vram_adr[clut_select], tx_p->clut_length[clut_select], 16, 16, 16 );
	tx_p -> clut_transferred[clut_select] = TRUE;
	return;
}

static void __grd_ps2_textovram( GRD_TEX_INFO *tx_p, int clut_select )
{
	int i;
	int modified = FALSE;
	int image_transfer = FALSE;

	/* transfer texture */
	for (i = 0; i < tx_p->mipmap_max; i++) {
		if (!(tx_p -> image_transferred[i]))
			image_transfer = TRUE;
	}
	if (image_transfer) {
		__grd_ps2_imagetovram( tx_p );
		modified = TRUE;
	}
	if ((tx_p -> clut_p) && !(tx_p -> clut_transferred[clut_select])) {
		__grd_ps2_cluttovram( tx_p, clut_select );
		modified = TRUE;
	}

	/* set tex tag */
	if (modified) {
		if (!(tx_p -> clut_p)) {
			tx_p -> tex_0 = GS_TEX0_TBP0( tx_p->image_vram_adr[0] ) |
				GS_TEX0_TBW((tx_p->image_size_w+63)/64) | GS_TEX0_PSM(tx_p->image_format) |
				GS_TEX0_TW(tx_p->image_bit_w) | GS_TEX0_TH(tx_p->image_bit_h) |
				GS_TEX0_TCC_RGBA | GS_TEX0_TFX(0);
		} else {
			tx_p -> tex_0 = GS_TEX0_TBP0( tx_p->image_vram_adr[0] ) |
				GS_TEX0_TBW((tx_p->image_size_w+63)/64) | GS_TEX0_PSM(tx_p->image_format) |
				GS_TEX0_TW(tx_p->image_bit_w) | GS_TEX0_TH(tx_p->image_bit_h) |
				GS_TEX0_TCC_RGBA | GS_TEX0_TFX(0) | GS_TEX0_CPSM(tx_p->clut_format) |
				GS_TEX0_CSM_N | GS_TEX0_CSA(0) | GS_TEX0_CLD(2);
		}
	}
	return;
}

void _gps2_tex_use_trans( GRD_TEX_INFO *tx_p, int clut_select )
{
	if (!tx_p)
		return;
	__grd_ps2_textovram( tx_p, clut_select );
	return;
}

void _gps2_tex_use_notag( GRD_TEX_INFO *tx_p, GRD_TEX_ATTR attr, int mipmap_max, int clut_select )
{
	int n;

	if (!tx_p)
		return;
	if (!tx_p -> clut_p) {
		/* No Clut */
		_gps2_setdma_6464ul( PS2_GS_TEX0_1+grd_get_currentcontext(), tx_p->tex_0 );
	} else {
		/* Clut */
		_gps2_setdma_6464ul( PS2_GS_TEX0_1+grd_get_currentcontext(), tx_p->tex_0 | GS_TEX0_CBP( tx_p->clut_vram_adr[clut_select] ) );
	}
	n = tx_p -> mipmap_max - 1;
	if ((mipmap_max >= 0) && (mipmap_max > tx_p->mipmap_max-1))
		n = mipmap_max;
	_gps2_setdma_6464ul( PS2_GS_TEX1_1+grd_get_currentcontext(), attr | GS_TEX1_MXL(n) );
	return;
}

void grd_tex_use( GRD_TEX_INFO *tx_p, GRD_TEX_ATTR attr, int mipmap_max, int clut_select )
{
	if (!tx_p)
		return;
	__grd_ps2_textovram( tx_p, clut_select );
	_gps2_setdma_dmagif( 2, 1, GIF_EOP | GIF_FLG_PACKED, GIF_REGS(GIF_REGS_A_D, 0) );
	_gps2_tex_use_notag( tx_p, attr, mipmap_max, clut_select );
	return;
}
