// Copyright (C) 2017  Mocchi (mocchi_2003@yahoo.co.jp)
// License: Boost Software License   See LICENSE.txt for the full license.

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace pdf_pp {
	abstract public class Item {
		abstract public string Serialize();
	}
	public class Dict : Item {
		public KeyValuePair<string, Item>[] dict = null;
		public Item GetValue(string name) {
			return dict.Where(kv => kv.Key == name).Select(kv=>kv.Value).FirstOrDefault();
		}
		override public string Serialize() {
			return "<<\n" + string.Join(" ", dict.Select(kv => kv.Key + " " + kv.Value.Serialize()).ToArray()) + "\n>>\n";
		}
		public void SetValue(string key, Item value){
			var new_kv = new KeyValuePair<string, Item>(key, value);
			for (int i = 0; i < dict.Length; ++i) {
				if (dict[i].Key != key) continue;
				dict[i] = new_kv;
				return;
			}
			dict = dict.Concat(new KeyValuePair<string, Item>[] { new_kv }).ToArray();
		}
		public Dict ShallowCopy() {
			Dict new_dict = new Dict();
			new_dict.dict = new KeyValuePair<string, Item>[dict.Length];
			dict.CopyTo(new_dict.dict, 0);
			return new_dict;
		}
	}
	public class Ary : Item {
		public Item[] ary = null;
		override public string Serialize() {
			return "[ " + string.Join(" ", ary.Select(a=>a.Serialize()).ToArray()) + " ]";
		}
	}
	public class Value : Item {
		public string val;
		public Value(string val) {
			this.val = val;
		}
		override public string Serialize() {
			return val;
		}
	}
	public class Ref : Item {
		public string[] reference = null;
		override public string Serialize() {
			return string.Join(" ", reference);
		}
	}
	public class Obj {
		public string[] objname;
		public Item item;
		public long streampos;
		public Obj(string[] obj_tokens, long streampos) {
			this.streampos = streampos;
			if (obj_tokens.Length < 2) throw new Exception("too short tokens for parsing object.");
			int idx = (obj_tokens[0] == "trailer") ? 1 : 3;
			objname = obj_tokens.Take(idx).ToArray();
			List<int> pos_bracket = new List<int>();
			List<Item> stack = new List<Item>();
			for (int j = idx; j < obj_tokens.Length; ++j) {
				string cur_tok = obj_tokens[j];
				if (cur_tok == "R") {
					if (stack.Count < 2) throw new Exception("lack of preceeding tokens of reference.");
					Ref refer = new Ref();
					int ref_start = stack.Count - 2;
					Value[] ref_itm = stack.Skip(ref_start).Take(2).Select(s => s as Value).ToArray();
					if (ref_itm[0] == null || ref_itm[1] == null) throw new Exception("reference type mismatch.");
					refer.reference = ref_itm.Select(ri=>ri.val).Concat(new []{"R"}).ToArray();
					stack.RemoveRange(ref_start, 2);
					stack.Add(refer);
				} else if (cur_tok == "<<") {
					pos_bracket.Add(stack.Count);
				} else if (cur_tok == ">>") {
					var cur_dict = new List<KeyValuePair<string, Item>>();
					int start_pos = pos_bracket.Last();
					pos_bracket.RemoveAt(pos_bracket.Count - 1);
					// start_pos  stack.Count Ƃ
					int cnt = stack.Count - start_pos;
					if ((cnt & 1) == 1) throw new Exception("the count of dictionary item is odd.");
					for (int i = start_pos; i < stack.Count; i += 2) {
						// 1߂ Value łȂƂ
						Value name = stack[i] as Value;
						if (name == null) throw new Exception("type mismatch on name cast");
						cur_dict.Add(new KeyValuePair<string, Item>(name.val, stack[i + 1]));
					}
					stack.RemoveRange(start_pos, cnt);
					var dict = new Dict();
					dict.dict = cur_dict.ToArray();
					stack.Add(dict);
				} else if (cur_tok == "[") {
					pos_bracket.Add(stack.Count);
				} else if (cur_tok == "]") {
					Ary ary = new Ary();
					int start_pos = pos_bracket.Last();
					pos_bracket.RemoveAt(pos_bracket.Count - 1);
					int cnt = stack.Count - start_pos;
					ary.ary = stack.Skip(start_pos).Take(cnt).ToArray();
					stack.RemoveRange(start_pos, cnt);
					stack.Add(ary);
				} else stack.Add(new Value(cur_tok));
				if (pos_bracket.Count == 0) {
					if (stack.Count != 1) throw new Exception(string.Format("parse failed in {0}", string.Join(" ", objname)));
					item = stack[0];
					break;
				}
			}
		}
	}
	public class Program {
		#region ȈՃeLXgo͏
		/// <summary>
		/// t@C܂łǂݍŕzɕϊĕԂB
		/// </summary>
		/// <param name="fs"></param>
		/// <returns></returns>
		static string[] GetLinesToEnd(FileStream fs, long filesize) {
			byte[] buf = new byte[filesize - fs.Position];
			fs.Read(buf, 0, buf.Length);
			return Encoding.ASCII.GetString(buf).Replace("\r\n", "\n").Replace("\r", "\n").Split('\n');
		}
		static string GetLine(FileStream fs) {
			var bytes = new List<byte>();
			for(;;){
				int b = fs.ReadByte();
				if (b == '\r'){
					int bn = fs.ReadByte();
					if (bn != '\n') fs.Seek(-1, SeekOrigin.Current);
					break;
				}else if (b == '\n' || b < 0) break;
				bytes.Add((byte)b);
			}
			return Encoding.ASCII.GetString(bytes.ToArray());
		}
		static void Write(FileStream fs, string str) {
			byte[] bary = Encoding.ASCII.GetBytes(str);
			fs.Write(bary, 0, bary.Length);
		}
		#endregion

		public static string[] GetTokens(string[] lines, ref int cur_lineidx, ref int cur_idx, int num_tokens) {
			List<string> tokens = new List<string>();
			string cur_line = lines[cur_lineidx];

			for (int i = 0; num_tokens < 0 || i < num_tokens; ++i) {
				// g[NE΂
				for (; ; ) {
					while (cur_idx < cur_line.Length && cur_line[cur_idx] == ' ') cur_idx++;
					if (cur_line.Length == cur_idx) {
						cur_lineidx++;
						cur_idx = 0;
						if (cur_lineidx >= lines.Length) return tokens.ToArray();
						cur_line = lines[cur_lineidx];
					} else break;
				}
				// g[N擾
				int token_start = cur_idx;
				char[] token_end = { '[', ']', '/' };
				while (cur_idx < cur_line.Length && cur_line[cur_idx] != ' ') {
					int cur_idx_ = cur_idx;
					if (cur_idx > token_start) {
						if (token_end.Any(t => t == cur_line[cur_idx_])) break;
						if (cur_line[cur_idx] == '<' && cur_line[cur_idx - 1] != '<') break;
						if (cur_line[cur_idx] == '>' && cur_line[cur_idx - 1] != '>') break;
					}
					cur_idx++;
					if (cur_line[cur_idx - 1] == '[') break;
				}
				tokens.Add(cur_line.Substring(token_start, cur_idx - token_start));
			}
			return tokens.ToArray();
		}
		public static string[] GetTokens(string[] lines, int num_tokens) {
			int cur_lineidx = 0, cur_idx = 0;
			return GetTokens(lines, ref cur_lineidx, ref cur_idx, num_tokens);
		}
		public static Obj GetObject(FileStream fs) {
			long stream_pos = -1;
			List<string> lines = new List<string>();
			for (; ; ) {
				string line = GetLine(fs);
				lines.Add(line);
				if (line == "stream" || line.Contains("endobj")) {
					if (line == "stream") stream_pos = fs.Position;
					string[] obj_tok = GetTokens(lines.ToArray(), -1);
					return new Obj(obj_tok, stream_pos);
				}
			}
		}
		public class Xref {
			public string[] xref_lines = null;
			public int obj_start = 0;
			public Xref(string[] tail_lines, ref int trailer_lineidx) {
				int idx = 0;
				while (idx < tail_lines.Length && tail_lines[idx] != "xref") ++idx;
				int cur_lineidx = idx + 1, cur_idx = 0;
				string[] r = GetTokens(tail_lines, ref cur_lineidx, ref cur_idx, 2);
				if (r.Length != 2) return;
				obj_start = int.Parse(r[0]);
				xref_lines = tail_lines.Skip(idx + 2).Take(int.Parse(r[1])).ToArray();
				trailer_lineidx = idx + 2 + xref_lines.Length;
			}
			public bool JumpToObject(FileStream fs, int obj_id){
				int line_idx = obj_id - obj_start;
				if (line_idx < 0 || line_idx >= xref_lines.Length) throw new Exception(string.Format("object id:{0} is out of range", obj_id));
				int idx = 0;
				string[] tokens = GetTokens(xref_lines, ref line_idx, ref idx, 3);
				if (tokens.Length != 3) return false;
				fs.Seek(long.Parse(tokens[0]), SeekOrigin.Begin);
				return true;
			}
			public bool JumpToObject(FileStream fs, string[] obj_ref){
				if (obj_ref.Length != 3 || obj_ref[2] != "R") return false;
				return JumpToObject(fs, int.Parse(obj_ref[0]));
			}
		}
		public static bool FindPDFStructure(FileStream fs, out long filesize, out long startxref, out Xref xref, out Obj trailer, out Obj root_obj) {
			filesize = 0;
			startxref = 0;
			xref = null;
			trailer = null;
			root_obj = null;

			string[] tail_lines = null;
			byte[] buf = null;

			fs.Seek(0, SeekOrigin.Begin);
			// PDF t@Cǂ`FbNB
			buf = new byte[4];
			fs.Read(buf, 0, 4);
			if (!Encoding.ASCII.GetBytes("%PDF").SequenceEqual(buf)) return false;

			// startxref TB
			fs.Seek(0, SeekOrigin.End);
			filesize = fs.Position;
			long bufsize = 6 + (int)Math.Log10(filesize) * 2 + 10;
			fs.Seek(-bufsize, SeekOrigin.End);
			tail_lines = GetLinesToEnd(fs, filesize).Where(l => !string.IsNullOrEmpty(l)).ToArray();
			if (tail_lines[tail_lines.Length - 3] != "startxref") return false;

			// xref 擾B
			startxref = long.Parse(tail_lines[tail_lines.Length - 2]);
			fs.Seek(startxref, SeekOrigin.Begin);
			tail_lines = GetLinesToEnd(fs, filesize);
			int trailer_lineidx = -1;
			xref = new Xref(tail_lines, ref trailer_lineidx);
			int trailer_idx = 0;
			trailer = new Obj(GetTokens(tail_lines, ref trailer_lineidx, ref trailer_idx, -1), -1);

			// "/Root" IuWFNg擾B
			Dict dict_trailer = trailer.item as Dict;
			if (dict_trailer == null) throw new Exception("invalid trailer.");
			Ref ref_root = dict_trailer.GetValue("/Root") as Ref;
			if (ref_root == null) throw new Exception("/Root is not found or invalid in trailer.");
			xref.JumpToObject(fs, ref_root.reference);
			root_obj = GetObject(fs);
			return true;
		}
		public static Ref[] TraversePages(FileStream fs, Ref root_pages_ref, Xref xref) {
			xref.JumpToObject(fs, root_pages_ref.reference);
			Obj root_pages = GetObject(fs);

			List<Ref> pages_refs = new List<Ref>();

			// Pagesc[TāAy[Wꗗ擾B
			var stack_pages = Enumerable.Range(0, 0).Select(s => new { obj = (Obj)null, idx = 0 }).ToList();
			stack_pages.Add(new { obj = root_pages, idx = 0 });
			while (stack_pages.Count > 0) {
				var cur_pages = stack_pages.Last();
				stack_pages.RemoveAt(stack_pages.Count - 1);
				string page_objname = string.Join(" ", cur_pages.obj.objname);
				Dict dict_pages = cur_pages.obj.item as Dict;
				if (dict_pages == null) throw new Exception(string.Format("pages \"{0}\" is not dictionary.", page_objname));
				Ary kids = dict_pages.GetValue("/Kids") as Ary;
				if (kids == null) throw new Exception(string.Format("/Kids is not found or invalid in pages {0}.", page_objname));
				for (int i = cur_pages.idx; i < kids.ary.Length; ++i) {
					Ref kids_itm = kids.ary[i] as Ref;
					if (kids_itm == null) throw new Exception(string.Format("item of /Kids is not reference in pages {0}.", page_objname));
					xref.JumpToObject(fs, kids_itm.reference);
					Obj cur = GetObject(fs);
					Dict cur_dict = cur.item as Dict;
					if (cur_dict == null) throw new Exception(
						string.Format("reference {0} of /Kids in {1} is not dict.", kids_itm.Serialize(), page_objname)
					);
					Value cur_type = cur_dict.GetValue("/Type") as Value;
					if (cur_type == null) throw new Exception(string.Format("/Type is not found or invalid in {0}", string.Join(" ", cur.objname)));
					if (cur_type.val == "/Page") {
						pages_refs.Add(kids_itm);
					} else if (cur_type.val == "/Pages") {
						stack_pages.Add(new { obj = cur_pages.obj, idx = i + 1 });
						stack_pages.Add(new { obj = cur, idx = 0 });
						break;
					} else throw new Exception(string.Format("type {0} is found in kids of pages in {1}.", cur_type.val, string.Join(" ", cur.objname)));
				}
			}
			return pages_refs.ToArray();
		}

		#region y[W
		public static string[] CreateSplittedPages(FileStream fs, Xref xref, Ref page_ref, int xdiv, int ydiv, double margin) {
			// ړĨy[W擾
			xref.JumpToObject(fs, page_ref.reference);
			Obj page = GetObject(fs);
			string pagename = string.Join(" ", page.objname);
			Dict dict_page = page.item as Dict;
			if (dict_page == null) throw new Exception(string.Format("page {0} is not dictionary", pagename));
			Ary mediabox = dict_page.GetValue("/MediaBox") as Ary;
			if (mediabox == null) throw new Exception(string.Format("/MediaBox not found in page {0}", pagename));
			List<string> objs = new List<string>();

			string[] other_boxes = new string[] { "/CropBox", "/BleedBox", "/TrimBox", "/ArtBox" };
			try {
				double[] mbox = mediabox.ary.Select(s => double.Parse((s as Value).val)).ToArray();
				double w = mbox[2] - mbox[0], h = mbox[3] - mbox[1];
				double dw = w / (double)xdiv, dh = h / (double)ydiv;
				double mw = dw * margin, mh = dh * margin;
				for (int j = ydiv - 1; j >= 0; --j) {
					double yl = (double)j * dh;
					double yh = yl + dh;
					if (j == 0) yh += mh;
					else if (j == ydiv - 1) yl -= mh;
					else {
						yl -= mh * 0.5; yh += mh * 0.5;
					}
					for (int i = 0; i < xdiv; ++i) {
						double xl = (double)i * dw;
						double xh = xl + dw;
						if (i == 0) xh += mw;
						else if (i == xdiv - 1) xl -= mw;
						else {
							xl -= mw * 0.5; xh += mw * 0.5;
						}
						Dict new_dict_page = dict_page.ShallowCopy();
						Ary new_mbox = new Ary();
						new_mbox.ary = new Item[] { new Value(xl.ToString()), new Value(yl.ToString()), new Value(xh.ToString()), new Value(yh.ToString()) };
						new_dict_page.SetValue("/MediaBox", new_mbox);
						// ̃{bNX`Ăꍇ́AVɍ쐬͈͂ŃNbvB
						foreach (var bname in other_boxes) {
							Ary obx_ary = new_dict_page.GetValue(bname) as Ary;
							if (obx_ary == null || obx_ary.ary.Length != 4) continue;
							double[] obox = obx_ary.ary.Select(s => double.Parse((s as Value).val)).ToArray();
							if (obox[0] < xl) obox[0] = xl;
							if (obox[1] < yl) obox[1] = yl;
							if (obox[2] > xh) obox[2] = xh;
							if (obox[3] > yh) obox[3] = yh;
							Ary new_obox = new Ary();
							new_obox.ary = obox.Select(ob=> (Item)new Value(ob.ToString())).ToArray();
							new_dict_page.SetValue(bname, new_obox);
						}
						objs.Add(new_dict_page.Serialize());
					}
				}
				return objs.ToArray();
			} catch {
				throw new Exception(string.Format("/MediaBox value is not numeric in page {0}", pagename));
			}
		}
		public static void CreateSplittedPDF(string cur_pdffilepath, string new_pdffilepath, int[] pages, int xdiv, int ydiv, double margin) {
			FileStream fs = null;
			try {
				File.Copy(cur_pdffilepath, new_pdffilepath, true);
				fs = new FileStream(new_pdffilepath, FileMode.Open, FileAccess.ReadWrite);

				long filesize, startxref;
				Obj trailer, root_obj;
				Xref xref;
				if (!FindPDFStructure(fs, out filesize, out startxref, out xref, out trailer, out root_obj)) {
					throw new Exception("failed to find PDF structure.");
				}

				// [gPages擾B
				Ref root_pages_ref = ((Dict)root_obj.item).GetValue("/Pages") as Ref;
				if (root_pages_ref == null) throw new Exception(string.Format("pages not found or invalid in {0}.", string.Join(" ", root_obj.objname)));

				Ref[] pages_ref = TraversePages(fs, root_pages_ref, xref);

				if (pages == null) pages = Enumerable.Range(0, pages_ref.Length).ToArray();
				string[] splitted_pages = pages.SelectMany(p => CreateSplittedPages(fs, xref, pages_ref[p], xdiv, ydiv, margin)).ToArray();
				long[] pos_splitted_pages = new long[splitted_pages.Length];

				int last_ref = xref.obj_start + xref.xref_lines.Length;

				//			fs.Seek(0, SeekOrigin.End);
				fs.Seek(startxref, SeekOrigin.Begin);

				// y[WIuWFNgt@CɒǋLB
				for (int i = 0; i < splitted_pages.Length; ++i) {
					pos_splitted_pages[i] = fs.Position;
					string obj = string.Format("{0} 0 obj\n", i + last_ref) + splitted_pages[i] + "\nendobj\n";
					Write(fs, obj);
				}

				// Pages 蒼At@CɒǋL
				long pos_pages = fs.Position;

				string new_refs = "";
				for (int i = last_ref; i < last_ref + splitted_pages.Length; ++i) {
					new_refs += string.Format("{0} 0 R ", i);
				}
				string pages_obj = string.Format("{0} 0 obj\n<<\n/Type /Pages\n/Kids [ {1}]\n/Count {2}\n>>\nendobj\n", root_pages_ref.reference[0], new_refs, splitted_pages.Length);
				Write(fs, pages_obj);

				// Xref 蒼
				long pos_xref = fs.Position;
				Write(fs, "xref\n");
				Write(fs, string.Format("{0} {1:D}\n", xref.obj_start, xref.xref_lines.Length + splitted_pages.Length));
				for (int i = 0; i < xref.xref_lines.Length; ++i) {
					if (i + xref.obj_start == int.Parse(root_pages_ref.reference[0])) {
						Write(fs, string.Format("{0:D10} 00000 n\n", pos_pages));
					} else Write(fs, xref.xref_lines[i] + "\n");
				}
				for (int i = 0; i < pos_splitted_pages.Length; ++i) {
					Write(fs, string.Format("{0:D10} 00000 n\n", pos_splitted_pages[i]));
				}

				// Trailer 蒼
				Dict new_dict_trailer = (trailer.item as Dict).ShallowCopy();
				new_dict_trailer.SetValue("/Size", new Value((xref.xref_lines.Length + splitted_pages.Length).ToString()));
				Write(fs, "trailer\n");
				Write(fs, new_dict_trailer.Serialize());

				// t@CI[
				Write(fs, string.Format("startxref\n{0}\n%%EOF\n", pos_xref));
				fs.Dispose();
			} catch (Exception e){
				if (fs != null) fs.Dispose();
				try {
					if (File.Exists(new_pdffilepath)) File.Delete(new_pdffilepath);
				} finally {
				}
				throw e;
			}
		}
		#endregion

		public static MainForm MainForm {
			get;
			set;
		}
		[STAThread]
		static void Main(string[] args) {
			MainForm = new MainForm(args.Length > 0 ? args[0] : null);
			Application.Run(MainForm);
			FileStream fs = null;
			if (args.Length < 1) return;
			try {
				// ŗ^ꂽt@CJB
				fs = new FileStream(args[0], FileMode.Open, FileAccess.Read);

				long filesize, startxref;
				Obj trailer, root_obj;
				Xref xref;
				if (!FindPDFStructure(fs, out filesize, out startxref, out xref, out trailer, out root_obj)) {
					throw new Exception("failed to find PDF structure.");
				}

				// [gPages擾B
				Ref root_pages_ref = ((Dict)root_obj.item).GetValue("/Pages") as Ref;
				if (root_pages_ref == null) throw new Exception(string.Format("pages not found or invalid in {0}.", string.Join(" ", root_obj.objname)));

				Ref[] pages_ref = TraversePages(fs, root_pages_ref, xref);
				fs.Dispose();
				fs = null;

				CreateSplittedPDF(args[0], args[0] + "_split.pdf", new int[] { 0, 1 }, 3, 3, 0.05);

				Console.WriteLine("Hello World!!");
				Console.ReadLine();
			} finally {
				if (fs != null) fs.Dispose();
			}
		}
	}
}
