#include <QHBoxLayout>
#include <QVBoxLayout>
#include <QtDebug>
#include <QLabel>
#include <QFrame>
#include <QPalette>
#include <QSlider>
#include <QTimer>
#include "player.h"

VideoFrame::VideoFrame(QWidget *parent = 0): QFrame(parent)
{
  // Set background color to black.
  QPalette palette = this->palette();
  palette.setColor(QPalette::Shadow, QColor("black"));
  setPalette(palette);
  setBackgroundRole(QPalette::Shadow);
  setAutoFillBackground(true);
}

QSize VideoFrame::sizeHint() const
{
  return QSize(320, 260); // QSize(320, 240 + menu bar)
}

IconButton::IconButton(QWidget *parent, QIcon icon, QString title = QString())
  : QToolButton(parent)
{
  setIcon(icon);
  setText(title);
  setToolTip(title);
  setFocusPolicy(Qt::NoFocus);
  setAutoRaise(true);
  setAutoFillBackground(true);
}

Player::Player(QWidget *parent = 0): QWidget(parent)
{
  // Options for VLC media player
  const char* vlc_args[] = {
    "-I", "dummy", // Don't use any interface.
    "--ignore-config", // Don't use VLC's config.
    "--verbose=0",
    "--no-video-title-show", // Don't show media title on video.
  };

  // Create a new libvlc instance.
  libvlc_exception_init(&e);
  vlc = libvlc_new(sizeof(vlc_args) / sizeof(vlc_args[0]), vlc_args, &e);
  Raise(&e);

  // Create a media player playing environment.
  vlc_player = libvlc_media_player_new(vlc, &e);
  Raise(&e);
  current_state = STATE_STOPPED;

  // Video frame (VLC video widget)
  VideoFrame *video_frame = new VideoFrame(this);
  SetFrame(video_frame->winId());

  // Seek bar
  seek_bar = new QSlider(Qt::Horizontal, this);
  seek_bar->setFocusPolicy(Qt::NoFocus);
  seek_bar->setMaximum(POSITION_RESOLUTION);
  seek_bar->setDisabled(true);

  // Controls (buttons + volume)
  QHBoxLayout *controls = CreateControls();

  QVBoxLayout *layout = new QVBoxLayout(this);
  layout->setContentsMargins(0, 0, 0, 0);
  layout->addWidget(video_frame);
  layout->addWidget(seek_bar);
  layout->addLayout(controls);
  setLayout(layout);

  // Status label
  status_label = new QLabel(this);
  status_label->setText(tr("Stopped"));

  // Duration label
  duration_label = new QLabel(this);
  duration_label->setText("0:00 / 0:00");

  // Start timer to trigger every 100 ms the UpdateInterface() slot.
  poller = new QTimer(this);
  poller->start(100);
  connect(poller, SIGNAL(timeout()), this, SLOT(UpdateInterface()));
}

Player::~Player()
{
  // Stop playing.
  libvlc_media_player_stop(vlc_player, &e);

  // Free the media player.
  libvlc_media_player_release(vlc_player);

  libvlc_release(vlc);
  Raise(&e);

  delete status_label;
  delete duration_label;
}

void Player::Raise(libvlc_exception_t *e)
{
  if (libvlc_exception_raised(e)) {
    fprintf(stderr, "error: %s\n", libvlc_exception_get_message(e));
    exit(-1);
  }
}

void Player::SetFrame(WId win_id)
{
  // Get our media instance to use our window.
#if defined(Q_OS_WIN)
  libvlc_media_player_set_hwnd(vlc_player, win_id, &e);
#elif defined(Q_OS_MAC)
  libvlc_media_player_set_agl(vlc_player, win_id, &e);
#else // Linux
  libvlc_media_player_set_xwindow(vlc_player, win_id, &e);
#endif
  Raise(&e);
}

bool Player::IsLoaded()
{
  libvlc_media_t *media = libvlc_media_player_get_media(vlc_player, &e);
  libvlc_exception_clear(&e);

  if (media == NULL) {
    return false;
  }

  libvlc_media_release(media);
  return true;
}

bool Player::IsPlaying()
{
  bool is_playing = libvlc_media_player_is_playing(vlc_player, &e);
  libvlc_exception_clear(&e);
  return is_playing;
}

bool Player::IsFinished()
{
  if (vlc_media == NULL) {
    return false;
  }

  libvlc_state_t state = libvlc_media_get_state(vlc_media, &e);
  Raise(&e);

  if (state == libvlc_Ended) {
    return true;
  } else {
    return false;
  }
}

void Player::Load(const QString location)
{
  // Create a new LibVLC media descriptor.
  vlc_media = libvlc_media_new(vlc, location.toAscii(), &e);
  Raise(&e);

  libvlc_media_player_set_media(vlc_player, vlc_media, &e);
  Raise(&e);
}

void Player::Play(const QString location)
{
  qDebug() << "location=" << location;
  Load(location);
  Play();
}

void Player::Play()
{
  if (!IsLoaded()) {
    return;
  }

  if (IsPlaying()) {
    Stop();
  }

  // Play.
  libvlc_media_player_play(vlc_player, &e);
  Raise(&e);

  current_state = STATE_PLAYING;
  status_label->setText(tr("Playing"));
  return;
}

void Player::Stop()
{
  libvlc_media_player_stop(vlc_player, &e);
  Raise(&e);

  seek_bar->setValue(0);

  current_state = STATE_STOPPED;
  status_label->setText(tr("Stopped"));
  return;
}

void Player::Pause()
{
  if (!IsPlaying()) {
    return;
  }

  libvlc_media_player_pause(vlc_player, &e);
  libvlc_exception_clear(&e);

  current_state = STATE_PAUSED;
  status_label->setText(tr("Paused"));
  return;
}

void Player::SetVolume(int volume)
{
  libvlc_exception_clear(&e);
  libvlc_audio_set_volume(vlc, volume , &e);
  Raise(&e);
}

void Player::UpdateInterface()
{
  if (current_state == STATE_PLAYING && IsFinished()) {
    Stop();
    emit Finished();
  }

  if (!IsPlaying()) {
    return;
  }

  // Seek bar
  float position = libvlc_media_player_get_position(vlc_player, &e);
  libvlc_exception_clear(&e);
  seek_bar->setValue((int)(position * (float)(POSITION_RESOLUTION)));

  // Duration label
  qint64 current = libvlc_media_player_get_time(vlc_player, &e);
  libvlc_exception_clear(&e);
  qint64 total = libvlc_media_player_get_length(vlc_player, &e);
  libvlc_exception_clear(&e);

  duration_label->setText(
    QString("%1:%2 / %3:%4")
    .arg((int)(current / 60000))
    .arg(QString().sprintf("%02d", (int)((current % 60000) / 1000)))
    .arg((int)(total / 60000))
    .arg(QString().sprintf("%02d", (int)((total % 60000) / 1000)))
  );
}

QLabel* Player::GetStatusLabel()
{
  return status_label;
}

QLabel* Player::GetDurationLabel()
{
  return duration_label;
}

void Player::SetStatusLabel(const QString status)
{
  status_label->setText(status);
}

QHBoxLayout* Player::CreateControls()
{
  // Buttons
  IconButton *play_button = new IconButton(
    this, QIcon(":/images/play.png"), tr("Play"));
  connect(play_button, SIGNAL(clicked()), this, SLOT(Play()));

  IconButton *pause_button = new IconButton(
    this, QIcon(":/images/pause.png"), tr("Pause"));
  connect(pause_button, SIGNAL(clicked()), this, SLOT(Pause()));

  IconButton *stop_button = new IconButton(
    this, QIcon(":/images/stop.png"), tr("Stop"));
  connect(stop_button, SIGNAL(clicked()), this, SLOT(Stop()));

  IconButton *volume_button = new IconButton(
    this, QIcon(":/images/volume.png"));

  QSlider *volume_slider = new QSlider(Qt::Horizontal, this);
  volume_slider->setFocusPolicy(Qt::NoFocus);
  volume_slider->setMaximumSize(75, 20);
  volume_slider->setMaximum(100); // The volume is between 0 and 100.
  volume_slider->setToolTip(tr("Volume"));
  volume_slider->setValue(50);
  SetVolume(50);
  connect(volume_slider, SIGNAL(sliderMoved(int)), this, SLOT(SetVolume(int)));

  QHBoxLayout *controls = new QHBoxLayout;
  controls->setContentsMargins(0, 0, 0, 0);
  controls->setSpacing(0);
  controls->addWidget(play_button);
  controls->addWidget(pause_button);
  controls->addWidget(stop_button);
  controls->addStretch();
  controls->addWidget(volume_button);
  controls->addWidget(volume_slider);

  return controls;
}

void Player::ShowSeekBar(bool flag = true)
{
  if (flag) {
    seek_bar->show();
  } else {
    seek_bar->hide();
  }
}
