#include <QtDebug>
#include "config.h"
#include "mainwindow.h"
#include "player.h"
#include "video.h"
#include "playlist.h"
#if defined(Q_OS_WIN)
#include <Windows.h>
#endif

MainWindow::MainWindow()
{
  player = new Player(this);
  setCentralWidget(player);
  connect(player, SIGNAL(Finished()), this, SLOT(PlayNext()));

  video = NULL;

  playlist = new Playlist(this);
  connect(playlist, SIGNAL(ItemSelected(QString)),
          this, SLOT(OpenPlaylistItem(QString)));
  CreateMenus();

  setAcceptDrops(true);
  setWindowTitle(tr("TubeBrowser"));
  statusBar()->addWidget(player->GetStatusLabel(), 1);
  statusBar()->addPermanentWidget(player->GetDurationLabel());

  QStringList args = QApplication::arguments();
  if (args.count() > 1) {
    QUrl url = QUrl(args[1]);
    PlayUrl(url);
  }
}

MainWindow::~MainWindow()
{
}

void MainWindow::closeEvent(QCloseEvent *event)
{
  player->Stop();
  event->accept();
}

void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
  // Accept file names or URLs.
  if (event->mimeData()->hasFormat("text/uri-list")) {
    event->acceptProposedAction();
  }
}

void MainWindow::dropEvent(QDropEvent *event)
{
  QList<QUrl> urls = event->mimeData()->urls();
  if (urls.isEmpty()) {
    return;
  }

  is_playlist = false;
  QUrl url = urls.first();
  PlayUrl(url);
}

void MainWindow::PlayRemoteVideo()
{
  player->Stop();
  player->Play(video->GetLocation());
  SetTitle(video->GetTitle());
  SetCurrentFile(video->GetOriginalUrl());
}


void MainWindow::StopRemoteVideo()
{
  player->Stop();
  player->SetStatusLabel(tr("Failed"));
}

void MainWindow::CreateMenus()
{
  // Actions
  QAction *open_file_action = new QAction(tr("&Open"), this);
  open_file_action->setShortcuts(QKeySequence::Open);
  connect(open_file_action, SIGNAL(triggered()), this, SLOT(OpenFile()));

  QAction *open_url_action = new QAction(tr("Open URL..."), this);
  open_url_action->setShortcut(tr("Ctrl+L"));
  connect(open_url_action, SIGNAL(triggered()), this, SLOT(OpenUrl()));

  QAction *copy_url_action = new QAction(tr("Copy Current URL"), this);
  connect(copy_url_action, SIGNAL(triggered()), this, SLOT(CopyCurrentUrl()));

  QAction *download_action = new QAction(tr("Download Current Video"), this);
  connect(download_action, SIGNAL(triggered()),
          this, SLOT(DownloadCurrentVideo()));

  QAction *exit_action = new QAction(tr("E&xit"), this);
  connect(exit_action, SIGNAL(triggered()), this, SLOT(close()));

  QAction *playlist_action = new QAction(tr("Playlist"), this);
  playlist_action->setCheckable(true);
  connect(playlist_action, SIGNAL(triggered()), this, SLOT(TogglePlaylist()));
  connect(playlist, SIGNAL(Closed()),
          playlist_action, SLOT(toggle()));

  QAction *seek_bar_action = new QAction(tr("Seek Bar"), this);
  connect(seek_bar_action, SIGNAL(triggered()), this, SLOT(ToggleSeekBar()));
  seek_bar_action->setCheckable(true);
  seek_bar_action->setChecked(true);

#if defined(Q_OS_WIN)
  QAction *topmost_action = new QAction(tr("On Top"), this);
  topmost_action->setCheckable(true);
  topmost_action->setShortcut(tr("Ctrl+T"));
  connect(topmost_action, SIGNAL(triggered()), this, SLOT(TopMost()));
#endif

  QAction *play_action = new QAction(tr("P&lay"), this);
  connect(play_action, SIGNAL(triggered()), player, SLOT(Play()));

  QAction *stop_action = new QAction(tr("&Stop"), this);
  connect(stop_action, SIGNAL(triggered()), player, SLOT(Stop()));

  QAction *pause_action = new QAction(tr("Pause"), this);
  connect(pause_action, SIGNAL(triggered()), player, SLOT(Pause()));

  QAction *homepage_action = new QAction(tr("Home Page"), this);
  connect(homepage_action, SIGNAL(triggered()), this, SLOT(OpenHomePage()));

  QAction *about_action = new QAction(tr("&About"), this);
  connect(about_action, SIGNAL(triggered()), this, SLOT(About()));

  // Recent files
  for (int i=0; i<MAX_RECENT_FILES; ++i) {
    recent_actions[i] = new QAction(this);
    recent_actions[i]->setVisible(false);
    connect(recent_actions[i], SIGNAL(triggered()),
            this, SLOT(OpenRecentFile()));
  }

  QAction *clear_recent_action = new QAction(tr("Clear All"), this);
  connect(clear_recent_action, SIGNAL(triggered()),
          this, SLOT(ClearRecentFiles()));

  // Menus
  QMenu *file_menu = menuBar()->addMenu(tr("&File"));
  file_menu->addAction(open_file_action);
  file_menu->addAction(open_url_action);

  QMenu *recent_menu = file_menu->addMenu(tr("Recent Files"));
  for (int i=0; i<MAX_RECENT_FILES; ++i) {
    recent_menu->addAction(recent_actions[i]);
  }
  recent_menu->addSeparator();
  recent_menu->addAction(clear_recent_action);
  UpdateRecentFileActions();

  file_menu->addSeparator();
  file_menu->addAction(copy_url_action);
  file_menu->addAction(download_action);

  file_menu->addSeparator();
  file_menu->addAction(exit_action);

  QMenu *view_menu = menuBar()->addMenu(tr("&View"));
  view_menu->addAction(playlist_action);
  view_menu->addAction(seek_bar_action);
#if defined(Q_OS_WIN)
  view_menu->addAction(topmost_action);
#endif

  QMenu *play_menu = menuBar()->addMenu(tr("&Play"));
  play_menu->addAction(play_action);
  play_menu->addAction(pause_action);
  play_menu->addAction(stop_action);

  QMenu *help_menu = menuBar()->addMenu(tr("&Help"));
  help_menu->addAction(homepage_action);
  help_menu->addAction(about_action);
}

void MainWindow::SetTitle(const QString& title)
{
  if (!title.isEmpty()) {
    setWindowTitle(tr("%1 - TubeBrowser").arg(title));
  } else {
    setWindowTitle(tr("TubeBrowser"));
  }
}

void MainWindow::About()
{
  QMessageBox::about(this, tr("About TubeBrowser"),
                     QString("TubeBrowser %1").arg(APP_VERSION));
}

void MainWindow::OpenFile()
{
  // TODO: remember the last used directory.
  QString file_name = QFileDialog::getOpenFileName(
    this,
    tr("Choose a file to open"),
    QString(),
    tr("FLV files (*.flv);;All files (*.*)"));

  if (!file_name.isEmpty()) {
    is_playlist = false;
    player->Stop();
    player->Play(file_name);
    SetTitle(QFileInfo(file_name).fileName());
    SetCurrentFile(file_name);
  }
}

void MainWindow::OpenUrl()
{
  QString default_text = "";
  QClipboard *clipboard = QApplication::clipboard();
  QString tmp = clipboard->text();
  if (tmp.indexOf("http://") != -1) {
    default_text = tmp;
  }

  bool ok;
  QString url = QInputDialog::getText(
    this,
    tr("Open URL..."),
    tr("Enter the address of the file you would like to open:"),
    QLineEdit::Normal,
    default_text,
    &ok
  );

  if (ok && !url.isEmpty()) {
    is_playlist = false;
    PlayUrl(QUrl(url));
  }
}

void MainWindow::OpenHomePage()
{
  QUrl url(PROJECT_URL);
  QDesktopServices::openUrl(url);
}

void MainWindow::PlayNext()
{
  if (!is_playlist) {
    return;
  }

  QString url = playlist->GetNext();
  if (!url.isEmpty()) {
    PlayUrl(QUrl(url));
  }
}

void MainWindow::OpenRecentFile()
{
  QAction *action = qobject_cast<QAction *>(sender());
  if (action) {
    is_playlist = false;
    QUrl url = QUrl(action->data().toString());
    PlayUrl(url);
  }
}

void MainWindow::UpdateRecentFileActions()
{
  QSettings settings(QSettings::IniFormat, QSettings::UserScope,
                     "TubeBrowser", "TubeBrowser", this);
  QStringList files = settings.value("recent_files").toStringList();

  int max = qMin(files.size(), (int)MAX_RECENT_FILES);
  for (int i=0; i<max; ++i) {
    QString text = QString("&%1 %2").arg(i + 1).arg(files[i]);
    recent_actions[i]->setText(text);
    recent_actions[i]->setData(files[i]);
    recent_actions[i]->setVisible(true);
  }
  for (int j=max; j<MAX_RECENT_FILES; ++j) {
    recent_actions[j]->setVisible(false);
  }
}

void MainWindow::SetCurrentFile(const QString& file_name)
{
  QSettings settings(QSettings::IniFormat, QSettings::UserScope,
                     "TubeBrowser", "TubeBrowser", this);
  QStringList files = settings.value("recent_files").toStringList();
  files.removeAll(file_name);
  files.prepend(file_name);
  while (files.size() > MAX_RECENT_FILES) {
    files.removeLast();
  }

  settings.setValue("recent_files", files);
  foreach (QWidget *widget, QApplication::topLevelWidgets()) {
    MainWindow *window = qobject_cast<MainWindow *>(widget);
    if (window) {
      window->UpdateRecentFileActions();
    }
  }
}

void MainWindow::ClearRecentFiles()
{
  QSettings settings(QSettings::IniFormat, QSettings::UserScope,
                     "TubeBrowser", "TubeBrowser", this);
  settings.remove("recent_files");
  UpdateRecentFileActions();
}

#if defined(Q_OS_WIN)
void MainWindow::TopMost()
{
  QAction *action = qobject_cast<QAction *>(sender());
  if (action && action->isChecked()) {
    SetWindowPos(this->winId(), HWND_TOPMOST, 0, 0, 0, 0,
                 SWP_NOMOVE|SWP_NOSIZE);
  } else {
    SetWindowPos(this->winId(), HWND_NOTOPMOST, 0, 0, 0, 0,
                 SWP_NOMOVE|SWP_NOSIZE);
  }
}
#endif

void MainWindow::TogglePlaylist()
{
  QAction *action = qobject_cast<QAction *>(sender());
  if (action && action->isChecked()) {
    playlist->show();
  } else {
    playlist->hide();
  }
}

void MainWindow::ToggleSeekBar()
{
  QAction *action = qobject_cast<QAction *>(sender());
  if (action && action->isChecked()) {
    player->ShowSeekBar(true);
  } else {
    player->ShowSeekBar(false);
  }
}

void MainWindow::OpenPlaylistItem(const QString url)
{
  is_playlist = true;
  PlayUrl(QUrl(url));
}

void MainWindow::PlayUrl(const QUrl& url)
{
  QString file_name = url.toLocalFile();

  if (!file_name.isEmpty()) {
    // TODO: check the file type?
    player->Stop();
    player->Play(file_name);
    SetTitle(QFileInfo(file_name).fileName());
    SetCurrentFile(url.toString());
  } else {
    delete video;
    video = VideoFactory::Create(url);
    video->FindVideo();
    connect(video, SIGNAL(Finished()), this, SLOT(PlayRemoteVideo()));
    connect(video, SIGNAL(Error()), this, SLOT(StopRemoteVideo()));
  }
}

void MainWindow::CopyCurrentUrl()
{
  if (video) {
    qDebug() << "current_url=" << video->GetOriginalUrl();
    QClipboard *clipboard = QApplication::clipboard();
    clipboard->setText(video->GetOriginalUrl());
  }
}

void MainWindow::DownloadCurrentVideo()
{
  if (video) {
    qDebug() << "current_video=" << video->GetLocation();
    QString url = QUrl::fromPercentEncoding(video->GetLocation().toAscii());
    QDesktopServices::openUrl(QUrl(url));
  }
}
