#include <QGridLayout>
#include <QListWidget>
#include <QListWidgetItem>
#include <QPushButton>
#include <QCloseEvent>
#include <QUrl>
#include <QFile>
#include <QFileDialog>
#include <QTextStream>
#include "playlist.h"

Playlist::Playlist(QWidget *parent): QWidget(parent, Qt::Tool)
{
  QPushButton *load_button = new QPushButton(tr("Load"), this);
  connect(load_button, SIGNAL(clicked()), this, SLOT(LoadPlaylist()));

  QPushButton *save_button = new QPushButton(tr("Save"), this);
  connect(save_button, SIGNAL(clicked()), this, SLOT(SavePlaylist()));

  QPushButton *clear_button = new QPushButton(tr("Clear All"), this);
  connect(clear_button, SIGNAL(clicked()), this, SLOT(ClearAll()));

  QPushButton *remove_button = new QPushButton(tr("Delete"), this);
  connect(remove_button, SIGNAL(clicked()), this, SLOT(RemoveEntry()));

  list_widget = new QListWidget(this);
  connect(list_widget, SIGNAL(itemDoubleClicked(QListWidgetItem*)),
          this, SLOT(SelectItem(QListWidgetItem*)));

  QGridLayout *layout = new QGridLayout(this);
  layout->setContentsMargins(0, 0, 0, 0);
  layout->setVerticalSpacing(0);
  layout->setHorizontalSpacing(0);
  layout->addWidget(load_button, 0, 0);
  layout->addWidget(save_button, 0, 1);
  layout->addWidget(remove_button, 0, 2);
  layout->addWidget(clear_button, 0, 3);
  layout->addWidget(list_widget, 1, 0, 1, 4);
  setLayout(layout);

  setAcceptDrops(true);
  setWindowTitle(tr("Playlist"));
  resize(QSize(400, 200));
}

void Playlist::closeEvent(QCloseEvent *event)
{
  event->accept();
  emit Closed();
}

void Playlist::dragEnterEvent(QDragEnterEvent *event)
{
  // Accept file names or URLs.
  if (event->mimeData()->hasFormat("text/uri-list")) {
    event->acceptProposedAction();
  }
}

void Playlist::dropEvent(QDropEvent *event)
{
  QList<QUrl> urls = event->mimeData()->urls();
  if (urls.isEmpty()) {
    return;
  }

  QUrl url = urls.first();
  AddEntry(url.toString());
}

void Playlist::AddEntry(const QString& url)
{
  list_widget->addItem(url);
}

void Playlist::SelectItem(QListWidgetItem *item)
{
  emit ItemSelected(item->text());
}

QString Playlist::GetNext()
{
  list_widget->setCurrentRow(list_widget->currentRow() + 1);
  QListWidgetItem *item = list_widget->currentItem();

  if (item) {
    return item->text();
  } else  {
    return QString();
  }
}

void Playlist::ClearAll()
{
  QListWidgetItem *item;
  while ((item = list_widget->takeItem(0)) != 0) {
    delete item;
  }
}

void Playlist::RemoveEntry()
{
  if (list_widget->currentRow() != -1) {
    QListWidgetItem *item = list_widget->takeItem(list_widget->currentRow());
    delete item;
  }
}

void Playlist::LoadPlaylist()
{
  QString file_name = QFileDialog::getOpenFileName(
    this,
    tr("Choose a file to open"),
    QString(),
    tr("Playlist files (*.txt);;All files (*.*)"));

  if (file_name.isEmpty()) {
    return;
  }

  QFile file(file_name);
  if (!file.open(QIODevice::ReadOnly|QIODevice::Text)) {
    return;
  }

  ClearAll();

  QTextStream in(&file);
  while (!in.atEnd()) {
    QString line = in.readLine();
    AddEntry(line);
  }
}

void Playlist::SavePlaylist()
{
  if (!list_widget->count()) {
    return;
  }

  QString file_name = QFileDialog::getSaveFileName(
    this,
    tr("Save playlist as ..."),
    QString(),
    tr("Playlist files (*.txt);;All files (*.*)"));

  if (file_name.isEmpty()) {
    return;
  }

  QFile file(file_name);
  if (!file.open(QIODevice::WriteOnly|QIODevice::Text)) {
    return;
  }

  QTextStream out(&file);

  int current_row = list_widget->currentRow();
  int count = list_widget->count();

  for (int i=0; i<count; i++) {
    list_widget->setCurrentRow(i);
    QListWidgetItem *item = list_widget->currentItem();
    out << item->text() + "\n";
  }

  list_widget->setCurrentRow(current_row);
  return;
}
