/* -*- c-file-style: "gnu" -*- */
/*************************************************************************
 *   dump.c		* <comment>
 *************************************************************************
 *
 * Description:
 *   <purpose>
 *
 *
 *************************************************************************
 * Copyright (c) 2000 The Pragmatic Programmers, LLC
 *************************************************************************/

#include <ruby.h>
#include <version.h>
#include <node.h>
#include <stdarg.h>

#include "node_names.h"
#include "nd_version.h"

#define SPACES "                                                               "

EXTERN NODE *ruby_eval_tree;
EXTERN NODE *ruby_eval_tree_begin;

struct global_entry {
#if RUBY_VERSION_CODE > 171
  struct global_variable *var;
#endif
  ID id;
};

static void doDump(NODE *n, int level);

static void lp(int level, const char *fmt, ...) {
  va_list ap;
  int i;
  for (i = 0; i < level; i++) {
    putchar(' '); putchar(' ');
  }
  va_start(ap, fmt);
  vprintf(fmt, ap);
  va_end(ap);
}

#define p printf

static void nl(void) {
  putchar('\n');
}

static char * tail(char* s, char *e)
{
  char *t = memchr(s, '\n', e - s);
  if (!t) return e;
  while ((*++t == '\n' || *t == '\r') && t < e);
  return t;
}

static void
dump_cpath(const char *s, NODE *cpath, int level)
{
  ID id;
  if (SPECIAL_CONST_P(cpath)) {
    id = (ID)cpath;
    cpath = 0;
  }
  else {
    id = cpath->nd_mid;
    if (cpath->nd_head) {
      p(" %s %d (%s) in\n", s, id, rb_id2name(id));
      doDump(cpath->nd_head, level+2);
      return;
    }
  }
  p(" class %d (%s)\n", id, rb_id2name(id));
}

static void dumpLiteral(long lit, int level) {
  
  if (FIXNUM_P(lit)) 
    p(" Fixnum: %d\n", FIX2LONG(lit));
  else if (SYMBOL_P(lit)) 
    p(" Symbol: %d (%s)\n", SYM2ID(lit), rb_id2name(SYM2ID(lit)));
  else if (TYPE(lit) == T_STRING) {
    VALUE str;
    char *pbeg, *pend, *s, *t;
    pbeg = RSTRING(lit)->ptr;
    pend = pbeg + RSTRING(lit)->len;
    t = tail(pbeg, pend);
    str = rb_inspect(rb_str_substr(lit, 0, t - pbeg));
    p("%.*s", RSTRING(str)->len, RSTRING(str)->ptr);
    nl();
    while ((s = t) < pend) {
      t = tail(s, pend);
      str = rb_inspect(rb_str_substr(lit, s - pbeg, t - s));
      lp(level, "         %.*s", RSTRING(str)->len, RSTRING(str)->ptr);
      nl();
    }
  }
  else {
    VALUE str = rb_inspect(lit);
    p("%.*s", RSTRING(str)->len, RSTRING(str)->ptr);
    nl();
  }
}

static void dumpValue(VALUE v, int level) {
  dumpLiteral(v, level);
}

static void doDump(NODE *n, int level)
{

  NODE *n1;
  int type;
  int i;
  int id, id1;

  if (!n)
    return;
  
  type = nd_type(n);
  lp(level, "%s: ", node_names[type]);

  switch (type) {

  case NODE_ALIAS:
  case NODE_VALIAS:
    id = n->nd_old;
    id1 = n->nd_new;
    p(" from %d (%s) to %d (%s)\n",
      id, rb_id2name(id), id1, rb_id2name(id1));
    break;

  case NODE_AND:
  case NODE_OR:
    nl();
    ++level;
    do {
      doDump(n->nd_1st, level);
    } while (n->nd_2nd && (type == nd_type(n = n->nd_2nd)));
    doDump(n, level);
    break;
    
  case NODE_ARGS:
    p(" count = %d\n", n->nd_cnt);
    if (n->nd_rest == -1) {
      lp(level+1, "additional default values:\n");
      n1 = n->nd_opt;
      doDump(n1, level+2);
    }
    break;
    
  case NODE_ARGSCAT:
  case NODE_ARGSPUSH:
    nl();
    lp(level+1, "Push onto:\n");
    doDump(n->nd_head, level+2);
    lp(level+1, "Values:\n");
    doDump(n->nd_body, level+2);
    break;  

  case NODE_ARRAY:
#ifdef NODE_VALUES
  case NODE_VALUES:
#endif
    p(" size = %d\n", n->nd_alen);
    n1 = n;
    while(n1) {
      doDump(n1->nd_head, level+1);
      n1 = n1->nd_next;
    }
    lp(level+1, "end:\n");
    break;

  case NODE_ATTRSET:
    id = n->nd_vid;
    p(" %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_BACK_REF:
    p(" ($%c)\n", n->nd_nth);
    break;

  case NODE_BEGIN:
    nl();
    doDump(n->nd_body, level+1);
    break;

  case NODE_BLOCK:
    nl();
    while (n->nd_next) {
      doDump(n->nd_head, level+1);
      n = n->nd_next;
    }
    doDump(n->nd_head,level+1);
    lp(level+1, "end:\n");
    break;

  case NODE_BLOCK_ARG:
    p(" (to LV %d (%s))\n", n->nd_cnt, rb_id2name(n->nd_vid));
    break;

  case NODE_BLOCK_PASS:
    nl();
    doDump(n->nd_body, level+1);
    doDump(n->nd_iter, level+1);
    break;

  case NODE_BMETHOD:
    lp(level+1, "BMETHOD");
    break;
    
  case NODE_BREAK:
    nl();
    if (n->nd_stts) doDump(n->nd_stts, level+1);
    break;

  case NODE_CALL:
  case NODE_FCALL:
#ifdef NODE_ATTRASGN
  case NODE_ATTRASGN:
#endif
    if (type == NODE_FCALL) {
      p(" to function: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
    }
    else {
      p(" to method: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
      lp(level+1, "Receiver:\n");
      if (n->nd_recv == (NODE *)1) {
	lp(level+1, "%s: ", node_names[NODE_SELF]);
      }
      else {
	doDump(n->nd_recv, level+2);
      }
    }
    
    lp(level+1, "Parameters:\n");
    n1 = n->nd_args;
    if (n1) {
      doDump(n1, level+2);
    }
    lp(level+2, "end:\n");
    break;

  case NODE_CASE:
    nl();
    if (n->nd_head)
      doDump(n->nd_head, level+1);
    n = n->nd_body;
    while (n) {
      if (nd_type(n) != NODE_WHEN) {
        lp(level+1, "Else:\n");
        doDump(n, level+2);
        break;
      }
      doDump(n->nd_head, level+1);
      doDump(n->nd_body, level+3);
      n = n->nd_next;
    }
    break;

  case NODE_WHEN:
    nl();
    n1 = n->nd_head;
    while (n1) {
      doDump(n1->nd_head, level+2);
      n1 = n1->nd_next;
    }
    doDump(n->nd_body, level+1);
    break;

#ifdef NODE_CASGN
  case NODE_CASGN:  // UNUSED?
    id = n->nd_vid;
    p(" to constant %d (%s)\n", id, rb_id2name(id));
    doDump(n->nd_value, level+1);
    break;
#endif
    
  case NODE_CDECL:
    id = n->nd_vid;
    p(" to %d (%s)\n", id, rb_id2name(id));
    doDump(n->nd_value, level+1);
    break;

  case NODE_CLASS:
#ifdef nd_cpath
    dump_cpath("class", n->nd_cpath, level);
#else
    id = n->nd_cname;
    p(" class %d (%s)\n", id, rb_id2name(id));
#endif
    if (n->nd_super) {
      lp(level+1, "Superclass:\n");
      doDump(n->nd_super, level+2);
    }
      
    doDump(n->nd_body, level + 1);

    lp(level+1, "end:\n");

    break;
    
  case NODE_COLON2:
    p(" ::%s in:\n", rb_id2name(n->nd_mid));
    doDump(n->nd_head, level+1);
    break;

  case NODE_COLON3:
    p(" ::%s\n", rb_id2name(n->nd_mid));
    break;

  case NODE_CONST:
    p(" %d (%s)\n", n->nd_vid, rb_id2name(n->nd_vid));
    break;

  case NODE_CREF:
    nl();
    if (n->u1.node) {
      //       doDump(n->u1.node, level+1);
     }
    if (n->u2.node) {
      doDump(n->u2.node, level+1);
    }
    if (n->u3.node) {
      doDump(n->u3.node, level+1);
    }
    break;
    
  case NODE_CVAR:
#ifdef NODE_CVAR2
  case NODE_CVAR2:
#endif
    id = n->nd_vid;
    p(" class variable %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_CVDECL:
#ifdef NODE_CVASGN2		/* RUBY_VERSION_CODE < 162 */
  case NODE_CVASGN2:
#endif
#ifdef NODE_CVASGN		/* RUBY_VERSION_CODE < 160 */
  case NODE_CVASGN:
#endif
    id = n->nd_vid;
    p(" to class variable %d (%s)\n", id, rb_id2name(id));
    doDump(n->nd_value, level+1);
    break;

  case NODE_DASGN:
  case NODE_DASGN_CURR:
    id = n->nd_vid;
    p(" to dynamic variable %d (%s)\n", id, rb_id2name(id));
    doDump(n->nd_value, level+1);
    break;

  case NODE_DEFINED:
    nl();
    doDump(n->nd_head, level+1);
    break;

  case NODE_DEFN:
    if (n->nd_defn) {
      p(" method %d (%s)\n", n->nd_mid, rb_id2name(n->nd_mid));
      doDump(n->nd_defn, level+1);
    }
    break;

  case NODE_DEFS:
    if (n->nd_defn) {
      id = n->nd_mid;
      p(" method %d (%s)\n", id, rb_id2name(id));
      lp(level+1, "Receiver:\n");
      doDump(n->nd_recv, level+2);
      doDump(n->nd_defn, level+1);
    }
    break;

#ifdef NODE_DMETHOD
  case NODE_DMETHOD:
    lp(level+1, "DMETHOD");
    break;
#endif
    
  case NODE_DVAR:
    id = n->nd_vid;
    p(" dynamic variable %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_DSTR:
  case NODE_DXSTR:
  case NODE_DREGX:
  case NODE_DREGX_ONCE:
#ifdef NODE_DSYM
  case NODE_DSYM:
#endif
    dumpLiteral(n->nd_lit, level+1);
    n1 = n->nd_next;
    while (n1) {

      doDump(n1->nd_head, level+1);
      n1 = n1->nd_next;

    }
    break;

  case NODE_ENSURE:
    nl();
    doDump(n->nd_head, level+1);
    if (n->nd_ensr) {
      lp(level+1, "Ensure:\n");
      doDump(n->nd_ensr, level+2);
    }   
    break;

#ifdef NODE_ERRINFO
  case NODE_ERRINFO:
    nl();
    break;
#endif

  case NODE_FALSE:
    nl();
    break;

  case NODE_DOT2:
  case NODE_DOT3:
  case NODE_FLIP2:
  case NODE_FLIP3:
    nl();
    lp(level+1, "From:\n");
    doDump(n->nd_beg, level+2);
    lp(level+1, "To:\n");
    doDump(n->nd_end, level+2);
    break;

  case NODE_FBODY:
    p(" for: %d (%s)\n",  n->nd_mid, rb_id2name(n->nd_mid));
    doDump(n->nd_head, level+1);
    break;
    
  case NODE_FOR:
  case NODE_ITER:
    nl();
    
    if (nd_type(n) == NODE_ITER) {
      lp(level+1, "Iterator:\n");
      doDump(n->nd_iter, level+2);
    }
    else {
      lp(level+1, "Each on:\n");
      doDump(n->nd_iter, level+2);
    }
    lp(level+1, "Into:\n");
    doDump(n->nd_var, level+2);
    doDump(n->nd_body, level+1);
    break;
    
  case NODE_GASGN:
    nl();
    doDump(n->nd_value, level+1);
    id = n->nd_entry->id;
    lp(level+1, "Assign to GV %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_GVAR:
    id = n->nd_entry->id;
    lp(level+1, "GV %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_HASH:
    nl();
    n1 = n->nd_head;
    while (n1) {
      doDump(n1->nd_head, level+1);
      n1 = n1->nd_next;
      if (!n1)
        break;
      lp(level+2, "=>\n");
      doDump(n1->nd_head, level+3);
      n1 = n1->nd_next;
    }
    lp(level+1, "end:\n");
    break;

  case NODE_IASGN:
    nl();
    doDump(n->nd_value, level+1);
    lp(level+1, "Assign to IV %d (%s)\n", n->nd_vid, rb_id2name(n->nd_vid));
    break;

  case NODE_IF:
    nl();
    doDump(n->nd_cond, level+1);
    lp(level, "then:\n");
    doDump(n->nd_body, level+1);
    lp(level, "else:\n");
    doDump(n->nd_else, level+1);
    lp(level, "end:\n");
    break;
    
  case NODE_IVAR:
    id = n->nd_vid;
    p(" instance variable %d (%s)\n", id, rb_id2name(id));
    break;

  case NODE_LASGN:
    nl();
    doDump(n->nd_value, level+1);
    lp(level+1, "Assign to LV %d (%s)\n", n->nd_cnt, rb_id2name(n->nd_vid));
    break;

  case NODE_EVSTR:
    if (!n->nd_lit) {
      nl();
      doDump(n->nd_body, level+1);
      break;
    }
  case NODE_LIT:
  case NODE_XSTR:
    dumpLiteral(n->nd_lit, level+1);
    break;

  case NODE_LVAR:
    p(" LV %d (%s)\n", n->nd_cnt, rb_id2name(n->nd_vid));
    break;

  case NODE_MASGN:
    nl();
    doDump(n->nd_head, level+1);
    if (n->nd_args) {
      if (n->nd_args == (NODE *)-1) {
        lp(level+1, "*");
      }
      else
        doDump(n->nd_args, level+2);
    }
    doDump(n->nd_value, level+1);
    break;

 case NODE_MATCH:
    dumpLiteral(n->nd_head->nd_lit, level+1);
    break;

  case NODE_MATCH3:
  case NODE_MATCH2:
    nl();
    doDump(n->nd_recv, level+1);     
    doDump(n->nd_value, level+1);
    break;

  case NODE_METHOD:
    nl();
    break;
    
  case NODE_MODULE:	
#ifdef nd_cpath
    dump_cpath("module", n->nd_cpath, level);
#else
    id = n->nd_cname;
    p(" module %d (%s)\n", id, rb_id2name(id));
#endif
    doDump(n->nd_body, level+1);
    break;

#ifdef NEW_NEWLINE
  case NODE_NEWLINE:
    p(" [%s:%d]\n", n->nd_file, n->nd_nth);
    n1 = n->nd_next;
    doDump(n1, level+1);
    break;
#endif

  case NODE_NEXT:
    nl();
    if (n->nd_stts) doDump(n->nd_stts, level+1);
    break;

  case NODE_NIL:
    nl();
    break;

  case NODE_NOT:
    nl();
    doDump(n->nd_body, level+1);
    break;

  case NODE_NTH_REF:
    p(" ($%d)\n", n->nd_nth);
    break;
    
  case NODE_OP_ASGN1:
    nl();
    doDump(n->nd_recv, level+1);
    lp(level+1, "Indices:\n");
    doDump(n->nd_args->nd_next, level+2);
    switch (n->nd_mid) {
    case 0:
      lp(level+1, "OR\n");
      break;
    case 1:
      lp(level+1, "AND\n");
      break;
    default:
      id = n->nd_mid;
      lp(level+1, "using: %d (%s)\n", id, rb_id2name(id));
    }
    doDump(n->nd_args->nd_head, level+1);
    break;

  case NODE_OP_ASGN2:
    id = n->nd_next->nd_vid;

    p(" to method: %d (%s)\n",  id, rb_id2name(id));
    lp(level, "Receiver:\n");
    doDump(n->nd_recv, level+1);
    switch (n->nd_next->nd_mid) {
    case 0:
      lp(level+1, "OR");
      break;
    case 1:
      lp(level+1, "AND");
      break;
    default:
      id = n->nd_next->nd_mid;
      lp(level+1, "using: %d (%s)\n", id, rb_id2name(id));
    }
    doDump(n->nd_value, level+1);
    break;

  case NODE_OP_ASGN_AND:
  case NODE_OP_ASGN_OR:
    nl();
    lp(level+1, "First:\n");
    doDump(n->nd_head, level+2);
    lp(level+1, "Second:\n");
    doDump(n->nd_value, level+2);
    break;

  case NODE_OPT_N:
    nl();
    doDump(n->nd_body, level+1);
    break;
    
  case NODE_POSTEXE:
    nl();
    break;

  case NODE_REDO:
    nl();
    break;

  case NODE_RESCUE:
    nl();
    doDump(n->nd_head, level+1);
    n1 = n->nd_resq;
    while (n1) {
      lp(level+1, "When:\n");
      if (!n1->nd_args)
        lp(level+2, "StandardError (default)\n");
      else
        doDump(n1->nd_args, level+2);
      
      doDump(n1->nd_body, level+3);
      n1 = n1->nd_head;
    }
    if (n->nd_else) {
      lp(level+1, "Else:\n");
      doDump(n->nd_else, level+2);
    }
    break;

#ifdef NODE_RESTARGS
  case NODE_RESTARGS:
#endif
#ifdef NODE_RESTARY
  case NODE_RESTARY:
#endif
#ifdef NODE_RESTARY2
  case NODE_RESTARY2:
#endif
#ifdef NODE_REXPAND
  case NODE_REXPAND:
#endif
#ifdef NODE_SPLAT
  case NODE_SPLAT:
#endif
#ifdef NODE_TO_ARY
  case NODE_TO_ARY:
#endif
#ifdef NODE_SVALUE
  case NODE_SVALUE:
#endif
    nl();
    doDump(n->nd_head, level+1);
    break;

  case NODE_RETRY:
    nl();
    break;

  case NODE_RETURN:
    if (n->nd_stts) {
      nl();
      doDump(n->nd_stts, level+1);
    }
    else
      p(" (nil)\n");
    break;

  case NODE_SCLASS:
    nl();
    lp(level+1, "Receiver:\n");
    doDump(n->nd_recv, level+2);
    doDump(n->nd_body->nd_body, level+1);
    break;

  case NODE_SCOPE:
    nl();
    doDump(n->nd_next, level+1);
    break;

  case NODE_SELF:
    nl();
    break;

  case NODE_STR:
    putchar(' ');
    dumpLiteral(n->nd_lit, level + 1);
    break;

  case NODE_SUPER:
    nl();
    doDump(n->nd_args, level+1);
    break;
    
  case NODE_TRUE:
    nl();
    break;

  case NODE_UNDEF:
    id = n->nd_mid;
    p(" %d (%s)\n", id, rb_id2name(id));
    break;


  case NODE_UNTIL:
  case NODE_WHILE:
    nl();
    if (n->nd_state) {
      lp(level+1, "Condition:\n");
      doDump(n->nd_cond, level+2);
    }
    lp(level+1, "Body:\n");
    doDump(n->nd_body, level+2);
    if (!n->nd_state) {
      lp(level+1, "Condition:\n");
      doDump(n->nd_cond, level+2);
    }
    break;
    
  case NODE_VCALL:
    p(" self.%s\n", rb_id2name(n->nd_mid));
    break;
    
  case NODE_YIELD:
    nl();
    if (n->nd_stts) {
      doDump(n->nd_stts, level+1);
    }
    break;

  case NODE_ZARRAY:
    p(" []\n");
    break;

  case NODE_ZSUPER:
    nl();
    break;
    
  default:
    puts("\n\n** unhandled**\n\n");
    break;


  }
}

static VALUE t_dump(void)
{
  NODE *n = ruby_eval_tree;

  if (ruby_eval_tree_begin) {
    doDump(ruby_eval_tree_begin, 0);
  }
  
  if (n) {
    p("NodeDump V" ND_VERSION "\n\n");
    doDump(n, 0);
    printf("\n");
    return Qtrue;
  }
  return Qfalse;
}

static VALUE t_trigger()
{
  if(t_dump()){
    exit(1);
  }
  return Qnil;
}

void Init_NodeDumpT()
{
  VALUE method;
#ifdef HAVE_RB_PROC_NEW
  method = rb_proc_new(t_trigger, Qnil);
#else
  VALUE cNodeDump;

  cNodeDump = rb_class_new_instance(0, 0, rb_cObject);
  rb_define_singleton_method(cNodeDump, "trigger", t_trigger, 6);

  method = rb_funcall(cNodeDump, rb_intern("method"), 1, ID2SYM(rb_intern("trigger")));
  method = rb_funcall3(method, rb_intern("to_proc"), 0, 0);
#endif
  method = rb_funcall(rb_mKernel, rb_intern("set_trace_func"),
                      1,
                      method);
  ruby_eval_tree_begin = ruby_eval_tree = 0;
}
