#!/usr/local/bin/perl -w
# $Id: trickster-lite.pl,v 1.17 2002/12/17 09:15:21 miyagawa Exp $

use strict;

use POE qw(Component::Server::TCP Filter::HTTPD Filter::Stream);
use Data::Properties;

# DEBUG
use Data::Dumper;
$Data::Dumper::Indent = 1;

my $Debug = 0;

# $Config: global configuration
my $conf_path = _find_config();
my $Config = TricksterLite::Config->new($conf_path);

_check_installation($Config);
_make_pid_file($Config, $$);

# streamer: watching directory and put it into streamer
TricksterLite::Streamer->create;

# server: HTTP streaming server
TricksterLite::Server->create;

# listeners directory
TricksterLite::Listeners->cleanup;

$poe_kernel->run;

END {
    if ($Config) {
	my $base_dir = $Config->get('base_directory');
	unlink "$base_dir/logs/trickster-lite.pid";
    }
}

sub _find_config {
    my @cand = (
	$ENV{TRICKSTER_LITE_CONF},
	"$ENV{HOME}/.trickster-lite.conf",
	'trickster-lite.conf',
    );
    for my $file (@cand) {
	return $file if $file && -e $file;
    }
    return;
}

sub _check_installation {
    my $config = shift;
    my $base_dir = $config->get('base_directory');
    for my $dir ( qw(listeners queue queue/history logs) ) {
	unless (-e "$base_dir/$dir" && -d _) {
	    mkdir "$base_dir/$dir", 0755 or die $!;
	}
    }
}

sub _make_pid_file {
    my($config, $pid) = @_;
    my $base_dir = $config->get('base_directory');
    my $fh = FileHandle->new("> $base_dir/logs/trickster-lite.pid") or die $!;
    $fh->print($pid, "\n");
}

#--------------------------------------------------

package TricksterLite::Streamer;
use POE;
use DirHandle;
use FileHandle;
use File::Copy;
use Time::HiRes qw(gettimeofday);

sub create {
    POE::Session->create(
	package_states =>
	    [ 'TricksterLite::Streamer' => [ qw(_start start_stream stream sigstop) ] ],
    );
}

sub _start {
    my($kernel, $heap, $dir) = @_[KERNEL, HEAP, ARG0];
    $kernel->alias_set('streamer');
    $kernel->sig(TSTP => 'sigstop');
    $kernel->yield('start_stream');
}

sub sigstop {
    warn "Ctrl-Z pressed" if $Debug;
    local $SIG{TSTP} = 'DEFAULT';
    kill TSTP => $$;
    $_[KERNEL]->sig_handled();
}

sub start_stream {
    my($kernel, $heap) = @_[KERNEL, HEAP];

    my $next_song = _next_song() || $Config->get('default_sound');
    warn "playing $next_song" if $Debug;

    my $media = TricksterLite::MediaFile->new($next_song);
    if ($media) {
	# set start time & file
	$heap->{time} = gettimeofday;
	$heap->{current} = $media;
	$kernel->yield('stream');
    } else {
	# Oops, file not found
	my $dir = $Config->get('base_directory');
	unlink "$dir/queue/now_playing";
	$kernel->yield('start_stream');
    }
}

sub _next_song {
    my $dir = $Config->get('base_directory') . '/queue';
    warn "directory is $dir" if $Debug;
    my $dh = DirHandle->new($dir) or return;
    my @files = sort { -M $b <=> -M $a } grep -f, map "$dir/$_", $dh->read;
    if (@files) {
	# slurp file path
	my $in = FileHandle->new($files[0]) or die "can't open $files[0]";
	chomp(my $song_path = <$in>);
	$in->close;

	# rename to "now_playing" file
	rename $files[0], "$dir/now_playing" or die "now_playing: $!";
	return $song_path;
    }
    return;
}

sub stream {
    my($kernel, $heap) = @_[KERNEL, HEAP];
    my $dir = $Config->get('base_directory') . '/queue';

    if (-e "$dir/now_playing.skip") {
	warn "force skipped!" if $Debug;
	_push_history($dir, "now_playing.skip");
	return $kernel->yield('start_stream');
    }

    my $media = $heap->{current};
    my $bytes_read = $media->fh->sysread(
	my $buffer = '', $Config->get('buffer_size'),
    ) or do {
	warn "no bytes left, skip to next song" if $Debug;
	_push_history($dir, "now_playing") if -e "$dir/now_playing";
	return $kernel->yield('start_stream');
    };

    my $delay = $media->delay_for($bytes_read);
    warn "delay is $delay for $bytes_read" if $Debug;
    _send_buffer($kernel, $heap, $buffer);

    $heap->{time} += $delay;
    warn "set alarm to $heap->{time}" if $Debug;
    $kernel->alarm_add(stream => $heap->{time});
}

sub _send_buffer {
    my($kernel, $heap, $buffer) = @_;
    for my $session_id (TricksterLite::Listeners->sessions) {
	$kernel->post($session_id => send_stream => $buffer);
    }
}

sub _push_history {
    my($dir, $file) = @_;
    my $hist_dir = "$dir/history";
    mkdir $hist_dir, 0755 unless -d $hist_dir;
    my $timestamp = gettimeofday;
    rename "$dir/$file", "$dir/history/$timestamp" or die "rename to $timestamp failed";
}

#--------------------------------------------------

package TricksterLite::Server;
use POE;
use HTTP::Response;

sub create {
    POE::Component::Server::TCP->new(
	Alias    => 'server',
	Port  	 => $Config->get('streaming_port'),
	ClientFilter => 'POE::Filter::HTTPD',
	InlineStates => {
	    start_stream => \&start_stream,
	    send_stream  => \&send_stream,
	},
	ClientDisconnected => sub {
	    my $session_id = $_[SESSION]->ID;
	    warn "disconnected: $session_id" if $Debug;
	    TricksterLite::Listeners->unregister($session_id);
	},
	ClientError => sub {
	    my $session_id = $_[SESSION]->ID;
	    warn "error: $session_id" if $Debug;
	    TricksterLite::Listeners->unregister($session_id);
	    $_[KERNEL]->yield("shutdown");
	},
	ClientInput => sub {
	    my($kernel, $heap, $request, $session) = @_[KERNEL, HEAP, ARG0, SESSION];
	    if ($request->isa('HTTP::Response')) {
		$heap->{client}->put($request);
		Trickster::Listeners->unregister($session->ID);
		$kernel->yield('shutdown');
		return;
	    }

	    # The request is real and fully formed.  Create and send back
	    # headers in preparation for streaming the music.

	    TricksterLite::Listeners->register($session->ID, $request, $heap->{remote_ip});
	    my $response = HTTP::Response->new(200);
	    $response->push_header( 'Content-type' => 'audio/x-mpeg' );
	    $response->push_header( 'X-Audiocast-name' => $Config->get('castname') );
	    _push_shout_header($response) if $request->header('Icy-MetaData');

	    $heap->{client}->put($response);

	    # Note that we do not shut down here.  Once the response's
	    # headers are flushed, the ClientFlushed callback will begin
	    # streaming the actual content.
	    $kernel->yield('start_stream');
	},
    );
}

sub _push_shout_header {
    my $response = shift;
    $response->push_header('Icy-Name' => $Config->get('castname'));
#    $response->push_header('Icy-MetaInt' => 8192);
}

sub start_stream {
    my($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
    warn "start stream: ", $session->ID if $Debug;
    $heap->{client}->set_output_filter(POE::Filter::Stream->new);
}

sub send_stream {
    my($kernel, $heap, $session, $buffer) = @_[KERNEL, HEAP, SESSION, ARG0];
    warn "send stream: ", $session->ID if $Debug;
    unless ($heap->{client}){
	warn "client goes away! ", $session->ID if $Debug;
	$kernel->yield('shutdown');
	TricksterLite::Listeners->unregister($session->ID);
	return;
    }
    my $pending_bytes = $heap->{client}->get_driver_out_octets();
    warn "pending_bytes: ", $pending_bytes, " at ", $session->ID if $Debug;
    $heap->{client}->put($buffer);
}

#--------------------------------------------------

package TricksterLite::MediaFile;
use MP3::Info;

sub new {
    my($class, $path) = @_;
    my $fh   = FileHandle->new($path) or return;
    bless {
	path => $path,
	info => scalar get_mp3info($path),
	tag  => scalar get_mp3tag($path),
	fh   => $fh,
    }, $class;
}

sub path { shift->{path} }
sub info { shift->{info} }
sub tag  { shift->{tag} }
sub fh   { shift->{fh} }

sub title  { shift->{tag}->{TITLE} }
sub artist { shift->{tag}->{ARTIST} }

sub delay_for {
    my($self, $bytes) = @_;
    return $bytes * 8 / $self->{info}->{BITRATE} / 1000;
}

#--------------------------------------------------

package TricksterLite::Listeners;
use DirHandle;
use FileHandle;

my(%listeners, %is_shout);

sub cleanup {
    my $class = shift;
    my $dir = $Config->get('base_directory') . '/listeners';
    my $dh  = DirHandle->new($dir);
    unlink $_ for grep -f, map "$dir/$_", $dh->read;
}

sub sessions {
    my $class = shift;
    return keys %listeners;
}

sub unregister {
    my($class, $sid) = @_;
    my $dir = $Config->get('base_directory') . '/listeners';
    warn "unlinking $dir/$sid" if $Debug;
    unlink "$dir/$sid";
    delete $listeners{$sid};
    delete $is_shout{$sid};
}

sub register {
    my($class, $sid, $request, $addr) = @_;
    warn "registering $sid - $addr" if $Debug;
    my $agent = $request->header('User-Agent') || 'unknown';
    my $uid   = $request->uri->query || '';
    warn "agent = $agent, uid = $uid" if $Debug;
    my $dir = $Config->get('base_directory') . '/listeners';
    my $out = FileHandle->new("> $dir/$sid");
    $out->print(<<EOF);
Remote-Addr: $addr
User-Agent: $agent
UID: $uid
EOF
    ;
    $listeners{$sid} = 1;
    $is_shout{$sid} = $request->header('Icy-MetaData');
}

sub is_shout_client {
    my($class, $sid) = @_;
    return $is_shout{$sid};
}

#--------------------------------------------------

package TricksterLite::Config;
use base qw(Data::Properties);
use FileHandle;

sub new {
    my($class, $path) = @_;
    my $self = $class->SUPER::new;
    $self->load(FileHandle->new($path));
    return $self;
}

BEGIN {
    *get = __PACKAGE__->can('get_property');	# alias
}

__END__

=head1 NAME

trickster-lite - Pure Perl MP3 Streaming Server

=head1 INSTALL

=head2 HOW TO SET UP

copy stub C<trickster-lite.conf.sample> to anywhere your like, and
edit it. Then set C<TRICKSTER_LITE_CONF> envirionment variable to its
path.

  % cp trickster-lite.conf.sample ~/.trickster-lite.conf
  % vi ~/.trickster-lite.conf
  % setenv TRICKSTER_LITE_CONF ~/.trickster-lite.conf

=head2 HOW TO RUN

just run C<trickster-lite.pl>, which runs on foreground.

  % ./trickster-lite.pl &

=head1 USAGE

=head2 HOW TO ENQUEUE YOUR SONG

make a text file which has a path to mp3 file.

  % echo /path/to/file.mp3 > queue/blahblah

=head2 HOW DO I SEE WHAT'S PLAYING RIGHT NOW

see C<now_playing> file in C<queue> directory

  % cat queue/now_playing

=head2 HOW TO SKIP CURRNET PLAYING SONG

rename C<now_playing> file to C<now_playing.skip>

  % cd queue
  % mv now_playing now_playing.skip

=head2 HOW DO I SEE WHO'S LISTENING

listeners are stored in C<listeners> directory. filenames are session
ids used in POE.

  % cat listeners/*

=head2 HOW DO I SEE WHICH SONGS PLAYED PAST

past songs are stored on C<history> directory inside C<queue>
directory. Filenames are timestamps with milliseconds.

  % cat queue/history/*

You can freely remove these files.

=head1 AUTHOR

Tatsuhiko Miyagawa E<lt>miyagawa@bulknews.netE<gt>

=cut
