#!/usr/local/bin/perl
# $Id: trickster-lite.cgi,v 1.17 2002/12/17 09:18:03 miyagawa Exp $
#
# Tatsuhiko Miyagawa <miyagawa@edge.co.jp>
# Livin' On The EDGE, Limited.
#

use strict;

package TricksterLite;

use vars qw($VERSION);
$VERSION = 0.01;

use CGI;
use CGI::Cookie;
use DirHandle;
use Digest::MD5 qw(md5_hex);
use File::Copy;
use File::Spec;
use FileHandle;
use Jcode;
use Template;
use Time::HiRes qw(gettimeofday usleep);
use URI;
use URI::QueryParam;

use vars qw($tt $Config);

my $config_path = $ENV{TRICKSTER_LITE_CONF} || '../trickster-lite.conf';
$Config = TricksterLite::Config->new($config_path);

$tt ||= Template->new({
    INCLUDE_PATH => $Config->get('template_dir'),
    FILTERS => {
	encoding => sub { Jcode->new(shift)->utf8 }, # XXX utf8?
    },
});

my $query = CGI->new;
my %jar   = CGI::Cookie->fetch;
my $uid   = exists $jar{trickster_uid} ? $jar{trickster_uid}->value : make_new_uid();

my $vars  = {
    config   => $Config,
    script   => $query->url,
    version  => $VERSION,
    uid      => $uid,
};

my $charset = $Config->get('http_charset');
$query->charset($charset) if $charset;

my %command = (
    browse  => \&do_browse,
    enqueue => \&do_enqueue,
    m3u     => \&do_m3u,
    skip    => \&do_skip,
);

my $action = $query->param('.action') || 'browse';
my $command = $command{$action} || \&do_error;
my $rc = $command->($query, $tt, $vars);

if ($rc) {
    my $new_cookie = CGI::Cookie->new(
	-name    => 'trickster_uid',
	-value   => $uid,
	-expires => '+7d',
    );
    print $query->header(-cookie => $new_cookie);

    eval {
	$tt->process("$action.tt", $vars) or die $tt->error;
    };
    if ($@) {
	$vars->{error} = $@;
	$tt->process("error.tt", $vars);
    }
}

sub make_new_uid {
    return substr(md5_hex(time . $$ . {}, rand(1000000)), 0, 16);
}

#--------------------------------------------------

sub do_error {
    my($query, $tt, $vars) = @_;
}

#--------------------------------------------------

sub do_browse {
    my($query, $tt, $vars) = @_;
    my $file = TricksterLite::Util::now_playing_file();
    $vars->{media} = $file if $file;
    $vars->{queue} = [ TricksterLite::Util::queue_files() ];
    $vars->{listeners} = [ TricksterLite::Util::listeners() ];

    my $subdir = $query->param('dir') || "";
    my $dir     = File::Spec->catfile($Config->get('media_directory'), $subdir);
    $vars->{base_dir} = $Config->get('media_directory');
    $vars->{subdir}   = $subdir;
    $vars->{dir}      = $dir;

    if ($subdir) {
	my @subdirs = File::Spec->splitdir($subdir);
	$vars->{parent_dir} = File::Spec->catfile(@subdirs[0..$#subdirs-1]);
    }

    return 1;
}

#--------------------------------------------------

sub do_skip {
    my($query, $tt, $vars) = @_;
    my $queue_dir = $Config->get('base_directory') . '/queue';

    # copy to temporary file
    my $tmp = "$queue_dir/now_playing.$$";
    copy("$queue_dir/now_playing", $tmp);

    # append: who skipped this song?
    my $fh = FileHandle->new(">> $tmp") or die "$tmp: $!";
    $fh->print("Skip: $vars->{uid}\n");
    $fh->close;

    # then rename now_playing
    rename $tmp, "$queue_dir/now_playing";
    rename "$queue_dir/now_playing", "$queue_dir/now_playing.skip";
    TricksterLite::Util::wait_next_song();

    $query->param('.action' => 'browse');
    print $query->redirect($query->self_url);
    return;
}

#--------------------------------------------------

sub do_m3u {
    my($query, $tt, $vars) = @_;
    my $url = URI->new($query->url);
       $url->port($Config->get('streaming_port'));
       $url->path("/");
       $url->query($vars->{uid});
    print $query->header(-type => 'audio/x-mpegurl'), $url;
    return;
}

#--------------------------------------------------

sub do_enqueue {
    my($query, $tt, $vars) = @_;
    my $sound_file = File::Spec->catfile($query->param('dir'), $query->param('file'));
    my $timestamp = gettimeofday();
    my $dir = $Config->get('base_directory') . '/queue';
    my $fh = FileHandle->new("> $dir/$timestamp");
    $fh->print($Config->get('media_directory'), "/$sound_file\n");
    $fh->print("Request: $vars->{uid}\n"); # who requested this song?
    $fh->close;

    my $uri = URI->new($query->self_url);
    $uri->query_param('dir' => $query->param('dir'));
    $uri->query_param('.action' => 'browse');
    TricksterLite::Util::wait_next_song();
    print $query->redirect($uri);
    return;
}

#--------------------------------------------------

package TricksterLite::Util;

sub now_playing_file {
    my $base_dir = $TricksterLite::Config->get('base_directory');
    return slurp_file("$base_dir/queue/now_playing");
}

sub slurp_file {
    my $filename = shift;
    my $fh = FileHandle->new($filename) or return;
    chomp(my $path = <$fh>);
    return $path;
}

sub queue_files {
    my $queue_dir = $TricksterLite::Config->get('base_directory') . '/queue';
    my $dh = DirHandle->new($queue_dir) or die "$queue_dir: $!";
    return map TricksterLite::Queue->new($_),
	sort { -M $b <=> -M $a } grep -f && !/now_playing/, map "$queue_dir/$_", $dh->read;
}

sub listeners {
    my $dir = $TricksterLite::Config->get('base_directory') . '/listeners';
    my $dh  = DirHandle->new($dir) or die "$dir: $!";
    my @listeners = map {
	my $fh = FileHandle->new($_);
	my %property;
	while (<$fh>) {
	    chomp;
	    my($key, $val) = split /: /, $_, 2;
	    $key =~ s/-/_/g;
	    $property{lc($key)} = $val;
	}
	\%property;
    } grep -f, map "$dir/$_", $dh->read;
}

sub wait_next_song {
    my $base_dir = $TricksterLite::Config->get('base_directory');
    my $try = 0;
    my $max = $TricksterLite::Config->get('max_waiting_try') || 1_000;
    while ($try++ <= $max) {
	last if -e "$base_dir/queue/now_playing";
	Time::HiRes::usleep 1_000;
    }
}

#--------------------------------------------------

package TricksterLite::Queue;

sub new {
    my($class, $path) = @_;
    my $media_path = TricksterLite::Util::slurp_file($path);
    bless {
	path  => $path,
	media => $media_path,
    }, $class;
}

#--------------------------------------------------

package TricksterLite::Config;
use base qw(Data::Properties);
use FileHandle;

sub new {
    my($class, $path) = @_;
    my $self = $class->SUPER::new;
    $self->load(FileHandle->new($path));
    return $self;
}

BEGIN {
    *get = __PACKAGE__->can('get_property');	# alias
}
