/* Copyright(C) 2004 Toshihiro Ookubo
*/

#include <stdlib.h>
#include <stdio.h>   
#include <string.h>
#include <assert.h>

/*===========================================================
       v0
       /\
      /  \  
  e2 /    \ e1
    /      \
    --------
   v1  e0   v2
===========================================================*/

/* data structure */

typedef struct tface{
  int   id; 
  int   v[3];  
  struct tface *f[3];  
  struct tface *next;  
} TFACE;

int  nPos;
double (*pos)[2];

int  nTri;
TFACE*  top;


/* stack tool */
#define STACKSIZE 1000
typedef struct fstack{
  int    size;
  TFACE  *f[STACKSIZE];     /* current face */
  TFACE  *of[STACKSIZE];    /* opposite face */
} FSTACK;

void stack_init(FSTACK* st)
{ 
  st->size = 0;
}

void stack_push_back(FSTACK* st,TFACE *f,TFACE *of)
{
  st->f[st->size] = f;
  st->of[st->size] = of;
  st->size++;
  assert(st->size < STACKSIZE);
}

void stack_pop_back(FSTACK* st,TFACE **f,TFACE **of)
{
  if(st->size == 0){
    *f  =  NULL;
    *of  =  NULL;
    return;
  }
  st->size--;
  *f   =  st->f[st->size];
  *of  =  st->of[st->size];
}


/*===========================================================
  function 
===========================================================*/
void  dela_print()
{
  int i = 0;
  int f0,f1,f2;
  TFACE* f;
  FILE *fout = fopen("debug.dat","w");

  fprintf(fout,"%s %d\n","$vertex",nPos + 4);
  for(i = 0; i < nPos + 4 ;i++){
    fprintf(fout,"%f %f %f\n",pos[i][0],pos[i][1],0.0);
  }
   
  fprintf(fout,"%s %d\n","$triangle",nTri);

  f = top;
  i = 0;
  while(f != NULL){
    f->id = i; i++;
    f = f->next;
  }

  f = top;
  while(f != NULL){
    fprintf(fout,"v(%d, %d, %d)  ", f->v[0],f->v[1],f->v[2]);
    f0 = (f->f[0] == NULL) ? -1 :f->f[0]->id;
    f1 = (f->f[1] == NULL) ? -1 :f->f[1]->id;
    f2 = (f->f[2] == NULL) ? -1 :f->f[2]->id;
    fprintf(fout,"f(%d, %d, %d)\n", f0,f1,f2); 
    f = f->next;
  }

  fclose(fout);
}

void  face_replace(TFACE *f[3],TFACE *a,TFACE *b)
{
  if(f[0] == a)  { f[0] = b; return; }   
  if(f[1] == a)  { f[1] = b; return; }   
  if(f[2] == a)  { f[2] = b; return; }   
}

int  face_index(TFACE *f[3],TFACE *a)
{
  if(f[0] == a)  return 0;
  if(f[1] == a)  return 1;
  if(f[2] == a)  return 2;
  return -1;
}

int  index_select(int a[3],int b)
{
  if(a[0] == b)  return 0;
  if(a[1] == b)  return 1;
  if(a[2] == b)  return 2;
  return -1;
}


int dela2d_read(char* fileName)
{
  int i;
  TFACE *tri0,*tri1;
  double  xMinMax[2], yMinMax[2];
  double  x,y,z;
  double  xRange,yRange;
  
  FILE *fin = fopen(fileName,"r");
  if(fin == NULL)
    return -1;

  /* read  node */
  fscanf (fin, "%d \n", &nPos);
  if(nPos < 1) 
    return -2;
  pos=  (double (*)[2])calloc(nPos + 4,sizeof(double[2]));
  for(i = 4; i < nPos + 4 ;i++){
    fscanf (fin,  "%lf %lf %lf\n", &x,&y,&z);
    pos[i][0] = x; pos[i][1] = y;
  }

  fclose(fin);

  /* min max  */
  xMinMax[0] = pos[4][0];  xMinMax[1] = pos[4][0];
  yMinMax[0] = pos[4][1];  yMinMax[1] = pos[4][1];

  for(i = 5; i < nPos + 4 ;i++){
    if(xMinMax[0] > pos[i][0]) xMinMax[0] = pos[i][0];
    if(xMinMax[1] < pos[i][0]) xMinMax[1] = pos[i][0];
    if(yMinMax[0] > pos[i][1]) yMinMax[0] = pos[i][1];
    if(yMinMax[1] < pos[i][1]) yMinMax[1] = pos[i][1];
  }

  xRange = xMinMax[1]  - xMinMax[0]; 
  yRange = yMinMax[1]  - yMinMax[0]; 

  xMinMax[0] = xMinMax[0] - xRange; xMinMax[1] = xMinMax[1] + xRange;
  yMinMax[0] = yMinMax[0] - yRange; yMinMax[1] = yMinMax[1] + yRange;

  /* initial triangle */
  pos[0][0] = xMinMax[0];  pos[0][1] = yMinMax[0];
  pos[1][0] = xMinMax[1];  pos[1][1] = yMinMax[0];
  pos[2][0] = xMinMax[0];  pos[2][1] = yMinMax[1];
  pos[3][0] = xMinMax[1];  pos[3][1] = yMinMax[1];

  tri0 = (TFACE*) malloc( sizeof(TFACE) ); 
  tri1 = (TFACE*) malloc( sizeof(TFACE) ); 

  tri0->v[0]= 0;    tri0->v[1]= 3;    tri0->v[2]= 2;
  tri0->f[0]= NULL; tri0->f[1]= NULL; tri0->f[2]= tri1;
  tri0->next = tri1;

  tri1->v[0]= 0;    tri1->v[1]= 1;    tri1->v[2]= 3;
  tri1->f[0]= NULL; tri1->f[1]= tri0; tri1->f[2]= NULL;
  tri1->next = NULL;
 
  top = tri0;
  nTri = 2;

  return 0;
}


double dela2d_area(double p0[2],
                   double p1[2],
                   double p2[2])
{
  return (     (p1[0] - p0[0]) * (p2[1] - p0[1]) 
             - (p1[1] - p0[1]) * (p2[0] - p0[0]));
}

int  dela2d_circle_in(double p0[2],
                      double p1[2],
                      double p2[2],
                      double p[2])
{
  double center[2];
  double r,r1,xc,yc,xp,yp;
  double xi = p0[0]; double yi = p0[1];
  double xj = p1[0]; double yj = p1[1];
  double xk = p2[0]; double yk = p2[1];

  double xij = 0.5 * (xi + xj );  double yij = 0.5 * (yi + yj );
  double xjk = 0.5 * (xj + xk );  double yjk = 0.5 * (yj + yk );
  double num = (xij - xjk) * (xj - xi) + (yij - yjk) * (yj - yi);
  double dn =  (xj  - xi ) * (yk - yj) - (xk  -  xj) * (yj - yi);

  if(dn > 0){
    center[0] =  xjk + num / dn * (yk - yj);
    center[1] =  yjk - num / dn * (xk - xj);
  }else{
    fprintf(stderr,"ERR  calc center of circle\n");
    return 0;
  }

  /* check in circle */
  xc = p0[0] - center[0];  yc = p0[1] - center[1];
  xp = p[0]  - center[0];  yp = p[1]  - center[1];
  r = xc * xc + yc * yc;  r1 = xp * xp + yp * yp;
  if( r > r1){
    return 1;
  }

  return 0;
}



int  dela2d_insert(int node)
{
  TFACE *f = top;
  TFACE *t = NULL;
  TFACE *f0,*f1,*f2;
  TFACE *n0,*n1,*n2,*n3;
  FSTACK stack;
  int v0,v1,v2,v3;
  int v[3];

  /* search tri */
  while(f != NULL){
    /* node include  check */
    if(      (dela2d_area(pos[f->v[0]],pos[f->v[1]],pos[node]) >= 0.0) 
        &&   (dela2d_area(pos[f->v[1]],pos[f->v[2]],pos[node]) >= 0.0)
	&&   (dela2d_area(pos[f->v[2]],pos[f->v[0]],pos[node]) >= 0.0) ){
       t = f;
       break;
    }else{
       f = f->next;
    }
  }

  if(t == NULL){
    fprintf(stderr,"ERR  Not found triangle \n");
     return -1;
  }

  /* insert node */

  f0 = (TFACE*) malloc( sizeof(TFACE) ); 
  f1 = (TFACE*) malloc( sizeof(TFACE) ); 

  f0->v[0] = node;     f0->v[1]= f->v[1]; f0->v[2]= f->v[2];
  f0->f[0] = f->f[0];  f0->f[1]= f1;      f0->f[2]= f;
  f0->next = f1;

  if(f->f[0] != NULL)  face_replace(f->f[0]->f,f,f0);

  f1->v[0]= node;     f1->v[1]= f->v[2]; f1->v[2]= f->v[0];
  f1->f[0]= f->f[1];  f1->f[1]= f;       f1->f[2]= f0;

  f1->next = top;

  if(f->f[1] != NULL)  face_replace(f->f[1]->f,f,f1);

  top = f0;
  f2 = f;

  v[0] = f->v[0];     v[1] = f->v[1];   v[2] = f->v[2];
  f2->v[0]= node;     f2->v[1]= v[0];   f2->v[2]= v[1];
  f2->f[0]= f->f[2];  f2->f[1]= f0;     f2->f[2]= f1;

  nTri += 2;

  /* initial stack */
  stack_init(&stack);
  if(f0->f[0] != NULL) stack_push_back(&stack,f0,f0->f[0]);
  if(f1->f[0] != NULL) stack_push_back(&stack,f1,f1->f[0]);
  if(f2->f[0] != NULL) stack_push_back(&stack,f2,f2->f[0]);

  /* check and swap  */
  while(stack.size != 0){
 
    stack_pop_back(&stack,&f1,&f0);

    if(dela2d_circle_in(pos[f0->v[0]],pos[f0->v[1]],
                        pos[f0->v[2]],pos[node])){
      /* swap */      
      v0 = node;
      v1 = f1->v[1];
      v2 = f0->v[face_index(f0->f,f1)];
      v3 = f1->v[2];

      n0 = f0->f[index_select(f0->v,v1)];
      n1 = f1->f[index_select(f1->v,v1)];
      n2 = f1->f[index_select(f1->v,v3)];
      n3 = f0->f[index_select(f0->v,v3)];

      f0->v[0] = v0;  f0->v[1] = v2;  f0->v[2] = v3;
      f0->f[0] = n0;  f0->f[1] = n1;  f0->f[2] = f1;

      f1->v[0] = v0;  f1->v[1] = v1;  f1->v[2] = v2;
      f1->f[0] = n3;  f1->f[1] = f0;  f1->f[2] = n2;

      if(n1 != NULL)  face_replace(n1->f,f1,f0);
      if(n3 != NULL)  face_replace(n3->f,f0,f1);

      if(n0 != NULL) stack_push_back(&stack,f0,n0);
      if(n3 != NULL) stack_push_back(&stack,f1,n3);
    }  
  }

  return 0;
}

int dela2d_write(char* fileName)
{

  TFACE* f;
  int i;

  FILE *fout = fopen(fileName,"w");
  if(fout == NULL)
    return -1;

  fprintf(fout,"%d\n",nPos + 4);

  for(i = 0; i < nPos + 4 ;i++){
    fprintf(fout,"%f %f %f\n",pos[i][0],pos[i][1],0.0);
  }
   
  fprintf(fout,"%d\n",nTri);

  f = top;
  while(f != NULL){
    fprintf(fout,"%d %d %d\n", f->v[0],f->v[1],f->v[2]);

    f = f->next;
  }

  fclose(fout);  

  return 0;
}

int  main(int argc,char** argv)
{
  int i; 
  char *infile;
  char *outfile;
  TFACE *f,*d; 

  if(argc != 3){
    fprintf (stderr, "Usage : \n");
    fprintf (stderr, "%s pchFileName(in) pchFileName(out) \n",argv[0]);
    exit (-1);
  }
   
  infile  = argv[1];
  outfile = argv[2];

  /* read file */
  if(dela2d_read(infile)){
    return -1;
  }
  
  /* insert node*/
  for(i = 4; i < nPos + 4;i++){
    if(dela2d_insert(i)){
       return -1;
    }
  }

  /* write file */
  if(dela2d_write(outfile)){
    return -1;
  }

  /* free memory */
  free(pos);
  f = top;
  while(f != NULL){
    d = f;
    f = f->next;
    free(d);
  }
  return 0;
}









