/**
 * \file ledblink.c
 * \brief アプリケーションの本体ファイル。
*/

/**
 * \mainpage
 * \details
 * LPCXpresso IDEによるTOPPERS/ASP for LPC サンプルプログラム
 *
 * このアプリケーションは、LPCXpresso用IDE上でTOPPERS/ASP for LPCのサンプルプログラムをビルドし、
 * フラッシュにロードする方法をデモするためのサンプル・アプリケーションである。
 *
 * ＊プログラムの概要
 *
 * プログラムを実行すると、はじめに#app_initialize()関数が呼び出され、LEDが接続されているピンをGPIO出力として
 * 初期化する。app_initialize()は、ledblink.cfgファイルで、静的APIによって初期化関数として登録されている。
 * 次に#main_task()関数がタスクとして起動される。main_task()は0.5秒休止し、休止後LED状態を反転する。これを
 * 繰り返す。main_task()関数は、ledblink.cfgファイルで静的APIによってタスクとして登録されている。
 *
 * ＊ビルド環境の構築
 *
 * ビルドはUbuntu 10.04上で行う。ビルドの前に以下の手順により開発環境を整備する。
 * \li Ubuntuをアップデートし、最新のセキュリティパッチを当てておく
 * \li SourceforgeのTOPPERS/ASP for LPC プロジェクトにしたがい、開発ツールをインストールする。 http://sourceforge.jp/projects/toppersasp4lpc/howto/install
 * \li code-techのサイトから、LPCXpressoのIDEをダウンロードし、インストールする。 http://lpcxpresso.code-red-tech.com/LPCXpresso/
 *
 * 上の順番でインストールすると、TOPPERS/ASP for LPCが提供しているインストーラがインストールしたcodesourceryは無視され、LPCXpresso IDEのarm-none-eabiツールチェーンが
 * 使用される。実害はないが気になるならばインストールスクリプトを編集して、codesourceryのインストールをスキップするとよい。
 *
 * ＊ビルドの手順
 *
 * 最初にソースツリー（このドキュメントを含む ledblink_lpcxpresso_1768以下のツリー）をLPCxpressoのIDEが作ったワークスペース下に展開する。
 *
 * 次に、ledblink_lpcxpresso_1768の中で、コマンドラインから .configure コマンドを実行する。このコマンドはASPカーネルのコンフィギュレータを
 * ビルドし、それを使ってledblinkアプリを構成し、ビルドする。
 *
 * いったんビルドが終わったら、LPCxpersso IDEにプロジェクトとして登録する。Menubar -> File -> New -> C Projectを実行する。
 * 現れたダイアログからMakefile project/Empty projectを選び、CodeRed MCU Toolを指定する。最後にプロジェクトディレクトリとして
 * 先に展開したディレクトリを指定する。これでLED点滅プロジェクトが登録される。
 *
 * プロジェクトを登録したら、Make Targetとして"asp.bin"を登録しておくとよい。デフォルトでASPカーネルのMakeファイルはbinファイルを生成しない。
 * フラッシュへの書き込みを行うにはbinファイルが必要である。
 *
 * コマンドラインから make asp.binを実行するか、先に登録したasp.binをターゲットとしてビルドする。
 *
 * ビルドが完了したら、IDEのツールバーから"Program Flash"ボタンをクリックし、ビルドしたasp.binをターゲットにプログラムする。完了したら
 * USBケーブルを抜き差ししてリセットし、動作を確認する。
 *
 * \date 2010/Oct/31
 * \author Suikan
 *
 */

#include <kernel.h>		/* カーネル API */
#include "kernel_cfg.h"	/* コンフィギュレータが作り出すインクルード・ファイル。無条件に読む */
#include "ledblink.h"	/* アプリケーション・インクルード・ファイル */
#include <LPC17xx.h>		/* CMSIS */

/**
 * \brief LPCXpresso のLEDピン番号 
 * \details
 * LPCXpresso 1768のLEDが接続されているピンの番号を宣言する。
 */
#define LED	(1<<22)

/**
 * \brief メインタスク
 * \param exinf コンフィギュレータから渡す引数。今回は利用しない
 * \details
 * 500ミリ秒休んでLEDを反転する。これを繰り返す。タスクからは戻らない。
 */
void main_task(intptr_t exinf)
{
	while(1)
	{
		tslp_tsk(500);		/* 500mS休む */
		LPC_GPIO0->FIOPIN ^= LED;	/* LED状態を反転する */
	}
}

/**
 * \brief 初期化ルーチン
 * \param exinf コンフィギュレータから渡す引数。今回は利用しない
 * \details
 * LED出力ポートの初期化を行う。LPCXpresso 1768のLEDは、ピンP0.22に
 * 接続されている。そこで、ITRONのイニシャライザ機能を使ってピンを初期化する。
 *
 * このデモに限っていえば、ピンの初期化は#main_task()でやればいい。しかし、
 * ここではITRONの機能説明のために初期化機能を使った。
 * 
 * この関数はledblink.cfgのなかで、ATT_INIにより、システムに初期化関数として
 * 登録される。
*/

void app_initialize(intptr_t exinf)
{

	/*
	 *  I/Oポートの初期化
	 */
	/* LEDはP0.22。1のときオン。*/
	LPC_GPIO0->FIOCLR = LED;		/* まず出力状態を「消灯」にしておく */
	LPC_GPIO0->FIODIR |= LED;		/* P0.22を出力にする */

}

