/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.util;

import java.lang.reflect.Array;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;

import jp.terasoluna.fw.dao.IllegalClassTypeException;
import jp.terasoluna.fw.util.PropertyUtil;

import org.apache.commons.logging.Log;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

/**
 * ob`p[eBeBB<br>
 * <br>
 * eob`ɂĎgp郆[eBeB\bh`B
 */
public class BatchUtil {

    /**
     * ėp񌋍\bhB
     * @param args Cӂ̒l
     * @return 
     */
    public static String cat(Object... args) {

        StringBuilder str = new StringBuilder();

        if (args == null) {
            return null;
        }

        for (Object o : args) {
            if (o != null) {
                str.append(o);
            }
        }

        return str.toString();
    }

    /**
     * CtHO̊JnbZ[W擾B
     * @param jobCd WuƖR[h
     * @return String bZ[W
     */
    public static String getInfoLogStartMsg(String jobCd) {

        return BatchUtil.cat("[", jobCd, "] ", "Jn");
    }

    /**
     * CtHȌIbZ[W擾B
     * @param jobCd WuƖR[h
     * @return String bZ[W
     */
    public static String getInfoLogEndMsg(String jobCd) {

        return BatchUtil.cat("[", jobCd, "] ", "I");
    }

    /**
     * ftHgTransactionDefinition擾
     * @return
     */
    public static TransactionDefinition getTransactionDefinition() {
        return new DefaultTransactionDefinition();
    }

    /**
     * ftHgTransactionDefinition擾
     * @param propagationBehavior gUNV`[h(@see TransactionDefinition) ftHgFTransactionDefinition.PROPAGATION_REQUIRED
     * @param isolationLevel gUNVx(@see TransactionDefinition) ftHgFTransactionDefinition.ISOLATION_DEFAULT
     * @param timeout gUNV^CAEg(b) ftHgFTransactionDefinition.TIMEOUT_DEFAULT (^CAEgȂ)
     * @param readOnly [hI[gUNV ftHgFfalse
     * @return
     */
    public static TransactionDefinition getTransactionDefinition(
            int propagationBehavior, int isolationLevel, int timeout,
            boolean readOnly) {
        DefaultTransactionDefinition td = new DefaultTransactionDefinition();
        td.setPropagationBehavior(propagationBehavior);
        td.setIsolationLevel(isolationLevel);
        td.setTimeout(timeout);
        td.setReadOnly(readOnly);
        return td;
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran) {
        return startTransaction(tran, getTransactionDefinition(), null);
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @param log Log
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran, Log log) {
        return startTransaction(tran, getTransactionDefinition(), log);
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @param propagationBehavior gUNV`[h(@see TransactionDefinition) ftHgFTransactionDefinition.PROPAGATION_REQUIRED
     * @param isolationLevel gUNVx(@see TransactionDefinition) ftHgFTransactionDefinition.ISOLATION_DEFAULT
     * @param timeout gUNV^CAEg(b) ftHgFTransactionDefinition.TIMEOUT_DEFAULT (^CAEgȂ)
     * @param readOnly [hI[gUNV ftHgFfalse
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran, int propagationBehavior,
            int isolationLevel, int timeout, boolean readOnly) {
        return startTransaction(tran, getTransactionDefinition(
                propagationBehavior, isolationLevel, timeout, readOnly), null);
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @param propagationBehavior gUNV`[h(@see TransactionDefinition) ftHgFTransactionDefinition.PROPAGATION_REQUIRED
     * @param isolationLevel gUNVx(@see TransactionDefinition) ftHgFTransactionDefinition.ISOLATION_DEFAULT
     * @param timeout gUNV^CAEg(b) ftHgFTransactionDefinition.TIMEOUT_DEFAULT (^CAEgȂ)
     * @param readOnly [hI[gUNV ftHgFfalse
     * @param log Log
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran, int propagationBehavior,
            int isolationLevel, int timeout, boolean readOnly, Log log) {
        return startTransaction(tran, getTransactionDefinition(
                propagationBehavior, isolationLevel, timeout, readOnly), log);
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @param def TransactionDefinition
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran, TransactionDefinition def) {
        return startTransaction(tran, def, null);
    }

    /**
     * gUNVJn
     * @param tran PlatformTransactionManager
     * @param def TransactionDefinition
     * @param log Log
     * @return TransactionStatus
     */
    public static TransactionStatus startTransaction(
            PlatformTransactionManager tran, TransactionDefinition def, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("startTransaction:", tran));
            if (def != null) {
                log.debug(BatchUtil.cat("TransactionDefinition:(propagation:[",
                        def.getPropagationBehavior(), "] isolation:[", def
                                .getIsolationLevel(), "] timeout:[", def
                                .getTimeout(), "] readonly:[",
                        def.isReadOnly(), "] name:[", def.getName(), "])"));
            }
        }

        TransactionStatus stat = null;
        if (tran != null) {
            stat = tran.getTransaction(def);
        }

        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(start):" + stat);
        }
        return stat;
    }

    /**
     * gUNVJn
     * @param tranDef TransactionDefinition
     * @param tranMap PlatformTransactionManager}bv
     * @return TransactionStatus}bv
     */
    public static Map<String, TransactionStatus> startTransactions(
            TransactionDefinition tranDef, Map<?, ?> tranMap) {
        return startTransactions(tranDef, tranMap, null);
    }

    /**
     * gUNVJn
     * @param tranDef TransactionDefinition
     * @param tranMap PlatformTransactionManager}bv
     * @param log Log
     * @return TransactionStatus}bv
     */
    public static Map<String, TransactionStatus> startTransactions(
            TransactionDefinition tranDef, Map<?, ?> tranMap, Log log) {
        int count = 0;
        Map<String, TransactionStatus> statMap = new HashMap<String, TransactionStatus>();

        Set<?> entrySet = tranMap.entrySet();

        if (entrySet != null) {
            for (Iterator<?> entIt = entrySet.iterator(); entIt.hasNext();) {
                Object entObj = (Object) entIt.next();
                String key = null;
                PlatformTransactionManager ptm = null;

                // }bvGg[
                if (entObj instanceof Map.Entry) {
                    Map.Entry<?, ?> ent = (Entry<?, ?>) entObj;
                    if (ent != null) {
                        // L[o
                        if (ent.getKey() instanceof String) {
                            key = (String) ent.getKey();
                        }
                        // gUNV}l[Wo
                        if (ent.getValue() instanceof PlatformTransactionManager) {
                            ptm = (PlatformTransactionManager) ent.getValue();
                        }
                    }
                }

                if (ptm != null) {

                    if (log != null && log.isDebugEnabled()) {
                        log.debug(BatchUtil.cat("startTransaction:", count));
                        if (tranDef != null) {
                            log.debug(BatchUtil.cat(
                                    "TransactionDefinition:(propagation:[",
                                    tranDef.getPropagationBehavior(),
                                    "] isolation:[", tranDef
                                            .getIsolationLevel(),
                                    "] timeout:[", tranDef.getTimeout(),
                                    "] readonly:[", tranDef.isReadOnly(),
                                    "] name:[", tranDef.getName(), "])"));
                        }
                    }

                    // gUNVJn
                    TransactionStatus trnStat = ptm.getTransaction(tranDef);

                    // gUNVXe[^Xi[
                    if (statMap != null) {
                        statMap.put(key, trnStat);
                    }

                    if (log != null && log.isDebugEnabled()) {
                        log.debug(BatchUtil.cat("CurrentConnection(start):",
                                trnStat, ",", count));
                        count++;
                    }
                }
            }
        }

        return statMap;
    }

    /**
     * gUNVR~bg RlNṼR~bgs
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     */
    public static void commitTransaction(PlatformTransactionManager tran,
            TransactionStatus stat) {
        commitTransaction(tran, stat, null);
    }

    /**
     * gUNVR~bg RlNṼR~bgs
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param log Log
     */
    public static void commitTransaction(PlatformTransactionManager tran,
            TransactionStatus stat, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("commitTransaction:", stat));
        }

        if (tran != null && stat != null) {
            tran.commit(stat);
        }
        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(commit):" + stat);
        }
    }

    /**
     * gUNVR~bg
     * @param sqlMapperList
     * @throws SQLException
     */
    public static void commitTransactions(Map<?, ?> tranMap,
            Map<String, TransactionStatus> statMap) {
        commitTransactions(tranMap, statMap, null);
    }

    /**
     * gUNVR~bg
     * @param sqlMapperList
     * @throws SQLException
     */
    public static void commitTransactions(Map<?, ?> tranMap,
            Map<String, TransactionStatus> statMap, Log log) {
        int count = 0;

        Set<?> entrySet = tranMap.entrySet();

        if (entrySet != null) {
            for (Iterator<?> entIt = entrySet.iterator(); entIt.hasNext();) {
                Object entObj = (Object) entIt.next();
                String key = null;
                PlatformTransactionManager ptm = null;

                // }bvGg[
                if (entObj instanceof Map.Entry) {
                    Map.Entry<?, ?> ent = (Entry<?, ?>) entObj;
                    if (ent != null) {
                        // L[o
                        if (ent.getKey() instanceof String) {
                            key = (String) ent.getKey();
                        }
                        // gUNV}l[Wo
                        if (ent.getValue() instanceof PlatformTransactionManager) {
                            ptm = (PlatformTransactionManager) ent.getValue();
                        }
                    }
                }

                if (ptm != null) {
                    TransactionStatus trnStat = null;

                    // gUNVXe[^X擾
                    if (statMap != null) {
                        trnStat = statMap.get(key);
                    }

                    if (trnStat != null) {

                        if (log != null && log.isDebugEnabled()) {
                            log.debug(BatchUtil.cat("commitTransaction:",
                                    count, ",", trnStat));
                            log.debug("CurrentConnection(commit):" + trnStat);
                            count++;
                        }

                        // R~bg
                        ptm.commit(trnStat);
                    }
                }
            }
        }
    }

    /**
     * gUNVIiR~bg[obNj
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     */
    public static void endTransaction(PlatformTransactionManager tran,
            TransactionStatus stat) {
        endTransaction(tran, stat, null);
    }

    /**
     * gUNVIiR~bg[obNj
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param log Log
     */
    public static void endTransaction(PlatformTransactionManager tran,
            TransactionStatus stat, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("endTransaction:", stat));
        }

        if (tran != null && stat != null && !stat.isCompleted()) {
            tran.rollback(stat);
        }

        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(release):" + stat);
        }
    }

    /**
     * gUNVIiR~bg[obNj
     * @param tranMap PlatformTransactionManager}bv
     * @param statMap TransactionStatus}bv
     * @return ŗ^ꂽPlatformTransactionManagerSĐɏIłꍇtrueԋp
     */
    public static boolean endTransactions(Map<?, ?> tranMap,
            Map<String, TransactionStatus> statMap) {
        return endTransactions(tranMap, statMap, null);
    }

    /**
     * gUNVIiR~bg[obNj
     * @param tranMap PlatformTransactionManager}bv
     * @param statMap TransactionStatus}bv
     * @param log Log
     * @return ŗ^ꂽPlatformTransactionManagerSĐɏIłꍇtrueԋp
     */
    public static boolean endTransactions(Map<?, ?> tranMap,
            Map<String, TransactionStatus> statMap, Log log) {
        boolean isNormal = true;
        int count = 0;

        Set<?> entrySet = tranMap.entrySet();

        if (entrySet != null) {
            for (Iterator<?> entIt = entrySet.iterator(); entIt.hasNext();) {
                Object entObj = (Object) entIt.next();
                String key = null;
                PlatformTransactionManager ptm = null;

                // }bvGg[
                if (entObj instanceof Map.Entry) {
                    Map.Entry<?, ?> ent = (Entry<?, ?>) entObj;
                    if (ent != null) {
                        // L[o
                        if (ent.getKey() instanceof String) {
                            key = (String) ent.getKey();
                        }
                        // gUNV}l[Wo
                        if (ent.getValue() instanceof PlatformTransactionManager) {
                            ptm = (PlatformTransactionManager) ent.getValue();
                        }
                    }
                }

                if (ptm != null) {
                    TransactionStatus trnStat = null;

                    // gUNVXe[^X擾
                    if (statMap != null) {
                        trnStat = statMap.get(key);
                    }

                    // gUNVIigUNV̏ꍇ̓[obNj
                    if (trnStat != null && !trnStat.isCompleted()) {

                        if (log != null && log.isDebugEnabled()) {
                            log.debug(BatchUtil.cat("endTransaction:", count,
                                    ",", trnStat));
                            log.debug("CurrentConnection(end):" + trnStat);
                        }

                        // [obN
                        try {
                            ptm.rollback(trnStat);
                        } catch (TransactionException e) {
                            if (log != null && log.isErrorEnabled()) {
                                log.error("CurrentConnection(end):"
                                        + Integer.toString(count), e);
                            }
                            isNormal = false;
                            // OĂrIÃgUNVI݂
                        }

                        if (log != null && log.isDebugEnabled()) {
                            log.debug("CurrentConnection(release):" + trnStat);
                        }
                        count++;

                    }
                }

            }
        }
        return isNormal;
    }

    /**
     * Z[u|Cgݒ肷
     * @param stat TransactionStatus
     * @return Object Z[u|Cg
     */
    public static Object setSavepoint(TransactionStatus stat) {
        return setSavepoint(stat, null);
    }

    /**
     * Z[u|Cgݒ肷
     * @param stat TransactionStatus
     * @param log Log
     * @return Object Z[u|Cg
     */
    public static Object setSavepoint(TransactionStatus stat, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("setSavepoint:", stat));
        }

        Object savepoint = stat.createSavepoint();

        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(setSavepoint):" + stat);
        }

        return savepoint;
    }

    /**
     * Z[u|Cg[X
     * @param stat TransactionStatus
     * @param savepoint Z[u|Cg
     */
    public static void releaseSavepoint(TransactionStatus stat, Object savepoint) {
        releaseSavepoint(stat, savepoint, null);
    }

    /**
     * Z[u|Cg[X
     * @param stat TransactionStatus
     * @param savepoint Z[u|Cg
     * @param log Log
     */
    public static void releaseSavepoint(TransactionStatus stat,
            Object savepoint, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("releaseSavepoint:", stat));
        }

        stat.releaseSavepoint(savepoint);

        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(releaseSavepoint):" + savepoint);
        }
    }

    /**
     * Z[u|Cg܂Ń[obN
     * @param stat TransactionStatus
     * @param savepoint Z[u|Cg
     */
    public static void rollbackSavepoint(TransactionStatus stat,
            Object savepoint) {
        rollbackSavepoint(stat, savepoint, null);
    }

    /**
     * Z[u|Cg܂Ń[obN
     * @param stat TransactionStatus
     * @param savepoint Z[u|Cg
     * @param log Log
     */
    public static void rollbackSavepoint(TransactionStatus stat,
            Object savepoint, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("rollbackSavepoint:", stat));
        }

        stat.rollbackToSavepoint(savepoint);

        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(rollback(savepoint)):" + stat);
        }
    }

    /**
     * gUNVJn܂Ń[obNB
     * @param tran gUNV}l[W
     * @param stat TransactionStatus
     */
    public static void rollbackTransaction(PlatformTransactionManager tran,
            TransactionStatus stat) {
        rollbackTransaction(tran, stat, null);
    }

    /**
     * gUNVJn܂Ń[obNB
     * @param tran gUNV}l[W
     * @param stat TransactionStatus
     * @param log Log
     */
    public static void rollbackTransaction(PlatformTransactionManager tran,
            TransactionStatus stat, Log log) {
        if (log != null && log.isDebugEnabled()) {
            log.debug(BatchUtil.cat("rollback:", stat));
        }
        if (tran != null && stat != null && !stat.isCompleted()) {
            tran.rollback(stat);
        }
        if (log != null && log.isDebugEnabled()) {
            log.debug("CurrentConnection(rollback):" + stat);
        }

    }

    /**
     * gUNVR~bgAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @return
     */
    public static TransactionStatus commitRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat) {
        commitTransaction(tran, stat, null);
        endTransaction(tran, stat, null);
        return startTransaction(tran);
    }

    /**
     * gUNVR~bgAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param log Log
     */
    public static TransactionStatus commitRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat, Log log) {
        commitTransaction(tran, stat, log);
        endTransaction(tran, stat, log);
        return startTransaction(tran, log);
    }

    /**
     * gUNVR~bgAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param def TransactionDefinition
     */
    public static TransactionStatus commitRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat,
            TransactionDefinition def) {
        commitTransaction(tran, stat, null);
        endTransaction(tran, stat, null);
        return startTransaction(tran, def);
    }

    /**
     * gUNVR~bgAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param def TransactionDefinition
     * @param log Log
     */
    public static TransactionStatus commitRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat,
            TransactionDefinition def, Log log) {
        commitTransaction(tran, stat, log);
        endTransaction(tran, stat, log);
        return startTransaction(tran, def, log);
    }

    /**
     * gUNV[obNAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     */
    public static TransactionStatus rollbackRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat) {
        rollbackTransaction(tran, stat, null);
        endTransaction(tran, stat, null);
        return startTransaction(tran);
    }

    /**
     * gUNV[obNAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param log Log
     */
    public static TransactionStatus rollbackRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat, Log log) {
        rollbackTransaction(tran, stat, log);
        endTransaction(tran, stat, log);
        return startTransaction(tran, log);
    }

    /**
     * gUNV[obNAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param def TransactionDefinition
     */
    public static TransactionStatus rollbackRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat,
            TransactionDefinition def) {
        rollbackTransaction(tran, stat, null);
        endTransaction(tran, stat, null);
        return startTransaction(tran, def);
    }

    /**
     * gUNV[obNAgUNVēxJn
     * @param tran PlatformTransactionManager
     * @param stat TransactionStatus
     * @param def TransactionDefinition
     * @param log Log
     */
    public static TransactionStatus rollbackRestartTransaction(
            PlatformTransactionManager tran, TransactionStatus stat,
            TransactionDefinition def, Log log) {
        rollbackTransaction(tran, stat, log);
        endTransaction(tran, stat, log);
        return startTransaction(tran, def, log);
    }

    /**
     * Listz^ɕϊ List̒ɕ̌^Ăꍇ͎gpłȂ
     * @param <E> ԋpľ^
     * @param list ̓f[^
     * @param clazz ԋpľ^킷Class^̃CX^X
     * @return List̒gzɂ
     */
    @SuppressWarnings("unchecked")
    public static <E> E[] changeListToArray(List<E> list, Class clazz) {

        if (clazz == null) {
            throw new IllegalClassTypeException();
        }

        // SQL̎sFl̎擾
        List<E> castedList = list;

        // zɕϊ
        E[] retArray = (E[]) Array.newInstance(clazz, castedList.size());
        try {
            castedList.toArray(retArray);
        } catch (ArrayStoreException e) {
            throw new IllegalClassTypeException(e);
        }

        return retArray;
    }

    /**
     * .propertiest@CO[vL[wŒlo O[vL[ɍvL[ɑ΂ď\[gsĂ ԋpXg֒lZbgĂ
     * @param propertyName .propertiest@C̖Oi.properties͕KvȂj
     * @param grpKey O[vL[
     * @return propertyNameɑ݂grpKeyPrefix
     */
    public static List<String> getProperties(String propertyName, String grpKey) {

        Properties properties = PropertyUtil.loadProperties(propertyName);
        Enumeration<String> propNames = PropertyUtil.getPropertyNames(
                properties, grpKey);

        List<String> propNamesList = Collections.list(propNames);
        Collections.sort(propNamesList);

        List<String> resultList = new ArrayList<String>();

        for (String key : propNamesList) {
            resultList.add(PropertyUtil.getProperty(key));
        }

        return resultList;
    }
}
