/*
 * Copyright (c) 2012 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.ex.web.struts.action;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.PageContext;

import org.apache.struts.Globals;
import org.apache.struts.action.ActionErrors;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.ActionMessages;
import org.apache.struts.config.ExceptionConfig;
import org.apache.struts.util.MessageResources;
import org.easymock.classextension.EasyMock;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class AbstractExceptionHandlerTest {

    private AbstractExceptionHandler abstractExceptionHandler;

    @Before
    public void setUp() throws Exception {
        abstractExceptionHandler = new AbstractExceptionHandler() {

            @Override
            protected void setModule(ActionForward forward, Exception ex,
                    ExceptionConfig ae, ActionMapping mapping) {
                // N/A
            }

            @Override
            protected ActionMessages getErrors(Exception ex,
                    ExceptionConfig ae, MessageResources resources) {
                // N/A
                return null;
            }
        };
    }

    @After
    public void tearDown() throws Exception {
        abstractExceptionHandler = null;
    }

    @Test
    public final void testGetActionForwardFromExceptionConfig() {
        String expectedPath = "aaa";
        ExceptionConfig exceptionConfigMock = new ExceptionConfig();
        exceptionConfigMock.setPath(expectedPath);
        ActionForward actionForward = abstractExceptionHandler
                .getActionForward(null, exceptionConfigMock, null, null);
        assertNotNull(actionForward);
        assertEquals(expectedPath, actionForward.getPath());
    }

    @Test
    public final void testGetActionForwardFromActionMappingInput() {
        final String expectedPath = "bbb";
        ExceptionConfig exceptionConfigMock = new ExceptionConfig();
        ActionMapping actionMappingMock = new ActionMapping() {
            private static final long serialVersionUID = -2461923715092926822L;

            @Override
            public ActionForward getInputForward() {
                return new ActionForward(expectedPath);
            }
        };
        actionMappingMock.setInput(expectedPath);
        ActionForward actionForward = abstractExceptionHandler
                .getActionForward(null, exceptionConfigMock, actionMappingMock,
                        null);
        assertNotNull(actionForward);
        assertEquals(expectedPath, actionForward.getPath());
    }

    @Test
    public final void testGetNullActionForward() {
        ExceptionConfig exceptionConfigMock = new ExceptionConfig();
        ActionMapping actionMappingMock = new ActionMapping() {
            private static final long serialVersionUID = 8228892313815527522L;

            @Override
            public ActionForward getInputForward() {
                return null;
            }
        };
        ActionForward actionForward = abstractExceptionHandler
                .getActionForward(null, exceptionConfigMock, actionMappingMock,
                        null);
        assertNull(actionForward);
    }

    @Test
    public final void testGetBundleKeyFromExceptionConfig() {
        String expectedBundleKey = "bundle";
        ExceptionConfig exceptionConfigMock = new ExceptionConfig();
        exceptionConfigMock.setBundle(expectedBundleKey);

        String actualBundleKey = abstractExceptionHandler
                .getBundleKey(exceptionConfigMock);
        assertEquals(expectedBundleKey, actualBundleKey);
    }

    @Test
    public final void testGetBundleKyeFromGlobalsMESSAGES_KEY() {
        ExceptionConfig exceptionConfigMock = new ExceptionConfig();
        String actualBundleKey = abstractExceptionHandler
                .getBundleKey(exceptionConfigMock);
        assertEquals(Globals.MESSAGES_KEY, actualBundleKey);
    }

    @Test
    public final void testGetMessageResourcesFromRequestAttribute() {
        String bundleKey = "bundleKey";
        HttpServletRequest httpServletRequestMock = EasyMock
                .createMock(HttpServletRequest.class);
        MessageResources messageResourcesMock = EasyMock
                .createMock(MessageResources.class);
        EasyMock.expect(httpServletRequestMock.getAttribute(bundleKey))
                .andReturn(messageResourcesMock);

        EasyMock.replay(httpServletRequestMock, messageResourcesMock);

        MessageResources actualMessageResources = abstractExceptionHandler
                .getMessageResources(httpServletRequestMock, bundleKey);
        assertEquals(messageResourcesMock, actualMessageResources);

        EasyMock.verify(httpServletRequestMock, messageResourcesMock);
    }

    @Test
    public final void testGetMessageResourcesFromServletContext() {
        String bundleKey = "bundleKey";
        HttpServletRequest httpServletRequestMock = EasyMock
                .createMock(HttpServletRequest.class);
        EasyMock.expect(httpServletRequestMock.getAttribute(bundleKey))
                .andReturn(null);
        HttpSession httpSessionMock = EasyMock.createMock(HttpSession.class);
        EasyMock.expect(httpServletRequestMock.getSession(true)).andReturn(
                httpSessionMock);
        ServletContext servletContextMock = EasyMock
                .createMock(ServletContext.class);
        EasyMock.expect(httpSessionMock.getServletContext()).andReturn(
                servletContextMock);
        MessageResources messageResourcesMock = EasyMock
                .createMock(MessageResources.class);
        EasyMock.expect(servletContextMock.getAttribute(bundleKey)).andReturn(
                messageResourcesMock);

        EasyMock.replay(httpServletRequestMock, httpSessionMock,
                servletContextMock, messageResourcesMock);

        MessageResources actualMessageResources = abstractExceptionHandler
                .getMessageResources(httpServletRequestMock, bundleKey);

        assertEquals(messageResourcesMock, actualMessageResources);
        EasyMock.verify(httpServletRequestMock, httpSessionMock,
                servletContextMock, messageResourcesMock);
    }

    @Test
    public final void testStoreExceptionToRequestScope() {
        HttpServletRequest httpServletRequestMock = EasyMock
                .createMock(HttpServletRequest.class);
        ActionErrors errors = new ActionErrors();
        String scope = "request";
        httpServletRequestMock.setAttribute(Globals.ERROR_KEY, errors);
        EasyMock.expectLastCall().times(1);

        EasyMock.replay(httpServletRequestMock);

        abstractExceptionHandler.storeException(httpServletRequestMock, errors,
                null, scope);

        EasyMock.verify(httpServletRequestMock);
    }

    @Test
    public final void testStoreExceptionToSessionScope() {
        HttpServletRequest httpServletRequestMock = EasyMock
                .createMock(HttpServletRequest.class);
        HttpSession httpSessionMock = EasyMock.createMock(HttpSession.class);
        EasyMock.expect(httpServletRequestMock.getSession()).andReturn(
                httpSessionMock);
        ActionErrors errors = new ActionErrors();
        httpSessionMock.setAttribute(Globals.ERROR_KEY, errors);
        EasyMock.expectLastCall().times(1);
        String scope = "session";

        EasyMock.replay(httpServletRequestMock, httpSessionMock);

        abstractExceptionHandler.storeException(httpServletRequestMock, errors,
                null, scope);

        EasyMock.verify(httpServletRequestMock, httpSessionMock);
    }

    @Test
    public final void testSetRequestAttribute() {
        HttpServletRequest httpServletRequestMock = EasyMock
                .createMock(HttpServletRequest.class);
        Exception exception = new Exception();
        httpServletRequestMock.setAttribute(Globals.EXCEPTION_KEY, exception);
        EasyMock.expectLastCall().times(1);
        httpServletRequestMock.setAttribute(PageContext.EXCEPTION, exception);
        EasyMock.expectLastCall().times(1);

        EasyMock.replay(httpServletRequestMock);

        abstractExceptionHandler.setRequestAttribute(httpServletRequestMock,
                exception, null, null);

        EasyMock.verify(httpServletRequestMock);
    }
}
