/*
 * Copyright (c) 2012 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.ex.aop.log;

import java.util.List;

import jp.terasoluna.fw.service.thin.BLogicResult;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * BLogicクラス処理実行前後のデバッグログを出力するためのインタセプタ
 * 
 * @version 1.0.0
 */
public class BLogicLogInterceptor implements MethodInterceptor {

    /**
     * ログインスタンス
     */
    private static Log logger = LogFactory.getLog(BLogicLogInterceptor.class);

    /**
     * エラーログとして出力しない例外クラスリスト。 
     * 例外クラスはクラスの完全修飾クラス名を設定する
     */
    private List<Class> noErrorLogExceptionLists = null;

    /**
     * エラーログとして出力しない例外クラスリストを設定する
     * 
     * @param noErrorLogExceptionLists 例外クラスリスト
     */
    public void setNoErrorLogExceptionLists(
            List<Class> noErrorLogExceptionLists) {
        this.noErrorLogExceptionLists = noErrorLogExceptionLists;
    }

    /**
     * BLogicクラスの各メソッドの実行前後に、デバッグログを出力します。 
     * また、メソッド実行中に例外が発生した場合、エラーログを出力します。
     * 
     * @param invocation インタセプタ対象のオブジェクト
     * @return インタセプタ適用後のオブジェクト
     * @throws Throwable 実行例外
     * 
     * @see org.aopalliance.intercept.MethodInterceptor#
     *       invoke(org.aopalliance.intercept.MethodInvocation)
     */
    public Object invoke(MethodInvocation invocation) throws Throwable {
        Class<? extends Object> blogicClass = null;
        if (logger.isDebugEnabled()) {
            blogicClass = invocation.getThis().getClass();
            logger.debug(blogicClass.getSimpleName().concat(
                    " is being executed..."));
            logger.debug(createParamsInfo(invocation.getArguments()));
        }
        Object result = null;
        try {
            result = invocation.proceed();
        } catch (Throwable e) {
            if (checkException(e)) {
                logger.error(e);
                throw e;
            } else {
                logger.debug(e);
                throw e;
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug(createResultInfo(result));
            logger.debug(blogicClass.getSimpleName().concat(
                    " was executed completely."));
        }
        return result;
    }

    /**
     * メソッドパラメータの文字列表現を返す。
     * 
     * @param args メソッドパラメータ
     * @return パラメータの文字列表現
     */
    private String createParamsInfo(Object[] args) {
        StringBuilder info = new StringBuilder("Params:");
        if (args == null) {
            info.append("null");
        } else {
            for (int i = 0; i < args.length; i++) {
                if (args[i] == null) {
                    info.append("null");
                } else {
                    info.append(args[i].toString());
                }
                if (i != args.length - 1) {
                    info.append(",");
                }
            }
        }
        return info.toString();
    }

    /**
     * 戻り値の文字列表現を返す。
     * 
     * @param result 　メソッドの戻り値
     * @return メソッド戻り値の文字列表現
     */
    private String createResultInfo(Object result) {
        StringBuilder info = new StringBuilder("Result:");
        if (result == null) {
            info.append("null");
        } else if (result instanceof BLogicResult) {
            BLogicResult blogicResult = (BLogicResult) result;
            info.append("resultString=").append(blogicResult.getResultString());
            info.append(",resultObject=");
            if (blogicResult.getResultObject() == null) {
                info.append("null");
            } else {
                info.append(blogicResult.getResultObject().toString());
            }
        } else {
            info.append(result.toString());
        }
        return info.toString();
    }

    /**
     * ログ出力レベルの判定を行う。<br>
     * ERRORレベルでログを出力する例外の場合はtrue<br>
     * DEGUGレベルでログを出力する例外の場合はfalse<br>
     * を返す。<br>
     * <br>
     * プロパティnoErrorLogExceptionListsに設定された例外、
     * もしくはそのサブクラスの例外であればfalseを返す。<br>
     * それ以外はtrueを返す。
     * 
     * @param target チェック対象となる例外
     * @return ログ出力レベル判定値
     */
    protected boolean checkException(Throwable target) {
        if ( noErrorLogExceptionLists != null ) {
            for (Class noCheckException : noErrorLogExceptionLists) {
    
                if (noCheckException.isAssignableFrom(target.getClass())) {
                    return false;
                }
            }
        }
        return true;
    }
}
