/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.executor;

import jp.terasoluna.fw.batch.constants.EventConstants;
import jp.terasoluna.fw.batch.constants.JobStatusConstants;
import jp.terasoluna.fw.batch.executor.vo.BLogicResult;
import jp.terasoluna.fw.batch.executor.vo.BatchJobData;
import jp.terasoluna.fw.batch.util.JobUtil;
import jp.terasoluna.fw.dao.QueryDAO;
import jp.terasoluna.fw.dao.UpdateDAO;
import jp.terasoluna.fw.util.PropertyUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

/**
 * 񓯊ob`GO[L[^ۃNXB<br>
 * <br>
 * 񓯊WuNp̃ob`GO[L[^B
 * @see jp.terasoluna.fw.batch.executor.BatchExecutor
 * @see jp.terasoluna.fw.batch.executor.AbstractBatchExecutor
 * @see jp.terasoluna.fw.batch.executor.AsyncBatchExecutor
 */
public abstract class AbstractJobBatchExecutor extends AbstractBatchExecutor {

    /**
     * O.
     */
    private static Log log = LogFactory.getLog(AbstractJobBatchExecutor.class);

    /**
     * Wu̎sԊui~bj擾pL[.
     */
    protected static final String JOB_INTERVAL_TIME = "polling.interval";

    /**
     * Executoȑ풓[h̏ItOĎt@CitpXŋLqj擾pL[.
     */
    protected static final String EXECUTOR_END_MONITORING_FILE = "executor.endMonitoringFile";

    /**
     * Executor̃WuI҂`FbNԊui~bj擾pL[.
     */
    protected static final String EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME = "executor.jobTerminateWaitInterval";

    /**
     * Wu̎sԊui~bj̃ftHgl
     */
    protected static final long DEFAULT_JOB_INTERVAL_TIME = 1000;

    /**
     * WusGCsitrue/falsej̃ftHgl
     */
    protected static final boolean DEFAULT_JOB_AFTER_GC = true;

    /**
     * Executoȑ풓[h̃WuXg擾Ԋui~bj̃ftHgl
     */
    protected static final long DEFAULT_EXECUTOR_LOOP_INTERVAL_TIME = 1000;

    /**
     * Executoȑ풓[h̏ItOĎt@CitpXŋLqj̃ftHgl
     */
    protected static final String DEFAULT_EXECUTOR_END_MONITORING_FILE = null;

    /**
     * Executor̃XbhL[TCY`FbNԊui~bj̃ftHgl
     */
    protected static final long DEFAULT_EXECUTOR_QUEUE_CHECK_INTERVAL_TIME = 1000;

    /**
     * Executor̃WuI҂`FbNԊui~bj̃ftHgl
     */
    protected static final long DEFAULT_EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME = 5000;

    /**
     * Wu̎sԊui~bj
     */
    protected long jobIntervalTime = DEFAULT_JOB_INTERVAL_TIME;

    /**
     * Executoȑ풓[h̏ItOĎt@CitpXŋLqj
     */
    protected String executorEndMonitoringFile = DEFAULT_EXECUTOR_END_MONITORING_FILE;

    /**
     * Executor̃WuI҂`FbNԊui~bj
     */
    protected long executorJobTerminateWaitIntervalTime = DEFAULT_EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME;

    /**
     * JñXe[^XύXsǂ
     */
    protected boolean changeStartStatus = false;

    /**
     * RXgN^
     */
    protected AbstractJobBatchExecutor() {
        super();
        initParameter();
    }

    /*
     * (non-Javadoc)
     * @seejp.terasoluna.fw.batch.light.executor.AbstractBatchExecutor# initDefaultAppContext()
     */
    @Override
    protected void initDefaultAppContext() {
        // VXeAvP[VReLXg擾
        String defaultAppContextName = getDefaultBeanFileName();
        if (defaultAppContextName == null || "".equals(defaultAppContextName)) {
            if (log.isErrorEnabled()) {
                log.error("Bean definition default file name is undefined.");
            }
            return;
        }
        // f[^\[XAvP[VReLXg擾
        String dataSourceAppContextName = getDataSourceBeanFileName();
        if (dataSourceAppContextName == null
                || "".equals(dataSourceAppContextName)) {
            if (log.isErrorEnabled()) {
                log.error("Bean definition default file name is undefined.");
            }
            return;
        }

        defaultApplicationContext = getApplicationContext(
                defaultAppContextName, dataSourceAppContextName);
        if (defaultApplicationContext == null) {
            StringBuilder sb = new StringBuilder();
            sb
                    .append("Default application context not found. defaultAppContextName:[");
            sb.append(defaultAppContextName);
            sb.append("],dataSourceAppContextName:[");
            sb.append(dataSourceAppContextName);
            sb.append("]");
            if (log.isErrorEnabled()) {
                log.error(sb.toString());
            }
            return;
        }
    }

    /*
     * (non-Javadoc)
     * @seejp.terasoluna.fw.batch.light.executor.AbstractBatchExecutor# initSystemDatasourceDao()
     */
    @Override
    protected void initSystemDatasourceDao() {
        if (defaultApplicationContext == null) {
            return;
        }

        String queryDaoKey = PropertyUtil
                .getProperty(SYSTEM_DATASOURCE_QUERY_DAO);
        String updateDaoKey = PropertyUtil
                .getProperty(SYSTEM_DATASOURCE_UPDATE_DAO);
        String transactionManagerKey = PropertyUtil
                .getProperty(SYSTEM_DATASOURCE_TRANSACTION_MANAGER);

        // QueryDAO擾
        if (queryDaoKey != null && queryDaoKey.length() != 0) {
            if (defaultApplicationContext.containsBean(queryDaoKey)) {
                try {
                    sysQueryDAO = (QueryDAO) defaultApplicationContext.getBean(
                            queryDaoKey, QueryDAO.class);
                } catch (Throwable e) {
                    if (log.isErrorEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("QueryDAO bean not found. beanName:[");
                        sb.append(queryDaoKey);
                        sb.append("]");
                        log.error(sb.toString(), e);
                    }
                }
            }
        }

        // UpdateDAO擾
        if (updateDaoKey != null && updateDaoKey.length() != 0) {
            if (defaultApplicationContext.containsBean(updateDaoKey)) {
                try {
                    sysUpdateDAO = (UpdateDAO) defaultApplicationContext
                            .getBean(updateDaoKey, UpdateDAO.class);
                } catch (Throwable e) {
                    if (log.isErrorEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("UpdateDAO bean not found. beanName:[");
                        sb.append(updateDaoKey);
                        sb.append("]");
                        log.error(sb.toString(), e);
                    }
                }
            }
        }

        // transactionManager擾
        if (transactionManagerKey != null
                && transactionManagerKey.length() != 0) {
            if (defaultApplicationContext.containsBean(transactionManagerKey)) {
                try {
                    sysTransactionManager = (PlatformTransactionManager) defaultApplicationContext
                            .getBean(transactionManagerKey,
                                    PlatformTransactionManager.class);
                } catch (Throwable e) {
                    if (log.isErrorEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("TransactionManager bean not found.");
                        sb.append(" beanName:[");
                        sb.append(transactionManagerKey);
                        sb.append("]");
                        log.error(sb.toString(), e);
                    }
                }
            }
        }

        if (sysQueryDAO == null) {
            if (log.isErrorEnabled()) {
                log.error("queryDAO is null.");
            }
        }
        if (sysUpdateDAO == null) {
            if (log.isErrorEnabled()) {
                log.error("updateDAO is null.");
            }
        }
        if (sysTransactionManager == null) {
            if (log.isErrorEnabled()) {
                log.error("transactionManager is null.");
            }
        }
    }

    /**
     * 
     */
    protected void initParameter() {
        // Wu̎sԊui~bj
        String jobIntervalTimeStr = PropertyUtil.getProperty(JOB_INTERVAL_TIME);
        if (jobIntervalTimeStr != null && jobIntervalTimeStr.length() != 0) {
            try {
                this.jobIntervalTime = Long.parseLong(jobIntervalTimeStr);
            } catch (NumberFormatException e) {
                this.jobIntervalTime = DEFAULT_JOB_INTERVAL_TIME;
            }
        } else {
            this.jobIntervalTime = DEFAULT_JOB_INTERVAL_TIME;
        }

        // Executoȑ풓[h̏ItOĎt@CitpXŋLqj
        String executorEndMonitoringFileStr = PropertyUtil
                .getProperty(EXECUTOR_END_MONITORING_FILE);
        if (executorEndMonitoringFileStr != null
                && executorEndMonitoringFileStr.length() != 0) {
            this.executorEndMonitoringFile = executorEndMonitoringFileStr;
        } else {
            this.executorEndMonitoringFile = DEFAULT_EXECUTOR_END_MONITORING_FILE;
        }

        // Executor̃WuI҂`FbNԊui~bj
        String executorJobTerminateWaitIntervalTimeStr = PropertyUtil
                .getProperty(EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME);
        if (executorJobTerminateWaitIntervalTimeStr != null
                && executorJobTerminateWaitIntervalTimeStr.length() != 0) {
            try {
                this.executorJobTerminateWaitIntervalTime = Long
                        .parseLong(executorJobTerminateWaitIntervalTimeStr);
            } catch (NumberFormatException e) {
                this.executorJobTerminateWaitIntervalTime = DEFAULT_EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME;
            }
        } else {
            this.executorJobTerminateWaitIntervalTime = DEFAULT_EXECUTOR_JOB_TERMINATE_WAIT_INTERVAL_TIME;
        }

    }

    /**
     * <h6>ob`s.</h6>
     * @param jobSequenceId WuV[PXR[h
     * @return rWlXWbNs
     */
    public BLogicResult executeBatch(String jobSequenceId) {
        BLogicResult result = new BLogicResult();
        boolean st = false;

        if (log.isInfoEnabled()) {
            log.info(new StringBuilder("BATCH START jobSequenceId:[").append(
                    jobSequenceId).append("]").toString());
        }

        try {
            // SqlMapClientgp\`FbN
            if (sysQueryDAO == null || sysUpdateDAO == null
                    || sysTransactionManager == null) {
                if (log.isErrorEnabled()) {
                    log.error(new StringBuilder(
                            "System sqlMapClient is null.(JOB_SEQ_CD:").append(
                            jobSequenceId).append(")").toString());
                }
                return result;
            }

            // WuR[h擾
            BatchJobData jobRecord = JobUtil.selectJob(jobSequenceId, false,
                    sysQueryDAO);
            if (jobRecord == null) {
                if (log.isErrorEnabled()) {
                    log.error(new StringBuilder(
                            "Job Record Not Found.(JOB_SEQ_CD:").append(
                            jobSequenceId).append(")").toString());
                }
                return result;
            }

            if (changeStartStatus) {
                // WuXe[^XݒiJnj
                st = startBatchStatus(jobSequenceId, sysQueryDAO, sysUpdateDAO,
                        sysTransactionManager);
                if (!st) {
                    if (log.isInfoEnabled()) {
                        log.info(new StringBuilder(
                                "Job status update error.(JOB_SEQ_CD:").append(
                                jobSequenceId).append(")").append(
                                " blogicStatus:[").append(
                                result.getBlogicStatus()).append("]")
                                .toString());
                    }
                    return result;
                }
            }

            // Ô߃g
            if (jobRecord.getJobAppCd() != null) {
                jobRecord.setJobAppCd(jobRecord.getJobAppCd().trim());
            }

            // ob`s
            result = executeBatch(jobRecord);

            if (log.isDebugEnabled()) {
                log.debug(new StringBuilder("batchStatus:").append(
                        result.getBlogicStatus()).toString());
            }
        } finally {
            // ς WuXe[^XݒiIj
            st = endBatchStatus(jobSequenceId, result, sysQueryDAO,
                    sysUpdateDAO, sysTransactionManager);

            if (!st) {
                if (log.isErrorEnabled()) {
                    log.error(new StringBuilder(
                            "Job status update error.(JOB_APP_CD:").append(
                            jobSequenceId).append(")")
                            .append(" blogicStatus:[").append(
                                    result.getBlogicStatus()).append("]")
                            .toString());
                }
                return result;
            }
        }

        if (log.isInfoEnabled()) {
            log.info(new StringBuilder("BATCH END jobSequenceId:[").append(
                    jobSequenceId).append("]").append(" blogicStatus:[")
                    .append(result.getBlogicStatus()).append("]").toString());
        }
        return result;
    }

    /**
     * <h6>WuXe[^XXViWuJnj.</h6>
     * @param jobSequenceId XVΏۂ̃WuV[PXR[h
     * @param queryDAO QueryDAO
     * @param updateDAO UpdateDAO
     * @param transactionManager TransactionManager
     * @return WuXe[^XXVtrue
     */
    protected boolean startBatchStatus(String jobSequenceId, QueryDAO queryDAO,
            UpdateDAO updateDAO, PlatformTransactionManager transactionManager) {
        return updateBatchStatus(jobSequenceId,
                EventConstants.EVENT_STATUS_START, null, queryDAO, updateDAO,
                transactionManager);
    }

    /**
     * <h6>WuXe[^XXViWuIj.</h6>
     * @param jobSequenceId XVΏۂ̃WuV[PXR[h
     * @param result Xe[^X
     * @param queryDAO QueryDAO
     * @param updateDAO UpdateDAO
     * @param transactionManager TransactionManager
     * @return Xe[^XXVtrue
     */
    protected boolean endBatchStatus(String jobSequenceId, BLogicResult result,
            QueryDAO queryDAO, UpdateDAO updateDAO,
            PlatformTransactionManager transactionManager) {
        String blogicStatus = null;
        if (result != null) {
            blogicStatus = Integer.toString(result.getBlogicStatus());
        }
        return updateBatchStatus(jobSequenceId,
                EventConstants.EVENT_STATUS_NORMAL_TERMINATION, blogicStatus,
                queryDAO, updateDAO, transactionManager);
    }

    /**
     * <h6>WuXe[^XXV.</h6>
     * <p>
     * Xe[^X}bvɂāAWũXe[^X𔽉f
     * </p>
     * @param jobSequenceId XVΏۂ̃WuV[PXR[h
     * @param eventCode CxgR[h
     * @param blogicStatus blogic̖߂l
     * @param queryDAO QueryDAO
     * @param updateDAO UpdateDAO
     * @param transactionManager TransactionManager
     * @return Xe[^XXVtrue
     */
    protected boolean updateBatchStatus(String jobSequenceId, String eventCode,
            String blogicStatus, QueryDAO queryDAO, UpdateDAO updateDAO,
            PlatformTransactionManager transactionManager) {
        TransactionStatus tranStatus = null;

        try {
            DefaultTransactionDefinition tranDef = new DefaultTransactionDefinition();

            // gUNVJn
            tranStatus = transactionManager.getTransaction(tranDef);
            if (log.isDebugEnabled()) {
                log.debug("startTransaction");
            }

            // WuR[h擾
            BatchJobData job = JobUtil.selectJob(jobSequenceId, true, queryDAO);
            if (job == null) {
                StringBuilder errStr = new StringBuilder();
                errStr.append("Job record Not Found. jobSequenceId:");
                errStr.append(jobSequenceId);
                if (log.isErrorEnabled()) {
                    log.error(errStr.toString());
                }
                return false;
            }

            // Xe[^X菈
            String changeStatus = judgmentStatus(job, jobSequenceId, eventCode,
                    blogicStatus);
            if (changeStatus == null) {
                return false;
            }

            if (log.isDebugEnabled()) {
                StringBuilder debugStr = new StringBuilder();
                debugStr.append("Xe[^XXV jobSequenceId:");
                debugStr.append(jobSequenceId);
                debugStr.append(" changeStatus:");
                debugStr.append(changeStatus);
                log.debug(debugStr.toString());
            }

            // Xe[^XXV
            JobUtil.updateJobStatus(job.getJobSequenceId(), changeStatus, null,
                    blogicStatus, queryDAO, updateDAO);

            // gUNVR~bg
            transactionManager.commit(tranStatus);
            if (log.isDebugEnabled()) {
                log.debug("commitTransaction");
            }
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("updateBatchStatus Error.", e);
            }
            return false;
        } finally {
            try {
                // gUNVIigUNV̏ꍇ̓[obNj
                if (tranStatus != null && !tranStatus.isCompleted()) {
                    transactionManager.rollback(tranStatus);
                }
                if (log.isDebugEnabled()) {
                    log.debug("endTransaction");
                }
            } catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("endTransaction error.", e);
                }
            }
        }

        return true;
    }

    /**
     * <h6>WuXe[^X̍XV胁\bh</h6> CxgR[hƃWuXe[^XmFAWuXe[^X̍XVKvsB<br>
     * XVKvȂꍇInfoOo͂AnullԋpB
     * @param job WuR[h
     * @param jobSequenceId XVΏۂ̃WuV[PXR[h
     * @param eventCode CxgR[h
     * @param blogicStatus blogic̖߂l
     * @return
     */
    protected String judgmentStatus(BatchJobData job, String jobSequenceId,
            String eventCode, String blogicStatus) {

        String judge = null;

        if (EventConstants.EVENT_STATUS_START.equals(eventCode)) {
            if (JobStatusConstants.JOB_STATUS_UNEXECUTION.equals(job
                    .getCurAppStatus())) {
                judge = JobStatusConstants.JOB_STATUS_EXECUTING;
            } else {
                judge = null;
            }
        } else {
            if (JobStatusConstants.JOB_STATUS_EXECUTING.equals(job
                    .getCurAppStatus())) {
                judge = JobStatusConstants.JOB_STATUS_PROCESSED;
            } else {
                judge = null;
            }
        }

        // Xe[^XNG`FbN
        if (judge == null) {
            if (log.isInfoEnabled()) {
                StringBuilder errStr = new StringBuilder();
                errStr.append("Xe[^XO");
                errStr.append("(WuV[PXR[h:");
                errStr.append(jobSequenceId);
                errStr.append(" blogic̖߂l:");
                errStr.append(blogicStatus);
                errStr.append(" Cxg:");
                errStr.append(eventCode);
                errStr.append(" WuR[h̃Xe[^Xl:");
                errStr.append(job.getCurAppStatus());
                errStr.append(" :");
                errStr.append(judge);
                errStr.append(")");
                log.info(errStr.toString());
            }
            return null;
        }

        return judge.toString();
    }

    /**
     * Wu̎sԊui~bj
     * @return the jobIntervalTime
     */
    public long getJobIntervalTime() {
        return jobIntervalTime;
    }

    /**
     * Executoȑ풓[h̏ItOĎt@CitpXŋLqj
     * @return the executorEndMonitoringFile
     */
    public String getExecutorEndMonitoringFile() {
        return executorEndMonitoringFile;
    }

    /**
     * Executor̃WuI҂`FbNԊui~bj
     * @return the executorJobTerminateWaitIntervalTime
     */
    public long getExecutorJobTerminateWaitIntervalTime() {
        return executorJobTerminateWaitIntervalTime;
    }

    /**
     * JñXe[^XύXsǂ
     * @param changeStartStatus the changeStartStatus to set
     */
    public void setChangeStartStatus(boolean changeStartStatus) {
        this.changeStartStatus = changeStartStatus;
    }

}
