﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Practices.Unity;
using System.ServiceModel;
using Terasoluna.ServiceModel;
using System.IO;
using Terasoluna.Threading;
using Terasoluna.Windows.Forms.Events;
using CalcBusinessApplication.Dto;
using CalcBusinessApplication.MtomServiceReference;

namespace CalcBusinessApplication.BizLogic
{
    public class FileDownloadBizLogic
    {
        /// <summary>
        /// エンドポイント名
        /// </summary>
        private const string EndpointName = "WSHttpBinding_MtomService";

        /// <summary>
        /// ストリームを読み取る際のバッファサイズ
        /// </summary>
        private const int BUFFER_SIZE = 1024;

        ///<summary>
        /// UnityContainer
        /// </summary>
        [Dependency]
        public IUnityContainer Container { get; set; }
        
        /// <summary>
        /// ファイルをダウンロードする
        /// </summary>
        /// <param name="inputDto">ダウンロードするファイル情報</param>
        /// <returns>ダウンロードファイルを保存した一時ファイルの情報</returns>
        public FileInfoDto Download(FileInfoDto inputDto)
        {
            MtomServiceClient client = null;
            string tmpFileName = "";
            Stream stream = null;
            try
            {
                ///エンドポイントアドレスの取得
                ///WCF接続先変更機能により、接続先サーバアドレスを一律置換してくれる。
                EndpointAddress address = 
                    CommunicationManager.GetReplacedEndpointAddress(Container, EndpointName);
                client = new MtomServiceClient(EndpointName, address);
                ///一時ファイルに保存してビジネスロジックを終了し、
                ///イベント完了時にSaveFileDialogを表示し指定した場所に一時ファイルの内容をコピーする
                long totalLength = 0;
                ///ファイルのダウンロード                
                TransferedFile transferedFile = 
                    client.Download(new DownloadInputDto(){
                        FileName = inputDto.FileName
                    });                
                if (transferedFile.FileData == null || transferedFile.FileData.Length == 0)
                {                    
                    return new FileInfoDto(null);
                }
                ///進捗管理クラスの取得
                IEventProcProgressManager progressManager =
                    InvocationScope.Current.GetContext<EventProcessContext>().ProgressManager;                
                ///一時ファイルへ書き込み
                stream = new MemoryStream(transferedFile.FileData);
                totalLength = transferedFile.FileData.Length;
                tmpFileName = Path.Combine(Path.GetTempPath(), transferedFile.FileName);
                using (FileStream outputFile = File.Create(tmpFileName))
                {
                    byte[] buf = new byte[BUFFER_SIZE];
                    int len = 0;
                    long readLength = 0;
                    while ((len = stream.Read(buf, 0, buf.Length)) != 0)
                    {
                        outputFile.Write(buf, 0, len);
                        readLength += len;
                        progressManager.SetProgress(new ProgressInfo((int) (readLength * 100 / totalLength)));
                    }
                }
                client.Close();
                return new FileInfoDto(tmpFileName);
            }
            catch (System.Exception)
            {
                if (client != null)
                {
                    client.Abort();
                }
                ///一時ファイルの削除
                if (!string.IsNullOrEmpty(tmpFileName) && File.Exists(tmpFileName))
                {
                    File.Delete(tmpFileName);
                }
                throw;
            }
            finally
            {
                if (stream != null)
                {
                    stream.Close();
                }
            }
        }


    }
}
