package jp.terasoluna.toursample.server.interfaces.tour.ws;

import javax.jws.WebMethod;
import javax.jws.WebService;
import javax.xml.ws.soap.MTOM;

import jp.terasoluna.toursample.server.application.tour.TourBizLogic;
import jp.terasoluna.toursample.server.interfaces.tour.assembler.TourAssembler;
import jp.terasoluna.toursample.server.interfaces.tour.dto.TourSearchKeyDto;
import jp.terasoluna.toursample.server.interfaces.validation.SampleValidator;
import jp.terasoluna.toursample.server.interfaces.ws.exception.SampleSoapFaultException;
import jp.terasoluna.toursample.server.interfaces.ws.exception.handler.SampleSoapFaultExceptionHandler;
import jp.terasoluna.toursample.server.interfaces.ws.mtom.FileObjectDto;
import jp.terasoluna.toursample.server.interfaces.ws.mtom.StreamBody;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

/**
 *
 * MTOMを使ったアップロード/ダウンロードのWebサービスサンプル
 *
 */
@MTOM
@WebService(serviceName = "TourMtomService")
public class TourMtomService extends SpringBeanAutowiringSupport {

    @Autowired
    private TourBizLogic bizLogic;
    @Autowired
    private SampleSoapFaultExceptionHandler exceptionHandler;

    @Autowired
    @Qualifier("defaultValidator")
    private SampleValidator defaultValidator;

    private TourAssembler assembler = new TourAssembler();

    /**
     * WCFクライアント側でStreamでファイルアップロードを扱うサンプル
     * メソッドの引数をStreamBodyを１つだけもつようにしないとクライアント側はbyte[]になってしまう。
     * このため、ストリームといっしょにテキスト情報を送ることができない
     */
    @WebMethod(operationName = "UploadTourListFileWithStream")
    public void uploadTourListFileWithStream(StreamBody streamBody)
            throws SampleSoapFaultException {
        try {
            bizLogic.addRegistTourJob(streamBody.getStream());
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
    }

    /**
     * WCFクライアント側でバイト配列でファイルアップロードを扱うサンプル
     * バイトデータ以外にも、テキスト情報も一緒に送れる
     */
    @WebMethod(operationName = "UploadTourListFileWithByteArray")
    public void uploadTourListFileWithByteArray(FileObjectDto inputDto)
            throws SampleSoapFaultException {
        try {
            bizLogic.addRegistTourJob(inputDto.getFileName(), inputDto.getFileData());
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
    }

    /**
     * WCFクライアント側でStreamでファイルダウンロードを扱うサンプル
     * メソッドの引数をStreamBodyを１つだけもつようにしないとクライアント側はbyte[]になってしまう。
     * このため、ストリームといっしょにテキスト情報を送ることができない
     */
    @WebMethod(operationName = "DownloadTourListFileWithStream")
    public StreamBody downloadTourListFileWithStream(TourSearchKeyDto inputDto)
            throws SampleSoapFaultException {
        try {
            // 入力チェック処理
            defaultValidator.validate(inputDto);
            return assembler.toStreamBody(bizLogic.findTourByName(inputDto
                    .getTourName()));
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
        return null;
    }

    /**
     * WCFクライアント側でByte配列でファイルダウンロードを扱うサンプル
     * バイトデータ以外にも、テキスト情報も一緒に送れる
     */
    @WebMethod(operationName = "DownloadTourListFileWithByteArray")
    public FileObjectDto downloadTourListFileWithByteArray(
            TourSearchKeyDto inputDto) throws SampleSoapFaultException {
        try {
            // 入力チェック処理
            defaultValidator.validate(inputDto);
            return assembler.toTourListFileDto(bizLogic.findTourByName(inputDto
                    .getTourName()));
        } catch (Exception e) {
            // 例外が発生した場合に、必要に応じてSampleSoapFaultExceptionをスロー
            exceptionHandler.handleException(e);
        }
        return null;
    }
}
