﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel;
using Terasoluna.SelectableValues;
using System.Reflection;
using System.Drawing.Design;
using System.Collections;

namespace Terasoluna.TourSample.Client.Common.View
{
    /// <summary>
    /// 
    /// SelectedItemsに対して画面データと
    /// 複合データバインディングできるようにした拡張ListBoxの簡易実装
    /// </summary>
    /// ComplexBindingProperties属性の付与により、データソース画面からの貼り付けが可能になる
    [System.ComponentModel.ComplexBindingProperties("SelectedDataSource", "SelectedDataMember")]
    public class SelectedItemsBindableListBox : ListBox
    {
        private bool selecting = false;
        private bool listChanging = false;
        private BindingSource bindingSource = null;

        public SelectedItemsBindableListBox()
        {
            SelectionMode = SelectionMode.MultiExtended;
            SelectedIndexChanged += new EventHandler(SelectedItemsBindableListBox_SelectedIndexChanged);
        }

        /// <summary>
        /// SelectedItemsとバインドされる画面データのプロパティ名
        /// </summary>
        public string StringValuePropertyName { get; set; }

        /// <summary>
        /// 複合バインディングの対象となる画面データのリスト（データソース）
        /// </summary>
        /// AttributeProvider属性の付与により、BindingSourceを選択するエディタになる
        [AttributeProvider(typeof(IListSource))]
        public object SelectedDataSource
        {
            get
            {
                return bindingSource;
            }
            set
            {
                //BindingSourceのみ対応
                BindingSource newDataSource = value as BindingSource;
                if (newDataSource != null)
                {
                    bindingSource = newDataSource;
                    bindingSource.ListChanged += new ListChangedEventHandler(bindingSource_ListChanged);
                    OnSelectedDataSourceChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// ListBoxのSelectedIndexChangedイベントに登録するデリゲート
        /// </summary>
        void SelectedItemsBindableListBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (listChanging)
            {
                //bindingSource_ListChanged実行中は動作させない
                return;
            }
            if (bindingSource == null)
            {
                return;
            }
            selecting = true;
            ///BindingSourceをクリア
            bindingSource.Clear();
            ///BindingSourceをセットし直す
            foreach (object obj in SelectedItems)
            {
                SelectableValue selectableValue = obj as SelectableValue;
                ///BindingSourceに要素を追加
                object newItem = bindingSource.AddNew();
                ///SelectedDataMemberプロパティで指定した値を持つ
                ///画面データプロパティに、SelectableValue.StringValueの値をセット
                PropertyInfo property = newItem.GetType().GetProperty(StringValuePropertyName);
                if (property != null && property.PropertyType.Equals(typeof(string)))
                {
                    property.SetValue(newItem, selectableValue.StringValue, null);
                }
            }
            selecting = false;
            /// SelectedDataSourceChangedイベントを通知
            OnSelectedDataSourceChanged(EventArgs.Empty);
        }

        /// <summary>
        /// BindingSourceのListChangedイベントに登録するデリゲート
        /// </summary>
        void bindingSource_ListChanged(object sender, ListChangedEventArgs e)
        {
            if (selecting)
            {
                //SelectedItemsBindableListBox_SelectedIndexChanged実行中は動作させない
                return;
            }
            listChanging = true;
            ClearSelected();
            //データソース内の画面データのリストを取得
            IList viewDataList = bindingSource.List;
            if (viewDataList != null)
            {
                ///画面データのリストに含まれるものを選択状態にする
                foreach (object viewData in viewDataList)
                {
                    ///SelectedDataMemberプロパティで指定した値を持つ
                    ///画面データプロパティの値と一致するものを選択状態にする
                    PropertyInfo property = viewData.GetType().GetProperty(StringValuePropertyName);
                    if (property != null && property.PropertyType.Equals(typeof(string)))
                    {
                        for (int i = 0; i < Items.Count; i++)
                        {
                            SelectableValue selectableValue = Items[i] as SelectableValue;
                            object value = property.GetValue(viewData, new object[0]);                               
                            if (value != null && string.Equals(selectableValue.StringValue, value.ToString(), StringComparison.Ordinal))
                            {
                                SelectedIndex = i;
                            }
                        }
                    }
                }
            }
            listChanging = false;
        }

        /// <summary>
        /// PropertyNameChangedパターン
        /// </summary>
        public event EventHandler SelectedDataSourceChanged;

        protected virtual void OnSelectedDataSourceChanged(EventArgs e)
        {
            EventHandler handler = SelectedDataSourceChanged;
            if (handler != null)
            {
                handler(this, e);
            }
        }


        /// <summary>
        /// 現状、使用していない
        /// </summary>
        [Editor("System.Windows.Forms.Design.DataMemberListEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(UITypeEditor))]
        public string SelectedDataMember { get; set; }
    }
}
